package org.lsst.ccs.drivers.ftdi;

/**
 ***************************************************************************
 **
 **  Accesses a locally-connected device which uses the FTDI chip.
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class FtdiLocal implements FtdiInterface {

   /**
    ***************************************************************************
    **
    **  Private fields.
    **
    ***************************************************************************
    */
    private long handle;

   /**
    ***************************************************************************
    **
    **  Static Initializer.
    **
    ***************************************************************************
    */
    static {
        System.loadLibrary("Ftdi");
    }


   /**
    ***************************************************************************
    **
    **  Opens a local device.
    **
    **  @param  index   The zero-based index of the FTDI device within the
    **                  list selected by the serial argument.
    **
    **  @param  serial  A string which, if non-null, restricts the list of
    **                  available devices to those with a serial number
    **                  containing this string.
    **
    **  @throws FtdiException
    **
    ***************************************************************************
    */
    @Override
    public void open(int index, String serial)
        throws FtdiException
    {
        handle = openJNI(index, serial);
    }


   /**
    ***************************************************************************
    **
    **  Opens a remote device.
    **
    **  This is an invalid operation.
    **
    **  @param  node    The name of the node where the device is located, or
    **                  null to specify a local device.
    **
    **  @param  index   The zero-based index of the FTDI device within the
    **                  list selected by the serial argument.
    **
    **  @param  serial  A string which, if non-null and non-empty, restricts
    **                  the list of available devices to those with a serial
    **                  number containing this string.
    **
    **  @throws FtdiException
    **
    ***************************************************************************
    */
    @Override
    public void open(String node, int index, String serial)
        throws FtdiException
    {
        throw new FtdiException("Invalid remote open call");
    }


   /**
    ***************************************************************************
    **
    **  Closes the device.
    **
    **  @throws FtdiException
    **
    ***************************************************************************
    */
    @Override
    public void close() throws FtdiException
    {
        closeJNI(handle);
        handle = 0;
    }


   /**
    ***************************************************************************
    **
    **  Sets the baud rate.
    **
    **  @param  baudrate  The baud rate to set.
    **
    **  @throws FtdiException
    **
    ***************************************************************************
    */
    @Override
    public void setBaudrate(int baudrate) throws FtdiException
    {
        setBaudrateJNI(handle, baudrate);
    }


   /**
    ***************************************************************************
    **
    **  Sets the data characteristics.
    **
    **  @param  wordLength  The encoded word length to set..
    **
    **  @param  stopBits    The encoded number of stop bits to set..
    **
    **  @param  parity      The encoded parity value to set.
    **
    **  @throws FtdiException
    **
    ***************************************************************************
    */
    @Override
    public void setDataCharacteristics(int wordLength, int stopBits, int parity)
        throws FtdiException
    {
        setDataCharacteristicsJNI(handle, wordLength, stopBits, parity);
    }


   /**
    ***************************************************************************
    **
    **  Sets the timeouts.
    **
    **  @param  rcveTimeout  The receive timeout to set (ms).  A value of 0
    **                       means no timeout.
    **
    **  @param  xmitTimeout  The transmit timeout to set (ms).  A value of 0
    **                       means no timeout.
    **
    **  @throws FtdiException
    **
    ***************************************************************************
    */
    @Override
    public void setTimeouts(int rcveTimeout, int xmitTimeout)
        throws FtdiException
    {
        setTimeoutsJNI(handle, rcveTimeout, xmitTimeout);
    }


   /**
    ***************************************************************************
    **
    **  Reads data.
    **
    **  Execution is blocked until either the byte array is filled, or a
    **  timeout occurs.  In the latter case the number of bytes read will be
    **  less than the array size.
    **
    **  @param  data  A byte array to receive the read data.
    **
    **  @return  The number of bytes read.
    **
    **  @throws FtdiException
    **
    ***************************************************************************
    */
    @Override
    public int read(byte[] data) throws FtdiException
    {
        return readJNI(handle, data, 0, data.length);
    }


   /**
    ***************************************************************************
    **
    **  Reads data.
    **
    **  Execution is blocked until either the requested number of bytes has
    **  been read, or a timeout occurs.  In the latter case the number of
    **  bytes read will be less than the requested number.
    **
    **  @param  data    A byte array to receive the read data.
    **
    **  @param  offset  The offset in the array to the start of the data.
    **
    **  @param  count   The maximum number of bytes to read.
    **
    **  @return  The number of bytes read.
    **
    **  @throws FtdiException
    **
    ***************************************************************************
    */
    @Override
    public int read(byte[] data, int offset, int count) throws FtdiException
    {
        return readJNI(handle, data, offset, count);
    }


   /**
    ***************************************************************************
    **
    **  Writes data.
    **
    **  @param  data  A byte array containing the data to write.
    **
    **  @return  The number of bytes written.
    **
    **  @throws FtdiException
    **
    ***************************************************************************
    */
    @Override
    public int write(byte[] data) throws FtdiException
    {
        return writeJNI(handle, data, 0, data.length);
    }


   /**
    ***************************************************************************
    **
    **  Writes data.
    **
    **  @param  data    A byte array containing the data to write.
    **
    **  @param  offset  The offset in the array to the start of the data.
    **
    **  @param  count   The number of bytes to write.
    **
    **  @return  The number of bytes written.
    **
    **  @throws FtdiException
    **
    ***************************************************************************
    */
    @Override
    public int write(byte[] data, int offset, int count) throws FtdiException
    {
        return writeJNI(handle, data, offset, count);
    }


   /**
    ***************************************************************************
    **
    **  Gets the read queue status.
    **
    **  This is the number of bytes available for immediate read.
    **
    **  @return  The number of bytes in the read queue.
    **
    **  @throws FtdiException
    **
    ***************************************************************************
    */
    @Override
    public int getQueueStatus() throws FtdiException
    {
        return getQueueStatusJNI(handle);
    }


   /**
    ***************************************************************************
    **
    **  Gets the modem status.
    **
    **  @return  The modem status as a set of bits.
    **
    **  @throws FtdiException
    **
    ***************************************************************************
    */
    @Override
    public int getModemStatus() throws FtdiException
    {
        return getModemStatusJNI(handle);
    }


   /**
    ***************************************************************************
    **
    **  JNI method declarations
    **
    ***************************************************************************
    */
    private native long openJNI(int index, String serial)
        throws FtdiException;

    private native void closeJNI(long handle) throws FtdiException;

    private native void setBaudrateJNI(long handle, int baudrate)
        throws FtdiException;

    private native void setDataCharacteristicsJNI(long handle, int wordLength,
                                                  int stopBits, int parity)
        throws FtdiException;

    private native void setTimeoutsJNI(long handle, int rcveTimeout,
                                       int xmitTimeout)
        throws FtdiException;

    private native int readJNI(long handle, byte[] data, int offset, int count)
        throws FtdiException;

    private native int writeJNI(long handle, byte[] data, int offset, int count)
        throws FtdiException;

    private native int getQueueStatusJNI(long handle) throws FtdiException;

    private native int getModemStatusJNI(long handle) throws FtdiException;

}
