package org.lsst.ccs.drivers.ftdi;

/**
 ***************************************************************************
 **
 **  Accesses a locally-connected device which uses the FTDI chip.
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class FtdiLocal implements FtdiInterface {

   /**
    ***************************************************************************
    **
    **  Private fields.
    **
    ***************************************************************************
    */
    private long handle;

   /**
    ***************************************************************************
    **
    **  Static Initializer.
    **
    ***************************************************************************
    */
    static {
        System.loadLibrary("Ftdi");
        initSys();
    }


   /**
    ***************************************************************************
    **
    **  Opens a local device.
    **
    **  @param  index   The zero-based index of the FTDI device within the
    **                  list selected by the serial argument.
    **
    **  @param  serial  A string which, if non-null, restricts the list of
    **                  available devices to those with a serial number
    **                  containing this string.
    **
    **  @throws FtdiException
    **
    ***************************************************************************
    */
    @Override
    public native void open(int index, String serial) throws FtdiException;


   /**
    ***************************************************************************
    **
    **  Opens a remote device.
    **
    **  This is an invalid operation.
    **
    **  @param  node    The name of the node where the device is located, or
    **                  null to specify a local device.
    **
    **  @param  index   The zero-based index of the FTDI device within the
    **                  list selected by the serial argument.
    **
    **  @param  serial  A string which, if non-null and non-empty, restricts
    **                  the list of available devices to those with a serial
    **                  number containing this string.
    **
    **  @throws FtdiException
    **
    ***************************************************************************
    */
    @Override
    public void open(String node, int index, String serial) throws FtdiException
    {
        throw new FtdiException("Invalid remote open call");
    }


   /**
    ***************************************************************************
    **
    **  Closes the device.
    **
    **  @throws FtdiException
    **
    ***************************************************************************
    */
    @Override
    public native void close() throws FtdiException;


   /**
    ***************************************************************************
    **
    **  Sets the baud rate.
    **
    **  @param  baudrate  The baud rate to set.
    **
    **  @throws FtdiException
    **
    ***************************************************************************
    */
    @Override
    public native void setBaudrate(int baudrate) throws FtdiException;


   /**
    ***************************************************************************
    **
    **  Sets the data characteristics.
    **
    **  @param  wordLength  The encoded word length to set..
    **
    **  @param  stopBits    The encoded number of stop bits to set..
    **
    **  @param  parity      The encoded parity value to set.
    **
    **  @throws FtdiException
    **
    ***************************************************************************
    */
    @Override
    public native void setDataCharacteristics(int wordLength, int stopBits,
                                              int parity)
        throws FtdiException;


   /**
    ***************************************************************************
    **
    **  Sets the timeouts.
    **
    **  @param  rcveTimeout  The receive timeout to set (ms).  A value of 0
    **                       means no timeout.
    **
    **  @param  xmitTimeout  The transmit timeout to set (ms).  A value of 0
    **                       means no timeout.
    **
    **  @throws FtdiException
    **
    ***************************************************************************
    */
    @Override
    public native void setTimeouts(int rcveTimeout, int xmitTimeout)
        throws FtdiException;


   /**
    ***************************************************************************
    **
    **  Reads data.
    **
    **  Execution is blocked until either the byte array is filled, or a
    **  timeout occurs.  In the latter case the number of bytes read will be
    **  less than the array size.
    **
    **  @param  data  A byte array to receive the read data.
    **
    **  @return  The number of bytes read.
    **
    **  @throws FtdiException
    **
    ***************************************************************************
    */
    @Override
    public int read(byte[] data) throws FtdiException
    {
        return read(data, 0, data.length);
    }


   /**
    ***************************************************************************
    **
    **  Reads data.
    **
    **  Execution is blocked until either the requested number of bytes has
    **  been read, or a timeout occurs.  In the latter case the number of
    **  bytes read will be less than the requested number.
    **
    **  @param  data    A byte array to receive the read data.
    **
    **  @param  offset  The offset in the array to the start of the data.
    **
    **  @param  count   The maximum number of bytes to read.
    **
    **  @return  The number of bytes read.
    **
    **  @throws FtdiException
    **
    ***************************************************************************
    */
    @Override
    public native int read(byte[] data, int offset, int count)
        throws FtdiException;


   /**
    ***************************************************************************
    **
    **  Writes data.
    **
    **  @param  data  A byte array containing the data to write.
    **
    **  @return  The number of bytes written.
    **
    **  @throws FtdiException
    **
    ***************************************************************************
    */
    @Override
    public int write(byte[] data) throws FtdiException
    {
        return write(data, 0, data.length);
    }


   /**
    ***************************************************************************
    **
    **  Writes data.
    **
    **  @param  data    A byte array containing the data to write.
    **
    **  @param  offset  The offset in the array to the start of the data.
    **
    **  @param  count   The number of bytes to write.
    **
    **  @return  The number of bytes written.
    **
    **  @throws FtdiException
    **
    ***************************************************************************
    */
    @Override
    public native int write(byte[] data, int offset, int count)
        throws FtdiException;


   /**
    ***************************************************************************
    **
    **  Gets the read queue status.
    **
    **  This is the number of bytes available for immediate read.
    **
    **  @return  The number of bytes in the read queue.
    **
    **  @throws FtdiException
    **
    ***************************************************************************
    */
    @Override
    public native int getQueueStatus() throws FtdiException;


   /**
    ***************************************************************************
    **
    **  Gets the modem status.
    **
    **  @return  The modem status as a set of bits.
    **
    **  @throws FtdiException
    **
    ***************************************************************************
    */
    @Override
    public native int getModemStatus() throws FtdiException;


   /**
    ***************************************************************************
    **
    **  Initializes the system.
    **
    ***************************************************************************
    */
    private native static void initSys();

}
