package org.lsst.ccs.drivers.eaton;

import org.lsst.ccs.drivers.ascii.Ascii;
import org.lsst.ccs.drivers.ascii.Session;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Routines for controlling an Eaton ePDU G3 power strip
 *
 *  @author Owen Saxton
 */
public class EpduG3 extends Session {

    /**
     *  Private data.
     */
    private static final int
        DEFAULT_BAUDRATE = 9600,
        LOG_TIMEOUT = 50,
        RUN_TIMEOUT = 50;
    private Integer numPhases, numGangs, numOutlets;


    /**
     *  Constructor
     */
    public EpduG3()
    {
        super(OPTN_KEEP_ALIVE, "pdu#", "Login:", "Password:", null, Ascii.Terminator.CR);
    }


    /**
     *  Opens a connection and logs in.
     *
     *  @param  connType  The enumerated connection type: TELNET, FTDI or SERIAL
     *  @param  ident     The host name (TELNET), USB ID (FTDI) or port name (SERIAL)
     *  @param  username  The user name
     *  @param  password  The password
     *  @throws  DriverException
     */
    public void open(ConnType connType, String ident, String username, String password) throws DriverException
    {
        open(connType, ident, DEFAULT_BAUDRATE, username, password, LOG_TIMEOUT, RUN_TIMEOUT);
    }


    /**
     *  Opens a connection.
     *
     *  @param  connType  The enumerated connection type: FTDI or SERIAL
     *  @param  ident     The USB ID (FTDI) or port name (SERIAL)
     *  @throws  DriverException
     */
    public void open(ConnType connType, String ident) throws DriverException
    {
        open(connType, ident, DEFAULT_BAUDRATE, RUN_TIMEOUT);
    }


    /**
     *  Closes the connection
     *
     *  @throws  DriverException
     */
    @Override
    public void close() throws DriverException
    {
        try {
            super.close();
        }
        finally {
            numPhases = null;
            numGangs = null;
            numOutlets = null;
        }
    }


    /**
     *  Gets the product name
     *
     *  @return  The product name
     *  @throws  DriverException
     */
    public String getProductName() throws DriverException
    {
        return receiveString("get PDU.PowerSummary.iProduct");
    }


    /**
     *  Gets the serial number
     *
     *  @return  The serial number
     *  @throws  DriverException
     */
    public String getSerialNumber() throws DriverException
    {
        return receiveString("get PDU.PowerSummary.iSerialNumber");
    }


    /**
     *  Gets the firmware version
     *
     *  @return  The firmware version
     *  @throws  DriverException
     */
    public String getFWVersion() throws DriverException
    {
        return receiveString("get PDU.PowerSummary.iVersion");
    }


    /**
     *  Reads the temperature
     *
     *  @return  The temperature (C)
     *  @throws  DriverException
     */
    public double readTemperature() throws DriverException
    {
        return receiveDouble("get PDU.PowerSummary.Temperature") - 273.15;
    }


    /**
     *  Gets the number of phases
     *
     *  @return  The number of phases
     *  @throws  DriverException
     */
    public int getPhaseCount() throws DriverException
    {
        if (numPhases == null) {
            numPhases = receiveInt("get PDU.Input[1].Phase.Count");
        }
        return numPhases;
    }


    /**
     *  Gets the number of gangs
     *
     *  @return  The number of gangs
     *  @throws  DriverException
     */
    public int getGangCount() throws DriverException
    {
        if (numGangs == null) {
            numGangs = receiveInt("get PDU.Gang.Count");
        }
        return numGangs;
    }


    /**
     *  Gets the phase number of a gang
     *
     *  @param  gang  The gang number, starting from 1
     *  @return  The phase number of the gang
     *  @throws  DriverException
     */
    public int getGangPhase(int gang) throws DriverException
    {
        checkGangNumber(gang);
        return receiveInt("get PDU.Gang[" + gang + "].PhaseID");
    }


    /**
     *  Gets the phase numbers of all gangs
     *
     *  @return  The array of gang phase numbers
     *  @throws  DriverException
     */
    public int[] getGangPhase() throws DriverException
    {
        return receiveInts("get PDU.Gang[x].PhaseID");
    }


    /**
     *  Reads gang voltage
     *
     *  @param  gang  The gang number, starting from 1
     *  @return  The gang voltage (volts)
     *  @throws  DriverException
     */
    public double readGangVoltage(int gang) throws DriverException
    {
        checkGangNumber(gang);
        return receiveDouble("get PDU.Gang[" + gang + "].Voltage");
    }


    /**
     *  Reads all gang voltages
     * 
     *  @return  The array of gang voltages (volts)
     *  @throws  DriverException
     */
    public double[] readGangVoltage() throws DriverException
    {
        double[] values = receiveDoubles("get PDU.Gang[x].Voltage");
        equalsGangCount(values.length);
        return values;
    }


    /**
     *  Reads gang current
     *
     *  @param  gang  The gang number, starting from 1
     *  @return  The gang current (amps)
     *  @throws  DriverException
     */
    public double readGangCurrent(int gang) throws DriverException
    {
        checkGangNumber(gang);
        return receiveDouble("get PDU.Gang[" + gang + "].Current");
    }


    /**
     *  Reads all gang currents
     * 
     *  @return  The array of gang currents (amps)
     *  @throws  DriverException
     */
    public double[] readGangCurrent() throws DriverException
    {
        double[] values = receiveDoubles("get PDU.Gang[x].Current");
        equalsGangCount(values.length);
        return values;
    }


    /**
     *  Reads gang power
     *
     *  @param  gang  The gang number, starting from 1
     *  @return  The gang power (watts)
     *  @throws  DriverException
     */
    public double readGangPower(int gang) throws DriverException
    {
        checkGangNumber(gang);
        return receiveDouble("get PDU.Gang[" + gang + "].ActivePower");
    }


    /**
     *  Reads all gang power values
     * 
     *  @return  The array of gang power values (watts)
     *  @throws  DriverException
     */
    public double[] readGangPower() throws DriverException
    {
        double[] values = receiveDoubles("get PDU.Gang[x].ActivePower");
        equalsGangCount(values.length);
        return values;
    }


    /**
     *  Gets the number of outlets
     *
     *  @return  The number of outlets
     *  @throws  DriverException
     */
    public int getOutletCount() throws DriverException
    {
        if (numOutlets == null) {
            numOutlets = receiveInt("get PDU.OutletSystem.Outlet.Count");
        }
        return numOutlets;
    }


    /**
     *  Gets the gang number of an outlet
     *
     *  @param  outlet  The outlet number, starting from 1
     *  @return  The gang number of the outlet
     *  @throws  DriverException
     */
    public int getOutletGang(int outlet) throws DriverException
    {
        checkOutletNumber(outlet);
        return receiveInt("get PDU.OutletSystem.Outlet[" + outlet + "].GangID");
    }


    /**
     *  Gets the gang numbers of all outlets
     *
     *  @return  The array of outlet gang numbers
     *  @throws  DriverException
     */
    public int[] getOutletGang() throws DriverException
    {
        return receiveInts("get PDU.OutletSystem.Outlet[x].GangID");
    }


    /**
     *  Gets the on (powered) state of an outlet
     *
     *  @param  outlet  The outlet number, starting from 1
     *  @return  Whether the outlet is powered
     *  @throws  DriverException
     */
    public boolean isOutletOn(int outlet) throws DriverException
    {
        checkOutletNumber(outlet);
        return receiveInt("get PDU.OutletSystem.Outlet[" + outlet + "].PresentStatus.SwitchOnOff") == 1;
    }


    /**
     *  Gets the on (powered) state of all outlets
     *
     *  @return  A boolean array containing the on state of all outlets
     *  @throws  DriverException
     */
    public boolean[] getOutletOnStates() throws DriverException
    {
        int[] states = receiveInts("get PDU.OutletSystem.Outlet[x].PresentStatus.SwitchOnOff");
        equalsOutletCount(states.length);
        boolean[] onStates = new boolean[states.length];
        for (int j = 0; j < states.length; j++) {
            onStates[j] = states[j] == 1;
        }
        return onStates;
    }


    /**
     *  Turns outlet power on.
     *
     *  The command always returns immediately, even if a non-zero delay is
     *  specified.  A delay of -1 cancels any delayed power-on in progress.
     *
     *  @param  outlet  The outlet number, starting from 1
     *  @param  delay   The delay (sec) before the outlet is powered on
     *  @throws  DriverException
     */
    public void powerOutletOn(int outlet, int delay) throws DriverException
    {
        checkOutletNumber(outlet);
        receiveInt("set PDU.OutletSystem.Outlet[" + outlet + "].DelayBeforeStartup " + delay);
    }


    /**
     *  Turns outlet power off.
     *
     *  The command always returns immediately, even if a non-zero delay is
     *  specified.  A delay of -1 cancels any delayed power-off in progress.
     *
     *  @param  outlet  The outlet number, starting from 1
     *  @param  delay   The delay (sec) before the outlet is powered off
     *  @throws  DriverException
     */
    public void powerOutletOff(int outlet, int delay) throws DriverException
    {
        checkOutletNumber(outlet);
        receiveInt("set PDU.OutletSystem.Outlet[" + outlet + "].DelayBeforeShutdown " + delay);
    }


    /**
     *  Gets configured outlet current
     *
     *  @param  outlet  The outlet number, starting from 1
     *  @return  The configured outlet current (amps)
     *  @throws  DriverException
     */
    public double getOutletConfigCurrent(int outlet) throws DriverException
    {
        checkOutletNumber(outlet);
        return receiveDouble("get PDU.OutletSystem.Outlet[" + outlet + "].ConfigCurrent");
    }


    /**
     *  Gets all configured outlet currents
     * 
     *  @return  The array of outlet currents (amps)
     *  @throws  DriverException
     */
    public double[] getOutletConfigCurrent() throws DriverException
    {
        double[] values = receiveDoubles("get PDU.OutletSystem.Outlet[x].ConfigCurrent");
        equalsOutletCount(values.length);
        return values;
    }


    /**
     *  Reads outlet current
     *
     *  @param  outlet  The outlet number, starting from 1
     *  @return  The outlet current (amps)
     *  @throws  DriverException
     */
    public double readOutletCurrent(int outlet) throws DriverException
    {
        checkOutletNumber(outlet);
        return receiveDouble("get PDU.OutletSystem.Outlet[" + outlet + "].Current");
    }


    /**
     *  Reads all outlet currents
     * 
     *  @return  The array of outlet currents (amps)
     *  @throws  DriverException
     */
    public double[] readOutletCurrent() throws DriverException
    {
        double[] values = receiveDoubles("get PDU.OutletSystem.Outlet[x].Current");
        equalsOutletCount(values.length);
        return values;
    }


    /**
     *  Reads outlet power
     *
     *  @param  outlet  The outlet number, starting from 1
     *  @return  The outlet power (watts)
     *  @throws  DriverException
     */
    public double readOutletPower(int outlet) throws DriverException
    {
        checkOutletNumber(outlet);
        return receiveDouble("get PDU.OutletSystem.Outlet[" + outlet + "].ActivePower");
    }


    /**
     *  Reads all outlet power values
     * 
     *  @return  The array of outlet power values (watts)
     *  @throws  DriverException
     */
    public double[] readOutletPower() throws DriverException
    {
        double[] values = receiveDoubles("get PDU.OutletSystem.Outlet[x].ActivePower");
        equalsOutletCount(values.length);
        return values;
    }


    /**
     *  Sends a command and receives a string
     *
     *  @param  command  The command to send
     *  @return  The string value
     *  @throws  DriverException
     */
    public synchronized String receiveString(String command) throws DriverException
    {
        String[] resp = receive(command);
        if (resp.length < 2) {
            throw new DriverException("No response received");
        }
        if (resp.length > 2) {
            StringBuilder text = new StringBuilder();
            for (int j = 1; j < resp.length; j++) {
                text.append(resp[j]);
                text.append("/");
            }
            throw new DriverException("Excess response received: " + text);
        }
        return resp[1];
    }


    /**
     *  Sends a command and receives an integer
     *
     *  @param  command  The command to send
     *  @return  The integer value
     *  @throws  DriverException
     */
    public int receiveInt(String command) throws DriverException
    {
        String resp = receiveString(command);
        try {
            return Integer.valueOf(resp);
        }
        catch (NumberFormatException e) {
            throw new DriverException("Invalid integer: " + resp);
        }
    }


    /**
     *  Sends a command and receives a double
     *
     *  @param  command  The command to send
     *  @return  The double value
     *  @throws  DriverException
     */
    public double receiveDouble(String command) throws DriverException
    {
        String resp = receiveString(command);
        try {
            return Double.valueOf(resp);
        }
        catch (NumberFormatException e) {
            throw new DriverException("Invalid double: " + resp);
        }
    }


    /**
     *  Sends a command and receives multiple strings
     *
     *  @param  command  The command to send
     *  @return  The array of string values
     *  @throws  DriverException
     */
    public synchronized String[] receiveStrings(String command) throws DriverException
    {
        return receiveString(command).split("\\|");
    }


    /**
     *  Sends a command and receives multiple integers
     *
     *  @param  command  The command to send
     *  @return  The array of integer values
     *  @throws  DriverException
     */
    public synchronized int[] receiveInts(String command) throws DriverException
    {
        String[] resp = receiveStrings(command);
        int[] values = new int[resp.length];
        for (int j = 0; j < values.length; j++) {
            try {
                values[j] = Integer.valueOf(resp[j]);
            }
            catch (NumberFormatException e) {
                throw new DriverException("Invalid integer: " + resp[j]);
            }
        }
        return values;
    }


    /**
     *  Sends a command and receives multiple doubles
     *
     *  @param  command  The command to send
     *  @return  The array of double values
     *  @throws  DriverException
     */
    public synchronized double[] receiveDoubles(String command) throws DriverException
    {
        String[] resp = receiveStrings(command);
        double[] values = new double[resp.length];
        for (int j = 0; j < values.length; j++) {
            try {
                values[j] = Double.valueOf(resp[j]);
            }
            catch (NumberFormatException e) {
                throw new DriverException("Invalid double: " + resp[j]);
            }
        }
        return values;
    }


    /**
     *  Checks whether a gang number is valid
     *
     *  @param  gang  The gang number
     *  @throws  DriverException if in valid
     */
    private void checkGangNumber(int gang) throws DriverException
    {
        if (gang <= 0 || gang > getGangCount()) {
            throw new DriverException("Invalid gang number: " + gang);
        }
    }


    /**
     *  Checks whether an outlet number is valid
     *
     *  @param  outlet  The outlet number
     *  @throws  DriverException if invalid
     */
    private void checkOutletNumber(int outlet) throws DriverException
    {
        if (outlet <= 0 || outlet > getOutletCount()) {
            throw new DriverException("Invalid outlet number: " + outlet);
        }
    }


    /**
     *  Checks whether a value equals the gang count
     *
     *  @param  count  The value to check
     *  @throws  DriverException if unequal
     */
    private void equalsGangCount(int count) throws DriverException
    {
        if (count != getGangCount()) {
            throw new DriverException("Incorrect number of values returned: " + count);
        }
    }


    /**
     *  Checks whether a value equals the outlet count
     *
     *  @param  count  The value to check
     *  @throws  DriverException if unequal
     */
    private void equalsOutletCount(int count) throws DriverException
    {
        if (count != getOutletCount()) {
            throw new DriverException("Incorrect number of values returned: " + count);
        }
    }

}
