package org.lsst.ccs.drivers.dscud;

/**
 ***************************************************************************
 **
 **  \file  TestDscud.java
 **
 **  Program to test the Java DSC universal driver routines
 **
 **  \author Owen Saxton
 **
 ***************************************************************************
 */
import java.io.IOException;
import java.util.Scanner;
import jline.ConsoleReader;
import org.lsst.ccs.utilities.sa.CmndProc;
import org.lsst.ccs.utilities.sa.Output;
import org.lsst.ccs.utilities.sa.ConsOut;

public class TestDscud implements CmndProc.Dispatch {

    private ConsoleReader reader;

    private final static int
        INT_TYPE_ALL = 255;

    private final static int
        CMD_INIT        = 0,
        CMD_FREE        = 1,
        CMD_SHOW        = 2,
        CMD_DIOSET      = 3,
        CMD_DIOINBYTE   = 4,
        CMD_DIOINWORD   = 5,
        CMD_DIOOUTBYTE  = 6,
        CMD_DIOOUTWORD  = 7,
        CMD_DIOINBIT    = 8,
        CMD_DIOOUTBIT   = 9,
        CMD_DIOSETBIT   = 10,
        CMD_DIOCLRBIT   = 11,
        CMD_DIOSHOW     = 12,
        CMD_ADSET       = 13,
        CMD_ADSETCHAN   = 14,
        CMD_ADSAMPLE    = 15,
        CMD_ADSAMPAVG   = 16,
        CMD_ADSAMPINT   = 17,
        CMD_ADSCAN      = 18,
        CMD_ADSCANAVG   = 19,
        CMD_ADSCANINT   = 20,
        CMD_ADAUTOCAL   = 21,
        CMD_ADCALVERIFY = 22,
        CMD_ADCTOV      = 23,
        CMD_ADVTOC      = 24,
        CMD_ADSHOW      = 25,
        CMD_ADSHOWFIFO  = 26,
        CMD_DASET       = 27,
        CMD_DASETPOL    = 28,
        CMD_DACONV      = 29,
        CMD_DASCAN      = 30,
        CMD_DAAUTOCAL   = 31,
        CMD_DACALVERIFY = 32,
        CMD_DACTOV      = 33,
        CMD_DAVTOC      = 34,
        CMD_DASHOW      = 35,
        CMD_INP         = 36,
        CMD_INPW        = 37,
        CMD_INPL        = 38,
        CMD_INPWS       = 39,
        CMD_OUT         = 40,
        CMD_OUTW        = 41,
        CMD_OUTL        = 42,
        CMD_OUTWS       = 43,
        CMD_DSHOW       = 44,
        CMD_SHOWREGS    = 45,
        CMD_SHOWERROR   = 46,
        CMD_CTRSETVALUE = 47,
        CMD_CTRSETRATE  = 48,
        CMD_CTRREAD     = 49,
        CMD_CTRSHOW     = 50,
        CMD_TIMEREAD    = 51,
        CMD_INTSTART    = 52,
        CMD_INTSTOP     = 53,
        CMD_INTCANCEL   = 54,
        CMD_INTPAUSE    = 55,
        CMD_INTRESUME   = 56,
        CMD_INTSHOW     = 57,
        CMD_DUMP        = 58,
        CMD_THRSHOW     = 59,
        CMD_THRSTOP     = 60,
        CMD_THRDAEMON   = 61,
        CMD_REGREAD     = 62,
        CMD_REGWRITE    = 63,
        CMD_SHOWBOARD   = 64,
        CMD_SHOWTIMER   = 65,
        NUM_CMDS        = 66;

    private static final String[] helpInit = {
        "Initialize a board",
        "init <type> <address> <level> <clkfreq0> <clkfreq1> <clkslct1>",
        "type       The board type (default helios)",
        "address    The board address (default 0x280)",
        "level      The interrupt level (default 5)",
        "clkfreq0   The frequency of clock 0 (default 0: 10 MHz)",
        "clkfreq1   The frequency of clock 1 (default 0: 10 MHz)",
        "clkslct1   The source for clock 1 (default 0: internal)",
    };

    private static final String[] helpShow = {
        "Show the current initialization parameters",
        "show",
    };

    private static final String[] helpShowregs = {
        "Read and display the board's registers",
        "showregs",
    };

    private static final String[] helpShowerror = {
        "Show last error, or the string for a code",
        "showerror <code>",
        "code   The error code (default last occuring error)"
    };

    private static final String[] helpFree = {
        "Free the board",
        "free",
    };

    private static final String[] helpAdset = {
        "Set parameters for analog input (ADCs)",
        "adset <chan> <gain> <range> <polarity> <loadcal> <scanint> <diff>",
        "chan      The analog channel to use in sample operations",
        "gain      The input gain to use (0 - 3)",
        "range     The input range to use (0 - 1)",
        "polarity  The polarity of the input (0 - 1)",
        "loadcal   If non-zero, load calibration values",
        "scanint   The interval between successive channels in a scan (0 - 5)",
        "diff      0 = single-ended input; 1 = differential input",
    };

    private static final String[] helpAdsetchan = {
        "Set ADC channel range for sampling",
        "adsetchan <lowchan> <highchan>",
        "lowchan    The first channel to use (0 - 15)",
        "highchan   The last channel to use (0 - 15, >= lowchan)",
    };

    private static final String[] helpAdshow = {
        "Show the current analog input parameters",
        "adshow",
    };

    private static final String[] helpAdshowfifo = {
        "Show the state of the analog input FIFO",
        "adshowfifo",
    };

    private static final String[] helpAdsample = {
        "Read and display current ADC channel value",
        "adsamp",
    };

    private static final String[] helpAdsampavg = {
        "Average several reads of current ADC channel",
        "adsampavg <count>",
        "count   The number of samples to read and average",
    };

    private static final String[] helpAdsampint = {
        "Read ADC channels using interrupts",
        "adsampint <nsamp> <lowchan> <highchan> <rate> <clksrc> <depth> <thresh>",
        "nsamp      The number of samples to read",
        "lowchan    The lowest channel number to read",
        "highchan   The highest channel number to read",
        "rate       The sample rate (Hz); cannot exceed 100000",
        "clksrc     The clock frequency; 0 = 100 KHz, 1 = 10 MHz",
        "depth      The FIFO depth to use, or -1 to disable the FIFO",
        "thresh     The FIFO threshold for triggering readouts",
    };

    private static final String[] helpAdscan = {
        "Read and display a set of ADC channels",
        "adscan <lowchan> <highchan>",
        "lowchan    The first channel to scan (0 - 15)",
        "highchan   The last channel to scan (0 - 15, >= lowscan)",
    };

    private static final String[] helpAdscanavg = {
        "Average several reads of a set of ADC channels",
        "adscanavg <lowchan> <highchan> <count>",
        "lowchan    The first channel to scan (0 - 15)",
        "highchan   The last channel to scan (0 - 15, >= lowscan)",
        "count      The number of samples to read and average",
    };

    private static final String[] helpAdscanint = {
        "Scan ADC channels using interrupts",
        "adscanint <nsamp> <lowchan> <highchan> <rate> <clksrc> <depth> <thresh>",
        "nsamp      The number of samples to read",
        "lowchan    The lowest channel number to read",
        "highchan   The highest channel number to read",
        "rate       The sample rate (Hz); cannot exceed 100000",
        "clksrc     The clock frequency; 0 = 100 KHz, 1 = 10 MHz",
        "depth      The FIFO depth to use, or -1 to disable the FIFO",
        "thresh     The FIFO threshold for triggering readouts",
    };

    private static final String[] helpAdautocal = {
        "Perform auto calibration of analog input",
        "adautocal <range> <bootrange>",
        "range      The range (0 - 15, or 255) to be auto-calibrated",
        "bootrange  The calibration range (0 - 15) to load at boot time",
    };

    private static final String[] helpAdcalverify = {
        "Verify the analog input auto calibration",
        "adcalverify <range> <bootrange>",
        "range      The range (0 - 15) to be verified",
    };

    private static final String[] helpAdctov = {
        "Convert analog input code to voltage",
        "adctov <code>",
        "code   The code value (-32768 - 32767) to be converted",
    };

    private static final String[] helpAdvtoc = {
        "Convert analog input voltage to code",
        "advtoc <voltage>",
        "voltage  The voltage to be converted to a code value",
    };

    private static final String[] helpDaset = {
        "Set parameters for analog output",
        "daset <polarity> <loadcal> <range> <gain> <simupdate> <polenab> <urset> <urchan>",
        "polarity   The polarity of the output (0 - 1)",
        "loadcal    If non-zero, load calibration values first",
        "range      The range to use for conversions: 0 = 5.0, 1 = 10.0",
        "gain       The output gain to use (0 - 2)",
        "simupdate  If non-zero, output all voltages simultaneously",
        "polenab    If non-zero, use software polarity, not hardware",
        "urset      If non-zero, the gain is set on a unique channel",
        "urchan     The unique channel for the gain setting", 
    };

    private static final String[] helpDasetpol = {
        "Set the polarity for analog output",
        "dasetpol <polarity>",
        "polarity  The polarity of the analog output (0 - 1)",
    };

    private static final String[] helpDashow = {
        "Show the current analog output parameters",
        "dashow",
    };

    private static final String[] helpDaautocal = {
        "Perform auto calibration of analog output",
        "daautocal <range>",
        "range  The range (0 - 15) to be auto-calibrated",
    };

    private static final String[] helpDacalverify = {
        "Verify the analog output auto calibration",
        "dacalverify <range>",
        "range  The range (0 - 15) to be verified",
    };

    private static final String[] helpDaconv = {
        "Write a value to an analog output channel",
        "daconv <chan> <code>",
        "chan    The analog output channel to use (0 - 3)",
        "code    The value to output",
    };

    private static final String[] helpDascan = {
        "Write values to a set of analog output channels",
        "dascan",
    };

    private static final String[] helpDactov = {
        "Convert analog output code to voltage",
        "dactov <code>",
        "code   The code value (0 - 4095) to be converted",
    };

    private static final String[] helpDavtoc = {
        "Convert analog output voltage to code",
        "davtoc <voltage>",
        "voltage  The voltage to be converted to a code value",
    };

    private static final String[] helpDioset = {
        "Set digital I/O parameters",
        "dioset <id> <value>",
        "id       The ID of the entity to set",
        "value    The value to set",
    };

    private static final String[] helpDioshow = {
        "Show the current digital I/O parameters",
        "dioshow",
    };

    private static final String[] helpDioinbyte = {
        "Read and display a byte from a digital I/O port",
        "dioinbyte <port>",
        "port    The digital I/O port to read",
    };

    private static final String[] helpDioinword = {
        "Read and display a word (16-bit) from a digital I/O port",
        "dioinword <port>",
        "port    The digital I/O port to read",
    };

    private static final String[] helpDiooutbyte = {
        "Write a byte to a digital I/O port",
        "diooutbyte <port> <value>",
        "port    The digital I/O port to write to",
        "value   The value to write",
    };

    private static final String[] helpDiooutword = {
        "Write a word (16-bit) to a digital I/O port",
        "diooutword <port> <value>",
        "port    The digital I/O port to write to",
        "value   The value to write",
    };

    private static final String[] helpDioinbit = {
        "Read and display a bit from a digital I/O port",
        "dioinbit <port> <bitnum>",
        "port    The digital I/O port to read",
        "bitnum  The number of the bit to read",
    };

    private static final String[] helpDiooutbit = {
        "Write a bit to a digital I/O port",
        "diooutbit <port> <bitnum> <value>",
        "port    The digital I/O port to write to",
        "bitnum  The number of the bit to write",
        "value   The value to set the bit to",
    };

    private static final String[] helpDiosetbit = {
        "Set a bit in a digital I/O port",
        "diosetbit <port> <bitnum>",
        "port    The digital I/O port to write to",
        "bitnum  The number of the bit to set",
    };

    private static final String[] helpDioclearbit = {
        "Clear a bit in a digital I/O port",
        "dioclearbit <port> <bitnum>",
        "port    The digital I/O port to write to",
        "bitnum  The number of the bit to clear",
    };

    private static final String[] helpCtrshow = {
        "Show the current counter parameters",
        "ctrshow [<option>]",
        "option  Display option bits: 1 = times, 2 = counts",
    };

    private static final String[] helpCtrsetvalue = {
        "Set the value of a counter",
        "ctrsetvalue <value> <counter> <code>",
        "value    The value to initialize the counter to",
        "counter  The counter to be initialized",
        "code     The control code to use",
    };

    private static final String[] helpCtrsetrate = {
        "Set the frequency of a counter",
        "ctrsetrate <freq> <counter>",
        "freq     The frequency to initialize the counter to",
        "counter  The counter to be initialized (default all)",
    };

    private static final String[] helpCtrread = {
        "Read the value of a counter",
        "ctrread <counter>",
        "counter  The counter to be read",
    };

    private static final String[] helpIntshow = {
        "Show counter interrupt statistics",
        "intshow [<option>]",
        "option  Display option bits: 1 = intervals, 2 = counts, 4 = combined",
    };

    private static final String[] helpIntstart = {
        "Start counter interrupts",
        "intstart <rate> <clock> <source> [<method>]",
        "rate     The interrupt rate, or 0 if pre-programmed",
        "clock    The clock frequency to use: 0 = 100 KHz, 1 = 10 MHz",
        "source   The interrupt source: 0 = ctr0, 1 = ctr1, 2 = ext (dflt 1)",
        "method   The interrupt handling method: 0 = local, 1 = DSC (dflt 0)",
    };

    private static final String[] helpIntstop = {
        "Stop counter interrupts",
        "intstop",
    };

    private static final String[] helpIntcancel = {
        "Cancel active interrupt processing",
        "intcancel <type>",
        "type   The type (cou, dioi, ad, da or all) of interrupt to cancel",
    };

    private static final String[] helpIntpause = {
        "Pause all interrupt processing",
        "intpause",
    };

    private static final String[] helpIntresume = {
        "Resume all interrupt processing",
        "intresume",
    };

    private static final String[] helpDshow = {
        "Show the current direct I/O parameters",
        "dshow",
    };

    private static final String[] helpInp = {
        "Read and display a byte from I/O space",
        "inp <address>",
        "address  The address from which to read a byte",
    };

    private static final String[] helpInpw = {
        "Read and display a 16-bit word from I/O space",
        "inpw <address>",
        "address  The address from which to read a 16-bit word",
    };

    private static final String[] helpInpl = {
        "Read and display a 32-bit word from I/O space",
        "inpl <address>",
        "address  The address from which to read a 32-bit integer",
    };

    private static final String[] helpInpws = {
        "Read and display 16-bit words from I/O space FIFO",
        "inpws <address> <count>",
        "address  The address from which to read 16-bit words",
        "count    The number of words to read at <address>",
    };

    private static final String[] helpOut = {
        "Write a byte to I/O space",
        "out <address> <value>",
        "address  The address at which to write a byte",
        "value    The value to write",
    };

    private static final String[] helpOutw = {
        "Write a 16-bit word to I/O space",
        "outw <address> <value>",
        "address  The address at which to write a 16-bit word",
        "value    The value to write",
    };

    private static final String[] helpOutl = {
        "Write a 32-bit word to I/O space",
        "outl <address> <value>",
        "address  The address at which to write a 32-bit integer",
        "value    The value to write",
    };

    private static final String[] helpOutws = {
        "Write 16-bit words to I/O space FIFO",
        "outws <address> <value1>... <value8>",
        "address  The address at which to write 16-bit words",
        "valuen   The values to write at <address>",
    };

    private static final String[] helpRegread = {
        "Read a board register",
        "regread <register>",
        "register  The register number from which to read a byte",
    };

    private static final String[] helpRegwrite = {
        "Write to a board register",
        "regwrite <register> <value>",
        "register  The register number to which to write a byte",
        "value     The value to write",
    };

    private static final String[] helpTimeread = {
        "Time reading from an address",
        "timeread <count> <address> <width>",
        "count    The number of times to perform the read",
        "address  The address at which to read (default board address)",
        "width    The size (1, 2 or 4) of the word to read (default 1)",
    };

    private static final String[] helpDump = {
        "Dump memory contents",
        "dump <address> <nbytes>",
        "address  The address at which to start the dump",
        "nbytes   The number of bytes to dump",
    };

    private static final String[] helpShowtimer = {
        "Show the value of the interval timer (ms)",
        "showtimer",
    };

    private static final String[] helpShowboard = {
        "Show the code for a named board type",
        "showboard",
    };

    private static final String[] helpThrshow = {
        "Show thread information",
        "thrshow",
    };

    private static final String[] helpThrstop = {
        "Stop a thread",
        "thrstop <name>",
    };

    private static final String[] helpThrdaemon = {
        "Check or set a thread's daemon status",
        "thrdaemon <name> <op>",
    };

    private static final String fifoStatus[] = { "U",   "E",   "H",   "HE",
                                                 "F",  "FE",  "FH",  "FHE",
                                                 "O",  "OE",  "OH",  "OHE",
                                                "OF", "OFE", "OFH", "OFHE"};

    private static final int MAX_INT_SAVED = 64;

    private final Output out;
    private Dscud board;
    private CmndProc proc;
    private CmndProc.Lookup brdTypes, diffTypes, gainTypes, sintTypes, polTypes,
                   rngTypes, itypTypes;

    // Board parameters
    private short   baseAddr = 0x280;
    private byte    boardType = Dscud.DSC_HELIOS,
                    intLevel = 5,
                    clkFreq0 = 0,
                    clkFreq1 = 0,
                    clkSlct1 = 0;
    private boolean initd = false;

    // AD parameters
    private int     adAvgCount = 1,
                    adSmpCount = 1,
                    adCurr = 0,
                    adThresh = 0;
    private short   adFifoDepth = 0;
    private byte    adChan = 0,
                    adGain = Dscud.GAIN_1,
                    adRange = Dscud.RANGE_10,
                    adPolarity = Dscud.BIPOLAR,
                    adScanIntv = 0,
                    adDiff = 0,
                    adLowChan = 0,
                    adHighChan = Dscud.ADCS_MAX_CHANNELS - 1,
                    adLow = 0,
                    adHigh = 0,
                    adClksrc = 0;
    private float   adRate = 100000F;
    private boolean adLoadCal = false;
        
    // DA parameters
    private int     daCode = 0,
                    daCodes[] = {0, 0, 0, 0},
                    daChanEnab[] = {0, 0, 0, 0};
    private byte    daPolarity = 0,
                    daGain = Dscud.GAIN_1,
                    daUrChan = 0,
                    daChan = 0;
    private float   daRange = 5.0F;
    private boolean daLoadCal = false,
                    daSimUpd = false,
                    daPolEnab = false,
                    daUrSet = false;

    // Digital I/O parameters
    private short   dioWord = 0;
    private byte    dioPort = 0,
                    dioByte = 0,
                    dioBit = 0,
                    dioConfig[] = {0, 0};

    // Direct I/O parameters
    private int     dAddress = 0,
                    dLong = 0,
                    dCount = 1;
    private short   dReg = 0,
                    dWord = 0,
                    dWords[] = new short[8];
    private byte    dByte = 0,
                    dWidth = 1;

    // Counter parameters
    private int     ctrValue = 0;
    private byte    ctrSelect = 1,
                    ctrCode = 0;
    private float   ctrRate = 1.0F;

    // Interrupt parameters
    private long    intStart = 0,
                    intStop = 0,
                    intLast = 0,
                    intSavst = 0,
                    intTimes[] = new long[MAX_INT_SAVED];
    private int     intType = 0,
                    intCount = 0,
                    intSaved = 0,
                    intCounts[] = new int[MAX_INT_SAVED];
    private byte    intClock = 0,
                    intSrce = 1;
    private float   intRate = 100.0F;
    private boolean intDsc = false;

    // Dump parameters
    private int     dmpAddr = 0,
                    dmpCount = 4,
                    dmpWidth = 4;


/**
 ***************************************************************************
 **
 **  Main constructor
 **
 ***************************************************************************
 */
    TestDscud(Output iOut) throws IOException
    {
        out = (iOut != null) ? iOut : new ConsOut();

        CmndProc.Command cmnd = new CmndProc.Command(NUM_CMDS);
        cmnd.add("init",        CMD_INIT,        helpInit);
        cmnd.add("show",        CMD_SHOW,        helpShow);
        cmnd.add("showregs",    CMD_SHOWREGS,    helpShowregs);
        cmnd.add("showerror",   CMD_SHOWERROR,   helpShowerror);
        cmnd.add("free",        CMD_FREE,        helpFree);
        cmnd.add("adset",       CMD_ADSET,       helpAdset);
        cmnd.add("adsetchan",   CMD_ADSETCHAN,   helpAdsetchan);
        cmnd.add("adshow",      CMD_ADSHOW,      helpAdshow);
        cmnd.add("adshowfifo",  CMD_ADSHOWFIFO,  helpAdshowfifo);
        cmnd.add("adsamp",      CMD_ADSAMPLE,    helpAdsample);
        cmnd.add("adsampavg",   CMD_ADSAMPAVG,   helpAdsampavg);
        cmnd.add("adsampint",   CMD_ADSAMPINT,   helpAdsampint);
        cmnd.add("adscan",      CMD_ADSCAN,      helpAdscan);
        cmnd.add("adscanavg",   CMD_ADSCANAVG,   helpAdscanavg);
        cmnd.add("adscanint",   CMD_ADSCANINT,   helpAdscanint);
        cmnd.add("adautocal",   CMD_ADAUTOCAL,   helpAdautocal);
        cmnd.add("adcalverify", CMD_ADCALVERIFY, helpAdcalverify);
        cmnd.add("adctov",      CMD_ADCTOV,      helpAdctov);
        cmnd.add("advtoc",      CMD_ADVTOC,      helpAdvtoc);
        cmnd.add("daset",       CMD_DASET,       helpDaset);
        cmnd.add("dasetpol",    CMD_DASETPOL,    helpDasetpol);
        cmnd.add("dashow",      CMD_DASHOW,      helpDashow);
        cmnd.add("daconv",      CMD_DACONV,      helpDaconv);
        cmnd.add("dascan",      CMD_DASCAN,      helpDascan);
        cmnd.add("daautocal",   CMD_DAAUTOCAL,   helpDaautocal);
        cmnd.add("dacalverify", CMD_DACALVERIFY, helpDacalverify);
        cmnd.add("dactov",      CMD_DACTOV,      helpDactov);
        cmnd.add("davtoc",      CMD_DAVTOC,      helpDavtoc);
        cmnd.add("dioset",      CMD_DIOSET,      helpDioset);
        cmnd.add("dioshow",     CMD_DIOSHOW,     helpDioshow);
        cmnd.add("dioinbyte",   CMD_DIOINBYTE,   helpDioinbyte);
        cmnd.add("dioinword",   CMD_DIOINWORD,   helpDioinword);
        cmnd.add("diooutbyte",  CMD_DIOOUTBYTE,  helpDiooutbyte);
        cmnd.add("diooutword",  CMD_DIOOUTWORD,  helpDiooutword);
        cmnd.add("dioinbit",    CMD_DIOINBIT,    helpDioinbit);
        cmnd.add("diooutbit",   CMD_DIOOUTBIT,   helpDiooutbit);
        cmnd.add("diosetbit",   CMD_DIOSETBIT,   helpDiosetbit);
        cmnd.add("dioclearbit", CMD_DIOCLRBIT,   helpDioclearbit);
        cmnd.add("ctrshow",     CMD_CTRSHOW,     helpCtrshow);
        cmnd.add("ctrsetvalue", CMD_CTRSETVALUE, helpCtrsetvalue);
        cmnd.add("ctrsetrate",  CMD_CTRSETRATE,  helpCtrsetrate);
        cmnd.add("ctrread",     CMD_CTRREAD,     helpCtrread);
        cmnd.add("intshow",     CMD_INTSHOW,     helpIntshow);
        cmnd.add("intstart",    CMD_INTSTART,    helpIntstart);
        cmnd.add("intstop",     CMD_INTSTOP,     helpIntstop);
        cmnd.add("intcancel",   CMD_INTCANCEL,   helpIntcancel);
        cmnd.add("intpause",    CMD_INTPAUSE,    helpIntpause);
        cmnd.add("intresume",   CMD_INTRESUME,   helpIntresume);
        cmnd.add("dshow",       CMD_DSHOW,       helpDshow);
        cmnd.add("inp",         CMD_INP,         helpInp);
        cmnd.add("inpw",        CMD_INPW,        helpInpw);
        cmnd.add("inpl",        CMD_INPL,        helpInpl);
        cmnd.add("inpws",       CMD_INPWS,       helpInpws);
        cmnd.add("out",         CMD_OUT,         helpOut);
        cmnd.add("outw",        CMD_OUTW,        helpOutw);
        cmnd.add("outl",        CMD_OUTL,        helpOutl);
        cmnd.add("outws",       CMD_OUTWS,       helpOutws);
        cmnd.add("regread",     CMD_REGREAD,     helpRegread);
        cmnd.add("regwrite",    CMD_REGWRITE,    helpRegwrite);
        cmnd.add("timeread",    CMD_TIMEREAD,    helpTimeread);
        cmnd.add("dump",        CMD_DUMP,        helpDump);
        cmnd.add("showtimer",   CMD_SHOWTIMER,   helpShowtimer);
        cmnd.add("showboard",   CMD_SHOWBOARD,   helpShowboard);
        cmnd.add("thrshow",     CMD_THRSHOW,     helpThrshow);
        cmnd.add("thrstop",     CMD_THRSTOP,     helpThrstop);
        cmnd.add("thrdaemon",   CMD_THRDAEMON,   helpThrdaemon);

        brdTypes = new CmndProc.Lookup(1);
        brdTypes.add("Helios", Dscud.DSC_HELIOS);

        diffTypes = new CmndProc.Lookup(2);
        diffTypes.add("SINGLE_ENDED", Dscud.SINGLE_ENDED);
        diffTypes.add("DIFFERENTIAL", Dscud.DIFFERENTIAL);

        gainTypes = new CmndProc.Lookup(4);
        gainTypes.add("GAIN_1", Dscud.GAIN_1);
        gainTypes.add("GAIN_2", Dscud.GAIN_2);
        gainTypes.add("GAIN_4", Dscud.GAIN_4);
        gainTypes.add("GAIN_8", Dscud.GAIN_8);

        sintTypes = new CmndProc.Lookup(6);
        sintTypes.add("SCAN_INTERVAL_4", Dscud.SCAN_INTERVAL_4);
        sintTypes.add("SCAN_INTERVAL_5", Dscud.SCAN_INTERVAL_5);
        sintTypes.add("SCAN_INTERVAL_9", Dscud.SCAN_INTERVAL_9);
        sintTypes.add("SCAN_INTERVAL_10", Dscud.SCAN_INTERVAL_10);
        sintTypes.add("SCAN_INTERVAL_15", Dscud.SCAN_INTERVAL_15);
        sintTypes.add("SCAN_INTERVAL_20", Dscud.SCAN_INTERVAL_20);

        polTypes = new CmndProc.Lookup(2);
        polTypes.add("BIPOLAR",  Dscud.BIPOLAR);
        polTypes.add("UNIPOLAR", Dscud.UNIPOLAR);

        rngTypes = new CmndProc.Lookup(2);
        rngTypes.add("RANGE_5",  Dscud.RANGE_5);
        rngTypes.add("RANGE_10", Dscud.RANGE_10);

        itypTypes = new CmndProc.Lookup(10);
        itypTypes.add("Inactive", 0);
        itypTypes.add("AD",       Dscud.INT_TYPE_AD);
        itypTypes.add("DA",       Dscud.INT_TYPE_DA);
        itypTypes.add("DIOIN",    Dscud.INT_TYPE_DIOIN);
        itypTypes.add("USER",     Dscud.INT_TYPE_USER);
        itypTypes.add("COUNTER",  Dscud.INT_TYPE_COUNTER);
        itypTypes.add("DIOREAD",  Dscud.INT_TYPE_DIOREAD);
        itypTypes.add("OPTO",     Dscud.INT_TYPE_OPTO);
        itypTypes.add("DIOWRITE", Dscud.INT_TYPE_DIOWRITE);
        itypTypes.add("ALL",      INT_TYPE_ALL);

        board = new Dscud();
        proc = new CmndProc();
        proc.add(this, cmnd);

        reader  = new ConsoleReader();

    }

    TestDscud() throws IOException
    {
        this(null);
    }


/**
 ***************************************************************************
 **
 **  Main program
 **
 **  Loops reading and processing each new typed command line.
 **
 ***************************************************************************
 */
    public static void main(String[] args) throws IOException
    {
        TestDscud test = new TestDscud();
        test.run();
        test.cleanup();

        System.exit(0);
    }


/**
 ***************************************************************************
 **
 **  Routine to run the test
 **
 ***************************************************************************
 */
    public void run() throws IOException
    {
        while (true) {
            String line = reader.readLine(">> ");
            if (line == null) break;
            try {
                if (!proc.process(line)) break;
            }
            catch (DscudException e) {
                out.println(e);
            }
        }
    }


/**
 ***************************************************************************
 **
 **  Routine to dispatch a command
 **
 ***************************************************************************
 */
    public boolean dispatch(int cCode, Scanner scan)
    {
        int found;
        Object[] args = new Object[16];

        switch (cCode) {

        case CMD_INIT:
            if ((found = CmndProc.scanArgs(scan, "swbbbb", args)) >= 0) {
                if (initd) {
                    initd = false;
                    board.freeBoard();
                }
                if ((found & 0x01) != 0) {
                    String sType = (String)args[0];
                    if (!sType.equals("*")) {
                        int type = brdTypes.encode(sType, true);
                        if (type >= 0) boardType = (byte)type;
                    }
                }
                if ((found & 0x02) != 0)
                    baseAddr = (Short)args[1];
                if ((found & 0x04) != 0)
                    intLevel = (Byte)args[2];
                if ((found & 0x08) != 0)
                    clkFreq0 = (Byte)args[3];
                if ((found & 0x10) != 0)
                    clkFreq1 = (Byte)args[4];
                if ((found & 0x20) != 0)
                    clkSlct1 = (Byte)args[5];
                dAddress = baseAddr;
                board.initBoard(boardType, baseAddr, intLevel, clkFreq0,
                                clkFreq1, clkSlct1);
                initd = true;
            }
            break;

        case CMD_SHOW:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0) {
                out.println("Initialization parameters:");
                out.println("  Board type       = "
                             + brdTypes.decode((int)boardType));
                out.format("  Base address     = %04x\n", baseAddr);
                out.println("  Interrupt level  = " + intLevel);
                out.format("  Clock0 frequency = %s (%s)\n", clkFreq0,
                           (clkFreq0 == 0) ? "10 MHz" : "1 MHz");
                out.format("  Clock1 frequency = %s (%s)\n", clkFreq1,
                           (clkFreq1 == 0) ? "10 MHz" : "100 KHz");
                out.format("  Clock1 source    = %s (%s)\n", clkSlct1,
                           (clkSlct1 == 0) ? "Internal" : "External");
            }
            break;

        case CMD_SHOWREGS:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0)
                showRegs();
            break;

        case CMD_SHOWERROR:
            if ((found = CmndProc.scanArgs(scan, "b", args)) >= 0) {
                if ((found & 0x01) != 0) {
                    byte eCode = (Byte)args[0];
                    out.println(eCode + ": " + Dscud.getErrorString(eCode));
                }
                else {
                    String[] text = new String[1];
                    int eCode = Dscud.getLastError(text);
                    out.println(eCode + ": " + text[0]);
                }
            }
            break;

        case CMD_FREE:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0) {
                initd = false;
                board.freeBoard();
            }
            break;

        case CMD_ADSET:
            if ((found = CmndProc.scanArgs(scan, "bbbbbbb", args)) >= 0) {
                if ((found & 0x01) != 0)
                    adChan = (Byte)args[0];
                if ((found & 0x02) != 0)
                    adGain = (Byte)args[1];
                if ((found & 0x04) != 0)
                    adRange = (Byte)args[2];
                if ((found & 0x08) != 0)
                    adPolarity = (Byte)args[3];
                if ((found & 0x10) != 0)
                    adLoadCal = (Byte)args[4] != 0;
                if ((found & 0x20) != 0)
                    adScanIntv = (Byte)args[5];
                if ((found & 0x40) != 0)
                    adDiff = (Byte)args[6];
                board.adSetSettings(adChan, adGain, adRange, adPolarity,
                                    adLoadCal, adScanIntv, adDiff);
                adCurr = adLow = adHigh = adChan;
            }
            break;

        case CMD_ADSETCHAN:
            if ((found = CmndProc.scanArgs(scan, "bb", args)) >= 0) {
                short[] value = new short[Dscud.ADCS_MAX_CHANNELS];
                if ((found & 0x01) != 0)
                    adLowChan = (Byte)args[0];
                if ((found & 0x02) != 0)
                    adHighChan = (Byte)args[1];
                board.adSetChannel(adLowChan, adHighChan);
                adCurr = adLow = adLowChan;
                adHigh = adHighChan;
            }
            break;

        case CMD_ADSHOW:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0) {
                out.println("Analog input parameters:");
                out.println("  Sample channel    = " + adChan);
                out.format("  Gain              = %s (%s)\n", adGain,
                           gainTypes.decode(adGain));
                out.format("  Range             = %s (%s)\n", adRange,
                           rngTypes.decode((int)adRange));
                out.format("  Polarity          = %s (%s)\n", adPolarity,
                           polTypes.decode((int)adPolarity));
                out.println("  Load calibration  = " + adLoadCal);
                out.format("  Scan interval     = %s (%s)\n", adScanIntv,
                           sintTypes.decode((int)adScanIntv));
                out.format("  Differential      = %s (%s)\n", adDiff,
                           diffTypes.decode(adDiff));
                out.println("  Low channel reqd  = " + adLowChan);
                out.println("  High channel reqd = " + adHighChan);
                out.println("  Averaging count   = " + adAvgCount);
                out.println("  Intr sample count = " + adSmpCount);
                out.println("  Interrupt rate    = " + adRate);
                out.println("  Intr clock source = " + adClksrc);
                out.println("  Intr FIFO depth   = " + adFifoDepth);
                out.println("  Intr threshold    = " + adThresh);
                out.println("  Current channel   = " + adCurr);
                out.println("  Low channel set   = " + adLow);
                out.println("  High channel set  = " + adHigh);
            }
            break;

        case CMD_ADSHOWFIFO:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0) {
                out.print("Analog input FIFO status: ");
                byte reg5 = board.registerRead(Dscud.REG_5);
                byte reg6 = board.registerRead(Dscud.REG_6);
                board.registerWrite(Dscud.REG_1, (byte)2);
                byte reg12 = board.registerRead(Dscud.REG_12);
                if ((reg12 & 1) == 0)
                    out.format("mode = basic, threshold = %s, depth = %s.\n",
                               reg5, reg6);
                else {
                    String fstatus = fifoStatus[reg6 & 0x0f];
                    int depth = (((int)reg6 & 0xf0) << 8) | ((int)reg5 & 0xff);
                    out.format("mode = expanded, status = %s, depth = %s.\n",
                               fstatus, depth);
                }
            }
            break;

        case CMD_ADSAMPLE:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0) {
                short value = board.adSample();
                double volts = board.adCodeToVoltage(adPolarity, adGain,
                                                     adRange, value);
                out.format("Channel %s: %04x ", adCurr, value);
                if (volts > -1.0 && volts < 1.0)
                    out.format("(%.5f)\n", volts);
                else
                    out.format("(%.6g)\n", volts);
                if (adCurr == adHigh)
                    adCurr = adLow;
                else if (++adCurr >= Dscud.ADCS_MAX_CHANNELS)
                    adCurr = 0;
            }
            break;

        case CMD_ADSAMPAVG:
            if ((found = CmndProc.scanArgs(scan, "i", args)) >= 0) {
                if ((found & 0x01) != 0)
                    adAvgCount = (Integer)args[0];
                double value = board.adSampleAvg(adAvgCount);
                double volts = board.adCodeToVoltage(adPolarity, adGain,
                                                     adRange,
                                                     (short)(value + 0.5));
                out.format("Channel %s: %.1f ", adCurr, value);
                if (volts > -1.0 && volts < 1.0)
                    out.format("(%.5f)\n", volts);
                else
                    out.format("(%.6g)\n", volts);
                int nchan = adHigh - adLow + 1;
                if (nchan <= 0)
                    nchan += Dscud.ADCS_MAX_CHANNELS;
                for (int j = 0; j < adAvgCount % nchan; j++) {
                    if (adCurr == adHigh)
                        adCurr = adLow;
                    else if (++adCurr >= Dscud.ADCS_MAX_CHANNELS)
                        adCurr = 0;
                }
            }
            break;

        case CMD_ADSAMPINT:
            if ((found = CmndProc.scanArgs(scan, "wbbfbwi", args)) >= 0) {
                if ((found & 0x01) != 0)
                    adSmpCount = (Short)args[0];
                if ((found & 0x02) != 0)
                    adLowChan = (Byte)args[1];
                if ((found & 0x04) != 0)
                    adHighChan = (Byte)args[2];
                if ((found & 0x08) != 0)
                    adRate = (Float)args[3];
                if ((found & 0x10) != 0)
                    adClksrc = (Byte)args[4];
                if ((found & 0x20) != 0)
                    adFifoDepth = (Short)args[5];
                if ((found & 0x40) != 0)
                    adThresh = (Integer)args[6];
                short[] value = new short[adSmpCount];
                board.adSampleInt(adSmpCount, adLowChan, adHighChan, adRate,
                                  adClksrc, adFifoDepth, adThresh, value);
                adCurr = adLow = adLowChan;
                adHigh = adHighChan;
                for (int j = 0; j < adSmpCount; j++) {
                    if (j > 0)
                        out.print(((j & 3) == 0) ? "\n" : " ");
                    double volts = board.adCodeToVoltage(adPolarity, adGain,
                                                         adRange, value[j]);
                    out.format("%2s: %04x ", adCurr, value[j]);
                    if (volts > -1.0 && volts < 1.0)
                        out.format("(% -8.5f)", volts);
                    else
                        out.format("(% -8.6g)", volts);
                    if (adCurr == adHigh)
                        adCurr = adLow;
                    else if (++adCurr >= Dscud.ADCS_MAX_CHANNELS)
                        adCurr = 0;
                }
                out.println();
            }
            break;

        case CMD_ADSCAN:
            if ((found = CmndProc.scanArgs(scan, "bb", args)) >= 0) {
                short[] value = new short[Dscud.ADCS_MAX_CHANNELS];
                if ((found & 0x01) != 0)
                    adLowChan = (Byte)args[0];
                if ((found & 0x02) != 0)
                    adHighChan = (Byte)args[1];
                board.adScan(adLowChan, adHighChan, value);
                adLow = (adLowChan < adHighChan) ? adLowChan : adHighChan;
                adHigh = (adLowChan > adHighChan) ? adLowChan : adHighChan;
                for (int j = 0, adCurr = adLow; j < adHigh - adLow + 1;
                     j++, adCurr++) {
                    if (j > 0)
                        out.print(((j & 3) == 0) ? "\n" : " ");
                    double volts = board.adCodeToVoltage(adPolarity, adGain,
                                                         adRange, value[j]);
                    out.format("%2s: %04x ", adCurr, value[j]);
                    if (volts > -1.0 && volts < 1.0)
                        out.format("(% -8.5f)", volts);
                    else
                        out.format("(% -8.6g)", volts);
                }
                out.println();
            }
            break;

        case CMD_ADSCANAVG:
            if ((found = CmndProc.scanArgs(scan, "bbi", args)) >= 0) {
                double[] value = new double[Dscud.ADCS_MAX_CHANNELS];
                double volts;
                if ((found & 0x01) != 0)
                    adLowChan = (Byte)args[0];
                if ((found & 0x02) != 0)
                    adHighChan = (Byte)args[1];
                if ((found & 0x04) != 0)
                    adAvgCount = (Integer)args[2];
                board.adScanAvg(adLowChan, adHighChan, value, adAvgCount);
                adLow = (adLowChan < adHighChan) ? adLowChan : adHighChan;
                adHigh = (adLowChan > adHighChan) ? adLowChan : adHighChan;
                for (int j = 0, adCurr = adLow; j < adHigh - adLow + 1;
                     j++, adCurr++) {
                    if (j > 0)
                        out.print(((j % 3) == 0) ? "\n" : "    ");
                    volts = board.adCodeToVoltage(adPolarity, adGain, adRange,
                                                  (short)(value[j] + 0.5));
                    out.format("%2s: %7.1f ", adCurr, value[j]);
                    if (volts > -1.0 && volts < 1.0)
                        out.format("(% -8.5f)", volts);
                    else
                        out.format("(% -8.6g)", volts);
                }
                out.println();
            }
            break;

        case CMD_ADSCANINT:
            if ((found = CmndProc.scanArgs(scan, "wbbfbwi", args)) >= 0) {
                if ((found & 0x01) != 0)
                    adSmpCount = (Short)args[0];
                if ((found & 0x02) != 0)
                    adLowChan = (Byte)args[1];
                if ((found & 0x04) != 0)
                    adHighChan = (Byte)args[2];
                if ((found & 0x08) != 0)
                    adRate = (Float)args[3];
                if ((found & 0x10) != 0)
                    adClksrc = (Byte)args[4];
                if ((found & 0x20) != 0)
                    adFifoDepth = (Short)args[5];
                if ((found & 0x40) != 0)
                    adThresh = (Integer)args[6];
                short[] value = new short[adSmpCount];
                board.adScanInt(adSmpCount, adLowChan, adHighChan, adRate,
                                adClksrc, adFifoDepth, adThresh, value);
                adCurr = adLow = adLowChan;
                adHigh = adHighChan;
                for (int j = 0; j < adSmpCount; j++) {
                    if (j > 0)
                        out.print(((j & 3) == 0) ? "\n" : " ");
                    double volts = board.adCodeToVoltage(adPolarity, adGain,
                                                         adRange, value[j]);
                    out.format("%2s: %04x ", adCurr, value[j]);
                    if (volts > -1.0 && volts < 1.0)
                        out.format("(% -8.5f)", volts);
                    else
                        out.format("(% -8.6g)", volts);
                    if (adCurr == adHigh)
                        adCurr = adLow;
                    else if (++adCurr >= Dscud.ADCS_MAX_CHANNELS)
                        adCurr = 0;
                }
                out.println();
            }
            break;

        case CMD_ADAUTOCAL:
            if ((found = CmndProc.scanArgs(scan, "Bb", args)) >= 0) {
                byte range, bootRange;
                range = (Byte)args[0];
                if ((found & 0x02) != 0)
                    bootRange = (Byte)args[1];
                else
                    bootRange = (byte)0;
                board.adAutoCal(range, bootRange);
            }
            break;

        case CMD_ADCALVERIFY:
            if ((found = CmndProc.scanArgs(scan, "B", args)) >= 0) {
                byte range;
                float[] offset = {0F}, gain = {0F};
                range = (Byte)args[0];
                board.adCalVerify(range, offset, gain);
                out.format("Offset error = %1.4f, gain error = %1.4f\n",
                           offset[0], gain[0]);
            }
            break;

        case CMD_ADCTOV:
            if ((found = CmndProc.scanArgs(scan, "W", args)) >= 0) {
                short code = (Short)args[0];
                double volts = board.adCodeToVoltage(adPolarity, adGain,
                                                     adRange, code);
                if (volts > -1.0 && volts < 1.0)
                    out.format("Voltage = %.5f\n", volts);
                else
                    out.format("Voltage = %.6g\n", volts);
            }
            break;

        case CMD_ADVTOC:
            if ((found = CmndProc.scanArgs(scan, "D", args)) >= 0) {
                double volts = (Double)args[0];
                int code = board.voltageToAdCode(adPolarity, adGain, adRange,
                                                 volts);
                out.println("Code = " + code);
            }
            break;

        case CMD_DASET:
            if ((found = CmndProc.scanArgs(scan, "bbbbbbbb", args)) >= 0) {
                if ((found & 0x01) != 0)
                    daPolarity = (Byte)args[0];
                if ((found & 0x02) != 0)
                    daLoadCal = (Byte)args[1] != 0;
                if ((found & 0x04) != 0)
                    daRange = (Byte)args[2] != 0 ? 10.0F : 5.0F;
                if ((found & 0x08) != 0)
                    daGain = (Byte)args[3];
                if ((found & 0x10) != 0)
                    daSimUpd = (Byte)args[4] != 0;
                if ((found & 0x20) != 0)
                    daPolEnab = (Byte)args[5] != 0;
                if ((found & 0x40) != 0)
                    daUrSet = (Byte)args[6] != 0;
                if ((found & 0x80) != 0)
                    daUrChan = (Byte)args[7];
                board.daSetSettings(daPolarity, daLoadCal, daRange, daGain,
                                    daSimUpd, daPolEnab, daUrSet, daUrChan);
            }
            break;

        case CMD_DASETPOL:
            if ((found = CmndProc.scanArgs(scan, "b", args)) >= 0) {
                if ((found & 0x01) != 0)
                    daPolarity = (Byte)args[0];
                board.daSetPolarity(daPolarity);
            }
            break;

        case CMD_DASHOW:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0) {
                out.println("Analog output parameters:");
                out.format("  Polarity            = %s (%s)\n", daPolarity,
                           polTypes.decode(1 - daPolarity));
                out.println("  Load calibration    = " + daLoadCal);
                out.println("  Range               = " + daRange);
                out.format("  Gain                = %s (%s)\n", daGain,
                           gainTypes.decode((int)daGain));
                out.println("  Simultaneous update = " + daSimUpd);
                out.println("  Polarity enable     = " + daPolEnab);
                out.println("  Set unique range    = " + daUrSet);
                out.println("  Unique channel      = " + daUrChan);
            }
            break;

        case CMD_DACONV:
            if ((found = CmndProc.scanArgs(scan, "bi", args)) >= 0) {
                if ((found & 0x01) != 0)
                    daChan = (Byte)args[0];
                if ((found & 0x02) != 0)
                    daCode = (Integer)args[1];
                board.daConvert(daChan, daCode);
            }
            break;

        case CMD_DASCAN:
            if ((found = CmndProc.scanArgs(scan, "bibibibi", args)) >= 0) {
                if (found != 0) {
                    for (int j = 0; j < daChanEnab.length; j++)
                        daChanEnab[j] = 0;
                    for (int j = 0; j < 2 * daChanEnab.length; j += 2) {
                        if ((found & (1 << j)) != 0) {
                            byte chan = (Byte)args[j];
                            daChanEnab[chan] = 1;
                            if ((found & (1 << (j + 1))) != 0)
                                daCodes[chan] = (Integer)args[j + 1];
                        }
                    }
                }
                board.daConvertScan(daChanEnab, daCodes);
            }
            break;

        case CMD_DAAUTOCAL:
            if ((found = CmndProc.scanArgs(scan, "B", args)) >= 0) {
                byte range;
                range = (Byte)args[0];
                board.daAutoCal(range);
            }
            break;

        case CMD_DACALVERIFY:
            if ((found = CmndProc.scanArgs(scan, "B", args)) >= 0) {
                byte range;
                float[] offset = {0F}, gain = {0F};
                range = (Byte)args[0];
                board.daCalVerify(range, offset, gain);
                out.format("Offset error = %1.4f, gain error = %1.4f\n",
                           offset[0], gain[0]);
            }
            break;

        case CMD_DACTOV:
            if ((found = CmndProc.scanArgs(scan, "I", args)) >= 0) {
                int code = (Integer)args[0];
                double volts = board.daCodeToVoltage(daPolarity, daGain,
                                                     daRange, code);
                out.format("Voltage = %1.5f\n", volts);
            }
            break;

        case CMD_DAVTOC:
            if ((found = CmndProc.scanArgs(scan, "D", args)) >= 0) {
                double volts = (Double)args[0];
                int code = board.voltageToDaCode(daPolarity, daGain, daRange,
                                                 volts);
                out.println("Code = " + code);
            }
            break;

        case CMD_DIOSET:
            if ((found = CmndProc.scanArgs(scan, "bb", args)) >= 0) {
                if ((found & 0x01) != 0)
                    dioConfig[0] = (Byte)args[0];
                if ((found & 0x02) != 0)
                    dioConfig[1] = (Byte)args[1];
                board.dioSetConfig(dioConfig);
            }
            break;

        case CMD_DIOSHOW:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0) {
                out.println("Digital I/O parameters:");
                out.format("  Configuration = (%02x, %02x)\n", dioConfig[0],
                           dioConfig[1]);
                out.println("  Port          = " + dioPort);
                out.format("  Word output   = %04x\n", dioWord);
                out.format("  Byte output   = %02x\n", dioByte);
                out.println("  Bit number    = " + dioBit);
            }
            break;

        case CMD_DIOINBYTE:
            if ((found = CmndProc.scanArgs(scan, "b", args)) >= 0) {
                if ((found & 0x01) != 0)
                    dioPort = (Byte)args[0];
                byte value = board.dioInputByte(dioPort);
                out.format("Value = 0x%02x\n", value);
            }
            break;

        case CMD_DIOOUTBYTE:
            if ((found = CmndProc.scanArgs(scan, "bb", args)) >= 0) {
                if ((found & 0x01) != 0)
                    dioPort = (Byte)args[0];
                if ((found & 0x02) != 0)
                    dioByte = (Byte)args[1];
                board.dioOutputByte(dioPort, dioByte);
            }
            break;

        case CMD_DIOINWORD:
            if ((found = CmndProc.scanArgs(scan, "b", args)) >= 0) {
                if ((found & 0x01) != 0)
                    dioPort = (Byte)args[0];
                short value = board.dioInputWord(dioPort);
                out.format("Value = 0x%04x\n", value);
            }
            break;

        case CMD_DIOOUTWORD:
            if ((found = CmndProc.scanArgs(scan, "bw", args)) >= 0) {
                if ((found & 0x01) != 0)
                    dioPort = (Byte)args[0];
                if ((found & 0x02) != 0)
                    dioWord = (Short)args[1];
                board.dioOutputWord(dioPort, dioWord);
            }
            break;

        case CMD_DIOINBIT:
            if ((found = CmndProc.scanArgs(scan, "bb", args)) >= 0) {
                if ((found & 0x01) != 0)
                    dioPort = (Byte)args[0];
                if ((found & 0x02) != 0)
                    dioBit = (Byte)args[1];
                byte value = board.dioInputBit(dioPort, dioBit);
                out.println("Value = " + value);
            }
            break;

        case CMD_DIOOUTBIT:
            if ((found = CmndProc.scanArgs(scan, "bbb", args)) >= 0) {
                if ((found & 0x01) != 0)
                    dioPort = (Byte)args[0];
                if ((found & 0x02) != 0)
                    dioBit = (Byte)args[1];
                if ((found & 0x04) != 0)
                    dioByte = (Byte)args[2];
                board.dioOutputBit(dioPort, dioBit, dioByte);
            }
            break;

        case CMD_DIOSETBIT:
            if ((found = CmndProc.scanArgs(scan, "bb", args)) >= 0) {
                if ((found & 0x01) != 0)
                    dioPort = (Byte)args[0];
                if ((found & 0x02) != 0)
                    dioBit = (Byte)args[1];
                board.dioSetBit(dioPort, dioBit);
            }
            break;

        case CMD_DIOCLRBIT:
            if ((found = CmndProc.scanArgs(scan, "bb", args)) >= 0) {
                if ((found & 0x01) != 0)
                    dioPort = (Byte)args[0];
                if ((found & 0x02) != 0)
                    dioBit = (Byte)args[1];
                board.dioClearBit(dioPort, dioBit);
            }
            break;

        case CMD_CTRSHOW:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0) {
                out.println("Counter Parameters:");
                out.println("  Counter      = " + ctrSelect);
                out.println("  Direct value = " + ctrValue);
                out.format ("  Direct code  = %02x\n", ctrCode);
                out.println("  Rate         = " + ctrRate);
            }
            break;

        case CMD_CTRSETVALUE:
            if ((found = CmndProc.scanArgs(scan, "ibb", args)) >= 0) {
                if ((found & 0x01) != 0)
                    ctrValue = (Integer)args[0];
                if ((found & 0x02) != 0)
                    ctrSelect = (Byte)args[1];
                if ((found & 0x04) != 0)
                    ctrCode = (Byte)args[2];
                board.counterDirectSet(ctrCode, ctrValue, ctrSelect);
            }
            break;

        case CMD_CTRSETRATE:
            if ((found = CmndProc.scanArgs(scan, "fb", args)) >= 0) {
                if ((found & 0x01) != 0)
                    ctrRate = (Float)args[0];
                if ((found & 0x02) != 0) {
                    byte ctr = (Byte)args[1];
                    board.counterSetRateSingle(ctrRate, ctr);
                }
                else
                    board.counterSetRate(ctrRate);
            }
            break;

        case CMD_CTRREAD:
            if ((found = CmndProc.scanArgs(scan, "b", args)) >= 0) {
                if ((found & 0x01) != 0)
                    ctrSelect = (Byte)args[0];
                int value = board.counterRead(ctrSelect);
                out.println("Counter = " + value);
            }
            break;

        case CMD_INTSHOW:
            if ((found = CmndProc.scanArgs(scan, "b", args)) >= 0) {
                boolean times = false, counts = false, comb = false;
                if ((found & 0x01) != 0) {
                    byte option = (Byte)args[0];
                    if ((option & 0x01) != 0) times = true;
                    if ((option & 0x02) != 0) counts = true;
                    if ((option & 0x04) != 0) comb = true;
                }
                long intvl = intStop;
                float rate = 0.0F;
                int count = intCount;
                if (intvl == 0)
                    intvl = System.nanoTime();
                intvl -= intStart;
                if (intvl != 0)
                    rate = 1.0e9F * count / intvl;
                out.println("Interrupt Parameters:");
                out.format("  Type         = %s (%s)\n",
                           intType, itypTypes.decode(intType));
                out.println("  Req. Rate    = " + intRate);
                out.println("  Clock        = " + intClock);
                out.println("  Source       = " + intSrce);
                out.println("  Count        = " + intCount);
                out.println("  Actual Rate  = " + rate);
                if (times) {
                    out.print("  Interval samples (nsec):");
                    long prev = intSavst;
                    for (int j = 0; j < intSaved; j++) {
                        if ((j & 3) == 0) out.println();
                        out.format("%14s", intTimes[j] - prev);
                        prev = intTimes[j];
                    }
                    out.println();
                }
                if (counts) {
                    out.print("  Counter samples:");
                    for (int j = 0; j < intSaved; j++) {
                        if ((j & 7) == 0) out.println();
                        out.format("%7s", intCounts[j]);
                    }
                    out.println();
                }
                if (comb) {
                    out.println("  Combined samples:");
                    for (int j = 0; j < 2; j++)
                        out.format("%13s %5s %8s %8s", "Intvl(ns)", "Cntr",
                                   "Dvn(ns)", "Dly(ns)");
                    out.println();
                    for (int j = 0; j < 2; j++)
                        out.format("%13s %5s %8s %8s", "---------", "----",
                                   "-------", "-------");
                    long prev = intSavst;
                    int expTime = (int)(1000000000.0 / intRate);
                    int mult = (intClock == 0 ? 10000 : 100);
                    int cntMax = expTime / mult;
                    if (cntMax >= 0x10000) {
                        cntMax -= 0x10000;
                        expTime = cntMax * mult;
                    }
                    for (int j = 0; j < intSaved; j++) {
                        if ((j & 1) == 0) out.println();
                        int period = (int)(intTimes[j] - prev);
                        int delay = cntMax - intCounts[j];
                        if (delay < 0) delay += 0x10000;
                        out.format("%13s %5s %8s %8s", period, intCounts[j],
                                   period - expTime, mult * delay);
                        prev = intTimes[j];
                    }
                    out.println();
                }
                if (times || counts || comb) {
                    intSaved = 0;
                    intSavst = intLast;
                }
            }
            break;

        case CMD_INTSTART:
            if ((found = CmndProc.scanArgs(scan, "fbbb", args)) >= 0) {
                if ((found & 0x01) != 0)
                    intRate = (Float)args[0];
                if ((found & 0x02) != 0)
                    intClock = (Byte)args[1];
                if ((found & 0x04) != 0)
                    intSrce = (Byte)args[2];
                if ((found & 0x08) != 0)
                    intDsc = (Byte)args[3] != 0;
                byte source = (byte)(intSrce >> 1),
                     counter = (byte)(intSrce & 1);
                intSavst = intStart = intLast = System.nanoTime();
                intStop = 0;
                intCount = 0;
                intSaved = 0;
                int type = board.userInt(source, counter, intClock, intRate,
                                         this, "intProc", intDsc);
                if (type != 0) {
                    intType = type;
                    out.format("Interrupt type = %s (%s)\n",
                               intType, itypTypes.decode(intType));
                }
            }
            break;

        case CMD_INTSTOP:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0) {
                board.cancelOpType(intType);
                intType = 0;
                intStop = System.nanoTime();
            }
            break;

        case CMD_INTCANCEL:
            if ((found = CmndProc.scanArgs(scan, "S", args)) >= 0) {
                int type = itypTypes.encode((String)args[0], true);
                if (type == INT_TYPE_ALL)
                    board.cancelOp();
                else if (type >= 0)
                    board.cancelOpType(type);
            }
            break;

        case CMD_INTPAUSE:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0)
                board.pauseOp();
            break;

        case CMD_INTRESUME:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0)
                board.resumeOp();
            break;

        case CMD_DSHOW:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0) {
                out.println("Direct I/O  Parameters:");
                out.println("  Register     = " + dReg);
                out.format ("  Address      = %08x\n", dAddress);
                out.format ("  Byte value   = %02x\n", dByte);
                out.format ("  Word value   = %04x\n", dWord);
                out.format ("  Long value   = %08x\n", dLong);
                out.println("  Repeat count = " + dCount);
                out.println("  Word size    = " + dWidth);
                out.print  ("  Word array   =");
                for (short j = 0; j < dWords.length; j++)
                    out.format(" %04x", dWords[j]);
                out.println();
            }
            break;

        case CMD_INP:
            if ((found = CmndProc.scanArgs(scan, "i", args)) >= 0) {
                if ((found & 0x01) != 0)
                    dAddress = (Integer)args[0];
                byte value = Dscud.inp(dAddress);
                out.format("Value = 0x%02x\n", value);
            }
            break;

        case CMD_INPW:
            if ((found = CmndProc.scanArgs(scan, "i", args)) >= 0) {
                if ((found & 0x01) != 0)
                    dAddress = (Integer)args[0];
                short value = Dscud.inpw(dAddress);
                out.format("Value = 0x%04x\n", value);
            }
            break;

        case CMD_INPL:
            if ((found = CmndProc.scanArgs(scan, "i", args)) >= 0) {
                if ((found & 0x01) != 0)
                    dAddress = (Integer)args[0];
                int value = Dscud.inpl(dAddress);
                out.format("Value = 0x%08x\n", value);
            }
            break;

        case CMD_INPWS:
            if ((found = CmndProc.scanArgs(scan, "ii", args)) >= 0) {
                if ((found & 0x01) != 0)
                    dAddress = (Integer)args[0];
                if ((found & 0x02) != 0)
                    dCount = (Integer)args[1];
                if (dCount > Short.MAX_VALUE)
                    dCount = Short.MAX_VALUE;
                short value[] = new short[dCount];
                Dscud.inpws(dAddress, value, (short)dCount);
                out.print("Values =");
                for (int j = 0; j < dCount; j++) {
                    if (j > 0 && (j & 7) == 0)
                        out.print("\n        ");
                    out.format(" %04x", value[j]);
                }
                out.println();
            }
            break;

        case CMD_OUT:
            if ((found = CmndProc.scanArgs(scan, "ib", args)) >= 0) {
                if ((found & 0x01) != 0)
                    dAddress = (Integer)args[0];
                if ((found & 0x02) != 0)
                    dByte = (Byte)args[1];
                Dscud.outp(dAddress, dByte);
            }
            break;

        case CMD_OUTW:
            if ((found = CmndProc.scanArgs(scan, "iw", args)) >= 0) {
                if ((found & 0x01) != 0)
                    dAddress = (Integer)args[0];
                if ((found & 0x02) != 0)
                    dWord = (Short)args[1];
                Dscud.outpw(dAddress, dWord);
            }
            break;

        case CMD_OUTL:
            if ((found = CmndProc.scanArgs(scan, "ii", args)) >= 0) {
                if ((found & 0x01) != 0)
                    dAddress = (Integer)args[0];
                if ((found & 0x02) != 0)
                    dLong = (Integer)args[1];
                Dscud.outpl(dAddress, dLong);
            }
            break;

        case CMD_OUTWS:
            if ((found = CmndProc.scanArgs(scan, "iiwwwwwwww", args)) >= 0) {
                if ((found & 0x01) != 0)
                    dAddress = (Integer)args[0];
                if ((found & 0x02) != 0)
                    dCount = (Integer)args[1];
                if (dCount > dWords.length)
                    dCount = dWords.length;
                for (int j = 0; j < dCount; j++) {
                    if ((found & (1 << (j + 2))) != 0)
                        dWords[j] = (Short)args[j + 2];
                }
                Dscud.outpws(dAddress, dWords, (short)dCount);
            }
            break;

        case CMD_REGREAD:
            if ((found = CmndProc.scanArgs(scan, "w", args)) >= 0) {
                if ((found & 0x01) != 0)
                    dReg = (Short)args[0];
                byte value = board.registerRead(dReg);
                out.format("Value = 0x%02x\n", value);
            }
            break;

        case CMD_REGWRITE:
            if ((found = CmndProc.scanArgs(scan, "wb", args)) >= 0) {
                if ((found & 0x01) != 0)
                    dReg = (Short)args[0];
                if ((found & 0x02) != 0)
                    dByte = (Byte)args[1];
                board.registerWrite(dReg, dByte);
            }
            break;

        case CMD_TIMEREAD:
            if ((found = CmndProc.scanArgs(scan, "iib", args)) >= 0) {
                if ((found & 0x01) != 0)
                    dCount = (Integer)args[0];
                if ((found & 0x02) != 0)
                    dAddress = (Integer)args[1];
                if ((found & 0x04) != 0)
                    dWidth = (Byte)args[2];
                long msec = System.currentTimeMillis();
                if (dWidth == 4) {
                    for (int j = 0; j < dCount; j++)
                        Dscud.inpl(dAddress);
                }
                else if (dWidth == 2) {
                    for (int j = 0; j < dCount; j++)
                        Dscud.inpw(dAddress);
                }
                else {
                    for (int j = 0; j < dCount; j++)
                        Dscud.inp(dAddress);
                }
                msec = System.currentTimeMillis() - msec;
                out.println("Performed " + dCount + " operations in " + msec
                             + " msec");
            }
            break;

        case CMD_DUMP:
            if ((found = CmndProc.scanArgs(scan, "iii", args)) >= 0) {
                int mask;
                String frmt;
                if ((found & 0x01) != 0)
                    dmpAddr = (Integer)args[0];
                if ((found & 0x02) != 0)
                    dmpCount = (Integer)args[1];
                if ((found & 0x04) != 0)
                    dmpWidth = (Integer)args[2];
                mask = (dmpWidth == 1) ? 0x0f : (dmpWidth == 2) ? 0x07 : 0x03;
                frmt = (dmpWidth == 1) ? "2" : (dmpWidth == 2) ? "4" : "8";
                frmt = " %0" + frmt + "x";
                for (int j = 0; j < dmpCount; j++) {
                    if ((j & mask) == 0) {
                        if (j != 0) out.println();
                        out.format("%08x:", dmpAddr);
                    }
                    out.format(frmt, Dscud.peek(dmpAddr, dmpWidth));
                    dmpAddr += dmpWidth;
                }
                if (dmpCount > 0) out.println();
            }
            break;

        case CMD_SHOWTIMER:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0)
                out.println("Timer value = " + Dscud.getTime() + " ms");
            break;

        case CMD_SHOWBOARD:
            if ((found = CmndProc.scanArgs(scan, "S", args)) >= 0)
                out.println("Board type = "
                              + Dscud.getBoardMacro((String)args[0]));
            break;

        case CMD_THRSHOW:
            if ((found = CmndProc.scanArgs(scan, "", args)) >= 0) {
                Thread[] threads = new Thread[Thread.activeCount()];
                int nThread = Thread.enumerate(threads);
                out.println("Thread information:");
                for (int j = 0; j < nThread; j++)
                    out.println("  " + threads[j]);
            }
            break;

        case CMD_THRSTOP:
            if ((found = CmndProc.scanArgs(scan, "S", args)) >= 0) {
                String name = (String)args[0];
                Thread[] threads = new Thread[Thread.activeCount()];
                int nThread = Thread.enumerate(threads), j;
                for (j = 0; j < nThread; j++) {
                    if (threads[j].getName().equals(name)) {
                        threads[j].stop();
                        break;
                    }
                }
                if (j >= nThread)
                    out.println("Thread " + name + " not found");
            }
            break;

        case CMD_THRDAEMON:
            if ((found = CmndProc.scanArgs(scan, "Si", args)) >= 0) {
                int oper = -1;
                String name = (String)args[0];
                if ((found & 0x02) != 0)
                    oper = (Integer)args[1];
                Thread[] threads = new Thread[Thread.activeCount()];
                int nThread = Thread.enumerate(threads), j;
                for (j = 0; j < nThread; j++)
                    if (threads[j].getName().equals(name)) break;
                if (j >= nThread)
                    out.println("Thread " + name + " not found");
                else if (oper < 0)
                    out.format("Thread %s is%s a daemon\n", name,
                               threads[j].isDaemon() ? "" : " not");
                else
                    threads[j].setDaemon(oper > 0);
            }
            break;

        default:
            out.println("Command not fully implemented");

        }

        return true;
    }


/**
 ***************************************************************************
 **
 **  Routine to perform any necessary cleanup
 **
 ***************************************************************************
 */
    public void cleanup()
    {
        try {
            board.freeBoard();
        }
        catch (DscudException e) {
        }
    }


/**
 ***************************************************************************
 **
 **  Routine to process user interrupts
 **
 ***************************************************************************
 */
    private void intProc()
    {
        intLast = System.nanoTime();
        if (intSaved < MAX_INT_SAVED) {
            intTimes[intSaved] = intLast;
            board.registerWrite(Dscud.REG_1, (byte)0);
            board.registerWrite(Dscud.REG_15, (byte)0xc0);
            intCounts[intSaved] = (board.registerRead(Dscud.REG_12) & 0x00ff)
                                    | ((board.registerRead(Dscud.REG_13) << 8)
                                          & 0xff00);
            intSaved++;
        }
        intCount++;
    }


/**
 ***************************************************************************
 **
 **  Routine to get and display a board's registers
 **
 ***************************************************************************
 */
    private final static String[] heliosDesc = {
        "| AD7  | AD6  | AD5  | AD4  | AD3  | AD2  | AD1  | AD0  |",
        "| AD15 | AD14 | AD13 | AD12 | AD11 | AD10 | AD9  | AD8  |",
        "|  H3  |  H2  |  H1  |  H0  |  L3  |  L2  |  L1  |  L0  |",
        "|ADBUSY|SE/DIF|ADWAIT|DACBSY|  OF  |SCANEN| ADG1 | ADG0 |",
        "|CKSEL1|FRQSL1|FRQSL0|ADCLK |  -   |TINTE |DINTE |AINTE |",
        "|  -   |  -   | FT5  | FT4  | FT3  | FT2  | FT1  | FT0  |",
        "|  -   |  -   | FD5  | FD4  | FD3  | FD2  | FD1  | FD0  |",
        "|  -   | TINT | DINT | AINT |ADCH3 |ADCH2 |ADCH1 |ADCH0 |",
        "|  A7  |  A6  |  A5  |  A4  |  A3  |  A2  |  A1  |  A0  |",
        "|  B7  |  B6  |  B5  |  B4  |  B3  |  B2  |  B1  |  B0  |",
        "|  C7  |  C6  |  C5  |  C4  |  C3  |  C2  |  C1  |  C0  |",
        "|DIOCTR|DAMODE|DASIM | DIRA |DIRCH |  -   | DIRB |DIRCL |",
        "|CTRD7 |CTRD6 |CTRD5 |CTRD4 |CTRD3 |CTRD2 |CTRD1 |CTRD0 |",
        "|CTRD15|CTRD14|CTRD13|CTRD12|CTRD11|CTRD10|CTRD9 |CTRD8 |",
        "|CTRD23|CTRD22|CTRD21|CTRD20|CTRD19|CTRD18|CTRD17|CTRD16|",
        "|                  FPGA Revision Code                   |",
        "|  D7  |  D6  |  D5  |  D4  |  D3  |  D2  |  D1  |  D0  |",
        "|  A7  |  A6  |  A5  |  A4  |  A3  |  A2  |  A1  |  A0  |",
        "|  -   |TDBUSY|EEBUSY|CALMUX|  -   |  -   |  -   |  -   |",
        "|                       Not Used                        |",
        "|  -   |  -   |  -   |  -   |  -   |  -   |  -   |EXFIFO|",
        "| DAUR |DACH1 |DACH0 |  -   |ADPOL |  -   | ADSD |  -   |",
        "|  -   |  -   |  -   |DASIZE|DAPOL |  -   | DAG1 | DAG0 |",
        "|                D/A Simultaneous Update                |",
        "| FD07 | FD06 | FD05 | FD04 | FD03 | FD02 | FD01 | FD00 |",
        "| FD11 | FD10 | FD09 | FD08 |  OF  |  FF  |  HF  |  EF  |",
    };

    private final static String[] pageDesc = {
        "Counter/Timer Access",
        "AutoCal Control",
        "Expanded FIFO and AD/DA Control"
    };

    private void showRegs()
    {
        if (boardType == Dscud.DSC_HELIOS) {
            byte[] regs = new byte[24];
            for (short r = 0; r < 12; r++)
                regs[r] = board.registerRead(r);
            for (byte p = 0; p < 3; p++) {
                board.registerWrite(Dscud.REG_1, p);
                for (short r = 12; r < 16; r++)
                    regs[4 * p + r] = board.registerRead(r);
            }
            out.println("Main registers:");
            for (short r = 0; r < 12; r++) {
                if ((r == 5 || r == 6) && ((regs[20] & 1) != 0))
                    out.format("  %2s: %02x  %s\n", r, regs[r],
                               heliosDesc[r + 19]);
                else
                    out.format("  %2s: %02x  %s\n", r, regs[r], heliosDesc[r]);
            }
            for (byte p = 0; p < 3; p++) {
                out.format("\nPage %s: %s\n", p, pageDesc[p]);
                for (short r = 12; r < 16; r++) {
                    int ri = 4 * p + r;
                    out.format("  %2s: %02x  %s\n", r, regs[ri],
                               heliosDesc[ri]);
                }
            }
        }

        else
            out.println("Unknown board type (" + boardType + ")");
    }

}
