package org.lsst.ccs.drivers.dscud;

import jline.console.ConsoleReader;
import org.lsst.ccs.utilities.sa.CmndProcess;
import org.lsst.ccs.utilities.sa.ConsOut;
import org.lsst.ccs.utilities.sa.Output;

/**
 ***************************************************************************
 **
 **  Program to test the Java DSC universal driver routines
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class TestDscud implements CmndProcess.Dispatch {

   /**
    ***************************************************************************
    **
    **  Private constants
    **
    ***************************************************************************
    */
    private final static int
        INT_TYPE_ALL = 255,
        MAX_INT_SAVED = 64;

    private final static String fifoStatus[] = { "U",   "E",   "H",   "HE",
                                                 "F",  "FE",  "FH",  "FHE",
                                                 "O",  "OE",  "OH",  "OHE",
                                                "OF", "OFE", "OFH", "OFHE"};

    /*
    **  Command codes
    */
    private final static int
        CMD_INIT        = 0,
        CMD_FREE        = 1,
        CMD_SHOW        = 2,
        CMD_DIOSET      = 3,
        CMD_DIOINBYTE   = 4,
        CMD_DIOINWORD   = 5,
        CMD_DIOOUTBYTE  = 6,
        CMD_DIOOUTWORD  = 7,
        CMD_DIOINBIT    = 8,
        CMD_DIOOUTBIT   = 9,
        CMD_DIOSETBIT   = 10,
        CMD_DIOCLRBIT   = 11,
        CMD_DIOSHOW     = 12,
        CMD_ADSET       = 13,
        CMD_ADSETCHAN   = 14,
        CMD_ADSAMPLE    = 15,
        CMD_ADSAMPAVG   = 16,
        CMD_ADSAMPINT   = 17,
        CMD_ADSCAN      = 18,
        CMD_ADSCANAVG   = 19,
        CMD_ADSCANINT   = 20,
        CMD_ADAUTOCAL   = 21,
        CMD_ADCALVERIFY = 22,
        CMD_ADCTOV      = 23,
        CMD_ADVTOC      = 24,
        CMD_ADSHOW      = 25,
        CMD_ADSHOWFIFO  = 26,
        CMD_DASET       = 27,
        CMD_DASETPOL    = 28,
        CMD_DACONV      = 29,
        CMD_DASCAN      = 30,
        CMD_DAAUTOCAL   = 31,
        CMD_DACALVERIFY = 32,
        CMD_DACTOV      = 33,
        CMD_DAVTOC      = 34,
        CMD_DASHOW      = 35,
        CMD_INP         = 36,
        CMD_INPW        = 37,
        CMD_INPL        = 38,
        CMD_INPWS       = 39,
        CMD_OUT         = 40,
        CMD_OUTW        = 41,
        CMD_OUTL        = 42,
        CMD_OUTWS       = 43,
        CMD_DSHOW       = 44,
        CMD_SHOWREGS    = 45,
        CMD_SHOWERROR   = 46,
        CMD_CTRSETVALUE = 47,
        CMD_CTRSETRATE  = 48,
        CMD_CTRREAD     = 49,
        CMD_CTRSHOW     = 50,
        CMD_TIMEREAD    = 51,
        CMD_INTSTART    = 52,
        CMD_INTSTOP     = 53,
        CMD_INTCANCEL   = 54,
        CMD_INTPAUSE    = 55,
        CMD_INTRESUME   = 56,
        CMD_INTSHOW     = 57,
        CMD_DUMP        = 58,
        CMD_THRSHOW     = 59,
        CMD_THRSTOP     = 60,
        CMD_THRDAEMON   = 61,
        CMD_REGREAD     = 62,
        CMD_REGWRITE    = 63,
        CMD_SHOWBOARD   = 64,
        CMD_SHOWTIMER   = 65,
        NUM_CMDS        = 66;

    /*
    **  Command help text
    */
    private final static String[] helpInit = {
        "Initialize a board",
        "init <type> <address> <level> <clkfreq0> <clkfreq1> <clkslct1>",
        "type       The board type (default helios)",
        "address    The board address (default 0x280)",
        "level      The interrupt level (default 5)",
        "clkfreq0   The frequency of clock 0 (default 0: 10 MHz)",
        "clkfreq1   The frequency of clock 1 (default 0: 10 MHz)",
        "clkslct1   The source for clock 1 (default 0: internal)",
    };

    private final static String[] helpShow = {
        "Show the current initialization parameters",
        "show",
    };

    private final static String[] helpShowregs = {
        "Read and display the board's registers",
        "showregs",
    };

    private final static String[] helpShowerror = {
        "Show last error, or the string for a code",
        "showerror <code>",
        "code   The error code (default last occuring error)"
    };

    private final static String[] helpFree = {
        "Free the board",
        "free",
    };

    private final static String[] helpAdset = {
        "Set parameters for analog input (ADCs)",
        "adset <chan> <gain> <range> <polarity> <loadcal> <scanint> <diff>",
        "chan      The analog channel to use in sample operations",
        "gain      The input gain to use (0 - 3)",
        "range     The input range to use (0 - 1)",
        "polarity  The polarity of the input (0 - 1)",
        "loadcal   If non-zero, load calibration values",
        "scanint   The interval between successive channels in a scan (0 - 5)",
        "diff      0 = single-ended input; 1 = differential input",
    };

    private final static String[] helpAdsetchan = {
        "Set ADC channel range for sampling",
        "adsetchan <lowchan> <highchan>",
        "lowchan    The first channel to use (0 - 15)",
        "highchan   The last channel to use (0 - 15, >= lowchan)",
    };

    private final static String[] helpAdshow = {
        "Show the current analog input parameters",
        "adshow",
    };

    private final static String[] helpAdshowfifo = {
        "Show the state of the analog input FIFO",
        "adshowfifo",
    };

    private final static String[] helpAdsample = {
        "Read and display current ADC channel value",
        "adsamp",
    };

    private final static String[] helpAdsampavg = {
        "Average several reads of current ADC channel",
        "adsampavg <count>",
        "count   The number of samples to read and average",
    };

    private final static String[] helpAdsampint = {
        "Read ADC channels using interrupts",
        "adsampint <nsamp> <lowchan> <highchan> <rate> <clksrc> <depth> <thresh>",
        "nsamp      The number of samples to read",
        "lowchan    The lowest channel number to read",
        "highchan   The highest channel number to read",
        "rate       The sample rate (Hz); cannot exceed 100000",
        "clksrc     The clock frequency; 0 = 100 KHz, 1 = 10 MHz",
        "depth      The FIFO depth to use, or -1 to disable the FIFO",
        "thresh     The FIFO threshold for triggering readouts",
    };

    private final static String[] helpAdscan = {
        "Read and display a set of ADC channels",
        "adscan <lowchan> <highchan>",
        "lowchan    The first channel to scan (0 - 15)",
        "highchan   The last channel to scan (0 - 15, >= lowscan)",
    };

    private final static String[] helpAdscanavg = {
        "Average several reads of a set of ADC channels",
        "adscanavg <lowchan> <highchan> <count>",
        "lowchan    The first channel to scan (0 - 15)",
        "highchan   The last channel to scan (0 - 15, >= lowscan)",
        "count      The number of samples to read and average",
    };

    private final static String[] helpAdscanint = {
        "Scan ADC channels using interrupts",
        "adscanint <nsamp> <lowchan> <highchan> <rate> <clksrc> <depth> <thresh>",
        "nsamp      The number of samples to read",
        "lowchan    The lowest channel number to read",
        "highchan   The highest channel number to read",
        "rate       The sample rate (Hz); cannot exceed 100000",
        "clksrc     The clock frequency; 0 = 100 KHz, 1 = 10 MHz",
        "depth      The FIFO depth to use, or -1 to disable the FIFO",
        "thresh     The FIFO threshold for triggering readouts",
    };

    private final static String[] helpAdautocal = {
        "Perform auto calibration of analog input",
        "adautocal <range> <bootrange>",
        "range      The range (0 - 15, or 255) to be auto-calibrated",
        "bootrange  The calibration range (0 - 15) to load at boot time",
    };

    private final static String[] helpAdcalverify = {
        "Verify the analog input auto calibration",
        "adcalverify <range> <bootrange>",
        "range      The range (0 - 15) to be verified",
    };

    private final static String[] helpAdctov = {
        "Convert analog input code to voltage",
        "adctov <code>",
        "code   The code value (-32768 - 32767) to be converted",
    };

    private final static String[] helpAdvtoc = {
        "Convert analog input voltage to code",
        "advtoc <voltage>",
        "voltage  The voltage to be converted to a code value",
    };

    private final static String[] helpDaset = {
        "Set parameters for analog output",
        "daset <polarity> <loadcal> <range> <gain> <simupdate> <polenab> <urset> <urchan>",
        "polarity   The polarity of the output (0 - 1)",
        "loadcal    If non-zero, load calibration values first",
        "range      The range to use for conversions: 0 = 5.0, 1 = 10.0",
        "gain       The output gain to use (0 - 2)",
        "simupdate  If non-zero, output all voltages simultaneously",
        "polenab    If non-zero, use software polarity, not hardware",
        "urset      If non-zero, the gain is set on a unique channel",
        "urchan     The unique channel for the gain setting", 
    };

    private final static String[] helpDasetpol = {
        "Set the polarity for analog output",
        "dasetpol <polarity>",
        "polarity  The polarity of the analog output (0 - 1)",
    };

    private final static String[] helpDashow = {
        "Show the current analog output parameters",
        "dashow",
    };

    private final static String[] helpDaautocal = {
        "Perform auto calibration of analog output",
        "daautocal <range>",
        "range  The range (0 - 15) to be auto-calibrated",
    };

    private final static String[] helpDacalverify = {
        "Verify the analog output auto calibration",
        "dacalverify <range>",
        "range  The range (0 - 15) to be verified",
    };

    private final static String[] helpDaconv = {
        "Write a value to an analog output channel",
        "daconv <chan> <code>",
        "chan    The analog output channel to use (0 - 3)",
        "code    The value to output",
    };

    private final static String[] helpDascan = {
        "Write values to a set of analog output channels",
        "dascan",
    };

    private final static String[] helpDactov = {
        "Convert analog output code to voltage",
        "dactov <code>",
        "code   The code value (0 - 4095) to be converted",
    };

    private final static String[] helpDavtoc = {
        "Convert analog output voltage to code",
        "davtoc <voltage>",
        "voltage  The voltage to be converted to a code value",
    };

    private final static String[] helpDioset = {
        "Set digital I/O parameters",
        "dioset <id> <value>",
        "id       The ID of the entity to set",
        "value    The value to set",
    };

    private final static String[] helpDioshow = {
        "Show the current digital I/O parameters",
        "dioshow",
    };

    private final static String[] helpDioinbyte = {
        "Read and display a byte from a digital I/O port",
        "dioinbyte <port>",
        "port    The digital I/O port to read",
    };

    private final static String[] helpDioinword = {
        "Read and display a word (16-bit) from a digital I/O port",
        "dioinword <port>",
        "port    The digital I/O port to read",
    };

    private final static String[] helpDiooutbyte = {
        "Write a byte to a digital I/O port",
        "diooutbyte <port> <value>",
        "port    The digital I/O port to write to",
        "value   The value to write",
    };

    private final static String[] helpDiooutword = {
        "Write a word (16-bit) to a digital I/O port",
        "diooutword <port> <value>",
        "port    The digital I/O port to write to",
        "value   The value to write",
    };

    private final static String[] helpDioinbit = {
        "Read and display a bit from a digital I/O port",
        "dioinbit <port> <bitnum>",
        "port    The digital I/O port to read",
        "bitnum  The number of the bit to read",
    };

    private final static String[] helpDiooutbit = {
        "Write a bit to a digital I/O port",
        "diooutbit <port> <bitnum> <value>",
        "port    The digital I/O port to write to",
        "bitnum  The number of the bit to write",
        "value   The value to set the bit to",
    };

    private final static String[] helpDiosetbit = {
        "Set a bit in a digital I/O port",
        "diosetbit <port> <bitnum>",
        "port    The digital I/O port to write to",
        "bitnum  The number of the bit to set",
    };

    private final static String[] helpDioclearbit = {
        "Clear a bit in a digital I/O port",
        "dioclearbit <port> <bitnum>",
        "port    The digital I/O port to write to",
        "bitnum  The number of the bit to clear",
    };

    private final static String[] helpCtrshow = {
        "Show the current counter parameters",
        "ctrshow [<option>]",
        "option  Display option bits: 1 = times, 2 = counts",
    };

    private final static String[] helpCtrsetvalue = {
        "Set the value of a counter",
        "ctrsetvalue <value> <counter> <code>",
        "value    The value to initialize the counter to",
        "counter  The counter to be initialized",
        "code     The control code to use",
    };

    private final static String[] helpCtrsetrate = {
        "Set the frequency of a counter",
        "ctrsetrate <freq> <counter>",
        "freq     The frequency to initialize the counter to",
        "counter  The counter to be initialized (default all)",
    };

    private final static String[] helpCtrread = {
        "Read the value of a counter",
        "ctrread <counter>",
        "counter  The counter to be read",
    };

    private final static String[] helpIntshow = {
        "Show counter interrupt statistics",
        "intshow [<option>]",
        "option  Display option bits: 1 = intervals, 2 = counts, 4 = combined",
    };

    private final static String[] helpIntstart = {
        "Start counter interrupts",
        "intstart <rate> <clock> <source> [<method>]",
        "rate     The interrupt rate, or 0 if pre-programmed",
        "clock    The clock frequency to use: 0 = 100 KHz, 1 = 10 MHz",
        "source   The interrupt source: 0 = ctr0, 1 = ctr1, 2 = ext (dflt 1)",
        "method   The interrupt handling method: 0 = local, 1 = DSC (dflt 0)",
    };

    private final static String[] helpIntstop = {
        "Stop counter interrupts",
        "intstop",
    };

    private final static String[] helpIntcancel = {
        "Cancel active interrupt processing",
        "intcancel <type>",
        "type   The type (cou, dioi, ad, da or all) of interrupt to cancel",
    };

    private final static String[] helpIntpause = {
        "Pause all interrupt processing",
        "intpause",
    };

    private final static String[] helpIntresume = {
        "Resume all interrupt processing",
        "intresume",
    };

    private final static String[] helpDshow = {
        "Show the current direct I/O parameters",
        "dshow",
    };

    private final static String[] helpInp = {
        "Read and display a byte from I/O space",
        "inp <address>",
        "address  The address from which to read a byte",
    };

    private final static String[] helpInpw = {
        "Read and display a 16-bit word from I/O space",
        "inpw <address>",
        "address  The address from which to read a 16-bit word",
    };

    private final static String[] helpInpl = {
        "Read and display a 32-bit word from I/O space",
        "inpl <address>",
        "address  The address from which to read a 32-bit integer",
    };

    private final static String[] helpInpws = {
        "Read and display 16-bit words from I/O space FIFO",
        "inpws <address> <count>",
        "address  The address from which to read 16-bit words",
        "count    The number of words to read at <address>",
    };

    private final static String[] helpOut = {
        "Write a byte to I/O space",
        "out <address> <value>",
        "address  The address at which to write a byte",
        "value    The value to write",
    };

    private final static String[] helpOutw = {
        "Write a 16-bit word to I/O space",
        "outw <address> <value>",
        "address  The address at which to write a 16-bit word",
        "value    The value to write",
    };

    private final static String[] helpOutl = {
        "Write a 32-bit word to I/O space",
        "outl <address> <value>",
        "address  The address at which to write a 32-bit integer",
        "value    The value to write",
    };

    private final static String[] helpOutws = {
        "Write 16-bit words to I/O space FIFO",
        "outws <address> <value1>... <value8>",
        "address  The address at which to write 16-bit words",
        "valuen   The values to write at <address>",
    };

    private final static String[] helpRegread = {
        "Read a board register",
        "regread <register>",
        "register  The register number from which to read a byte",
    };

    private final static String[] helpRegwrite = {
        "Write to a board register",
        "regwrite <register> <value>",
        "register  The register number to which to write a byte",
        "value     The value to write",
    };

    private final static String[] helpTimeread = {
        "Time reading from an address",
        "timeread <count> <address> <width>",
        "count    The number of times to perform the read",
        "address  The address at which to read (default board address)",
        "width    The size (1, 2 or 4) of the word to read (default 1)",
    };

    private final static String[] helpDump = {
        "Dump memory contents",
        "dump <address> <nbytes>",
        "address  The address at which to start the dump",
        "nbytes   The number of bytes to dump",
    };

    private final static String[] helpShowtimer = {
        "Show the value of the interval timer (ms)",
        "showtimer",
    };

    private final static String[] helpShowboard = {
        "Show the code for a named board type",
        "showboard",
    };

    private final static String[] helpThrshow = {
        "Show thread information",
        "thrshow",
    };

    private final static String[] helpThrstop = {
        "Stop a thread",
        "thrstop <name>",
    };

    private final static String[] helpThrdaemon = {
        "Check or set a thread's daemon status",
        "thrdaemon <name> <op>",
    };

    /*
    **  Lookup tables
    */
    private final static CmndProcess.Lookup brdTypes;
    static {
        brdTypes = new CmndProcess.Lookup(1);
        brdTypes.add("Helios", Dscud.DSC_HELIOS);
    }

    private final static CmndProcess.Lookup diffTypes;
    static {
        diffTypes = new CmndProcess.Lookup(2);
        diffTypes.add("SINGLE_ENDED", Dscud.SINGLE_ENDED);
        diffTypes.add("DIFFERENTIAL", Dscud.DIFFERENTIAL);
    }

    private final static CmndProcess.Lookup gainTypes;
    static {
        gainTypes = new CmndProcess.Lookup(4);
        gainTypes.add("GAIN_1", Dscud.GAIN_1);
        gainTypes.add("GAIN_2", Dscud.GAIN_2);
        gainTypes.add("GAIN_4", Dscud.GAIN_4);
        gainTypes.add("GAIN_8", Dscud.GAIN_8);
    }

    private final static CmndProcess.Lookup sintTypes;
    static {
        sintTypes = new CmndProcess.Lookup(6);
        sintTypes.add("SCAN_INTERVAL_4", Dscud.SCAN_INTERVAL_4);
        sintTypes.add("SCAN_INTERVAL_5", Dscud.SCAN_INTERVAL_5);
        sintTypes.add("SCAN_INTERVAL_9", Dscud.SCAN_INTERVAL_9);
        sintTypes.add("SCAN_INTERVAL_10", Dscud.SCAN_INTERVAL_10);
        sintTypes.add("SCAN_INTERVAL_15", Dscud.SCAN_INTERVAL_15);
        sintTypes.add("SCAN_INTERVAL_20", Dscud.SCAN_INTERVAL_20);
    }

    private final static CmndProcess.Lookup polTypes;
    static {
        polTypes = new CmndProcess.Lookup(2);
        polTypes.add("BIPOLAR",  Dscud.BIPOLAR);
        polTypes.add("UNIPOLAR", Dscud.UNIPOLAR);
    }

    private final static CmndProcess.Lookup rngTypes;
    static {
        rngTypes = new CmndProcess.Lookup(2);
        rngTypes.add("RANGE_5",  Dscud.RANGE_5);
        rngTypes.add("RANGE_10", Dscud.RANGE_10);
    }

    private final static CmndProcess.Lookup itypTypes;
    static {
        itypTypes = new CmndProcess.Lookup(10);
        itypTypes.add("Inactive", 0);
        itypTypes.add("AD",       Dscud.INT_TYPE_AD);
        itypTypes.add("DA",       Dscud.INT_TYPE_DA);
        itypTypes.add("DIOIN",    Dscud.INT_TYPE_DIOIN);
        itypTypes.add("USER",     Dscud.INT_TYPE_USER);
        itypTypes.add("COUNTER",  Dscud.INT_TYPE_COUNTER);
        itypTypes.add("DIOREAD",  Dscud.INT_TYPE_DIOREAD);
        itypTypes.add("OPTO",     Dscud.INT_TYPE_OPTO);
        itypTypes.add("DIOWRITE", Dscud.INT_TYPE_DIOWRITE);
        itypTypes.add("ALL",      INT_TYPE_ALL);
    }

    /*
    **  Command definition table
    */
    private final static CmndProcess.Command cmnd;
    static {
        cmnd = new CmndProcess.Command(NUM_CMDS);
        cmnd.add("init",        CMD_INIT,        helpInit,        "kwbbbb",
                 brdTypes);
        cmnd.add("show",        CMD_SHOW,        helpShow,        "");
        cmnd.add("showregs",    CMD_SHOWREGS,    helpShowregs,    "");
        cmnd.add("showerror",   CMD_SHOWERROR,   helpShowerror,   "b");
        cmnd.add("free",        CMD_FREE,        helpFree,        "");
        cmnd.add("adset",       CMD_ADSET,       helpAdset,       "bbbbbbb");
        cmnd.add("adsetchan",   CMD_ADSETCHAN,   helpAdsetchan,   "bb");
        cmnd.add("adshow",      CMD_ADSHOW,      helpAdshow,      "");
        cmnd.add("adshowfifo",  CMD_ADSHOWFIFO,  helpAdshowfifo,  "");
        cmnd.add("adsamp",      CMD_ADSAMPLE,    helpAdsample,    "");
        cmnd.add("adsampavg",   CMD_ADSAMPAVG,   helpAdsampavg,   "i");
        cmnd.add("adsampint",   CMD_ADSAMPINT,   helpAdsampint,   "wbbfbwi");
        cmnd.add("adscan",      CMD_ADSCAN,      helpAdscan,      "bb");
        cmnd.add("adscanavg",   CMD_ADSCANAVG,   helpAdscanavg,   "bbi");
        cmnd.add("adscanint",   CMD_ADSCANINT,   helpAdscanint,   "wbbfbwi");
        cmnd.add("adautocal",   CMD_ADAUTOCAL,   helpAdautocal,   "Bb");
        cmnd.add("adcalverify", CMD_ADCALVERIFY, helpAdcalverify, "B");
        cmnd.add("adctov",      CMD_ADCTOV,      helpAdctov,      "W");
        cmnd.add("advtoc",      CMD_ADVTOC,      helpAdvtoc,      "D");
        cmnd.add("daset",       CMD_DASET,       helpDaset,       "bbbbbbbb");
        cmnd.add("dasetpol",    CMD_DASETPOL,    helpDasetpol,    "b");
        cmnd.add("dashow",      CMD_DASHOW,      helpDashow,      "");
        cmnd.add("daconv",      CMD_DACONV,      helpDaconv,      "bi");
        cmnd.add("dascan",      CMD_DASCAN,      helpDascan,      "bibibibi");
        cmnd.add("daautocal",   CMD_DAAUTOCAL,   helpDaautocal,   "B");
        cmnd.add("dacalverify", CMD_DACALVERIFY, helpDacalverify, "B");
        cmnd.add("dactov",      CMD_DACTOV,      helpDactov,      "I");
        cmnd.add("davtoc",      CMD_DAVTOC,      helpDavtoc,      "D");
        cmnd.add("dioset",      CMD_DIOSET,      helpDioset,      "bb");
        cmnd.add("dioshow",     CMD_DIOSHOW,     helpDioshow,     "");
        cmnd.add("dioinbyte",   CMD_DIOINBYTE,   helpDioinbyte,   "b");
        cmnd.add("dioinword",   CMD_DIOINWORD,   helpDioinword,   "b");
        cmnd.add("diooutbyte",  CMD_DIOOUTBYTE,  helpDiooutbyte,  "bb");
        cmnd.add("diooutword",  CMD_DIOOUTWORD,  helpDiooutword,  "bw");
        cmnd.add("dioinbit",    CMD_DIOINBIT,    helpDioinbit,    "bb");
        cmnd.add("diooutbit",   CMD_DIOOUTBIT,   helpDiooutbit,   "bbb");
        cmnd.add("diosetbit",   CMD_DIOSETBIT,   helpDiosetbit,   "bb");
        cmnd.add("dioclearbit", CMD_DIOCLRBIT,   helpDioclearbit, "bb");
        cmnd.add("ctrshow",     CMD_CTRSHOW,     helpCtrshow,     "");
        cmnd.add("ctrsetvalue", CMD_CTRSETVALUE, helpCtrsetvalue, "ibb");
        cmnd.add("ctrsetrate",  CMD_CTRSETRATE,  helpCtrsetrate,  "fb");
        cmnd.add("ctrread",     CMD_CTRREAD,     helpCtrread,     "b");
        cmnd.add("intshow",     CMD_INTSHOW,     helpIntshow,     "b");
        cmnd.add("intstart",    CMD_INTSTART,    helpIntstart,    "fbbb");
        cmnd.add("intstop",     CMD_INTSTOP,     helpIntstop,     "");
        cmnd.add("intcancel",   CMD_INTCANCEL,   helpIntcancel,   "K",
                 itypTypes);
        cmnd.add("intpause",    CMD_INTPAUSE,    helpIntpause,    "");
        cmnd.add("intresume",   CMD_INTRESUME,   helpIntresume,   "");
        cmnd.add("dshow",       CMD_DSHOW,       helpDshow,       "");
        cmnd.add("inp",         CMD_INP,         helpInp,         "i");
        cmnd.add("inpw",        CMD_INPW,        helpInpw,        "i");
        cmnd.add("inpl",        CMD_INPL,        helpInpl,        "i");
        cmnd.add("inpws",       CMD_INPWS,       helpInpws,       "ii");
        cmnd.add("out",         CMD_OUT,         helpOut,         "ib");
        cmnd.add("outw",        CMD_OUTW,        helpOutw,        "iw");
        cmnd.add("outl",        CMD_OUTL,        helpOutl,        "ii");
        cmnd.add("outws",       CMD_OUTWS,       helpOutws,       "iiwwwwwwww");
        cmnd.add("regread",     CMD_REGREAD,     helpRegread,     "w");
        cmnd.add("regwrite",    CMD_REGWRITE,    helpRegwrite,    "wb");
        cmnd.add("timeread",    CMD_TIMEREAD,    helpTimeread,    "iib");
        cmnd.add("dump",        CMD_DUMP,        helpDump,        "iii");
        cmnd.add("showtimer",   CMD_SHOWTIMER,   helpShowtimer,   "");
        cmnd.add("showboard",   CMD_SHOWBOARD,   helpShowboard,   "S");
        cmnd.add("thrshow",     CMD_THRSHOW,     helpThrshow,     "");
        cmnd.add("thrstop",     CMD_THRSTOP,     helpThrstop,     "S");
        cmnd.add("thrdaemon",   CMD_THRDAEMON,   helpThrdaemon,   "Si");
    }

   /**
    ***************************************************************************
    **
    **  Private fields
    **
    ***************************************************************************
    */
    private final Output out;
    private final Dscud board;
    private final CmndProcess proc;

    // Board parameters
    private short   baseAddr = 0x280;
    private byte    boardType = Dscud.DSC_HELIOS,
                    intLevel = 5,
                    clkFreq0 = 0,
                    clkFreq1 = 0,
                    clkSlct1 = 0;
    private boolean initd = false;

    // AD parameters
    private int     adAvgCount = 1,
                    adSmpCount = 1,
                    adCurr = 0,
                    adThresh = 0;
    private short   adFifoDepth = 0;
    private byte    adChan = 0,
                    adGain = Dscud.GAIN_1,
                    adRange = Dscud.RANGE_10,
                    adPolarity = Dscud.BIPOLAR,
                    adScanIntv = 0,
                    adDiff = 0,
                    adLowChan = 0,
                    adHighChan = Dscud.ADCS_MAX_CHANNELS - 1,
                    adLow = 0,
                    adHigh = 0,
                    adClksrc = 0;
    private float   adRate = 100000F;
    private boolean adLoadCal = false;
        
    // DA parameters
    private int     daCode = 0,
                    daCodes[] = {0, 0, 0, 0},
                    daChanEnab[] = {0, 0, 0, 0};
    private byte    daPolarity = 0,
                    daGain = Dscud.GAIN_1,
                    daUrChan = 0,
                    daChan = 0;
    private float   daRange = 5.0F;
    private boolean daLoadCal = false,
                    daSimUpd = false,
                    daPolEnab = false,
                    daUrSet = false;

    // Digital I/O parameters
    private short   dioWord = 0;
    private byte    dioPort = 0,
                    dioByte = 0,
                    dioBit = 0,
                    dioConfig[] = {0, 0};

    // Direct I/O parameters
    private int     dAddress = 0,
                    dLong = 0,
                    dCount = 1;
    private short   dReg = 0,
                    dWord = 0,
                    dWords[] = new short[8];
    private byte    dByte = 0,
                    dWidth = 1;

    // Counter parameters
    private int     ctrValue = 0;
    private byte    ctrSelect = 1,
                    ctrCode = 0;
    private float   ctrRate = 1.0F;

    // Interrupt parameters
    private long    intStart = 0,
                    intStop = 0,
                    intLast = 0,
                    intSavst = 0,
                    intTimes[] = new long[MAX_INT_SAVED];
    private int     intType = 0,
                    intCount = 0,
                    intSaved = 0,
                    intCounts[] = new int[MAX_INT_SAVED];
    private byte    intClock = 0,
                    intSrce = 1;
    private float   intRate = 100.0F;
    private boolean intDsc = false;

    // Dump parameters
    private int     dmpAddr = 0,
                    dmpCount = 4,
                    dmpWidth = 4;


/**
 ***************************************************************************
 **
 **  Main constructor
 **
 ***************************************************************************
 */
    TestDscud(Output out)
    {
        this.out = (out != null) ? out : new ConsOut();
        board = new Dscud();
        proc = new CmndProcess();
        proc.add(this, cmnd);
    }

    TestDscud()
    {
        this(null);
    }


/**
 ***************************************************************************
 **
 **  Main program
 **
 **  Loops reading and processing each new typed command line.
 **
 ***************************************************************************
 */
    public static void main(String[] args)
    {
        (new TestDscud()).run();

        System.exit(0);
    }


/**
 ***************************************************************************
 **
 **  Routine to run the test
 **
 ***************************************************************************
 */
    public void run()
    {
        try {
            ConsoleReader reader = new ConsoleReader();
            while (true) {
                String line = reader.readLine(">> ");
                if (line == null) break;
                try {
                    if (!proc.process(line)) break;
                }
                catch (DscudException e) {
                    out.println(e);
                }
            }
        }
        catch (Exception e) {
            out.println(e);
        }

        cleanup();
    }


/**
 ***************************************************************************
 **
 **  Dispatches a command
 **
 ***************************************************************************
 */
    @Override
    public boolean dispatch(int code, int found, Object[] args)
    {
        switch (code) {
        case CMD_INIT:
            procInit(found, args); break;
        case CMD_SHOW:
            procShow(found, args); break;
        case CMD_SHOWREGS:
            procShowregs(found, args); break;
        case CMD_SHOWERROR:
            procShowerror(found, args); break;
        case CMD_FREE:
            procFree(found, args); break;
        case CMD_ADSET:
            procAdset(found, args); break;
        case CMD_ADSETCHAN:
            procAdsetchan(found, args); break;
        case CMD_ADSHOW:
            procAdshow(found, args); break;
        case CMD_ADSHOWFIFO:
            procAdshowfifo(found, args); break;
        case CMD_ADSAMPLE:
            procAdsample(found, args); break;
        case CMD_ADSAMPAVG:
            procAdsampavg(found, args); break;
        case CMD_ADSAMPINT:
            procAdsampint(found, args); break;
        case CMD_ADSCAN:
            procAdscan(found, args); break;
        case CMD_ADSCANAVG:
            procAdscanavg(found, args); break;
        case CMD_ADSCANINT:
            procAdscanint(found, args); break;
        case CMD_ADAUTOCAL:
            procAdautocal(found, args); break;
        case CMD_ADCALVERIFY:
            procAdcalverify(found, args); break;
        case CMD_ADCTOV:
            procAdctov(found, args); break;
        case CMD_ADVTOC:
            procAdvtoc(found, args); break;
        case CMD_DASET:
            procDaset(found, args); break;
        case CMD_DASETPOL:
            procDasetpol(found, args); break;
        case CMD_DASHOW:
            procDashow(found, args); break;
        case CMD_DACONV:
            procDaconv(found, args); break;
        case CMD_DASCAN:
            procDascan(found, args); break;
        case CMD_DAAUTOCAL:
            procDaautocal(found, args); break;
        case CMD_DACALVERIFY:
            procDacalverify(found, args); break;
        case CMD_DACTOV:
            procDactov(found, args); break;
        case CMD_DAVTOC:
            procDavtoc(found, args); break;
        case CMD_DIOSET:
            procDioset(found, args); break;
        case CMD_DIOSHOW:
            procDioshow(found, args); break;
        case CMD_DIOINBYTE:
            procDioinbyte(found, args); break;
        case CMD_DIOOUTBYTE:
            procDiooutbyte(found, args); break;
        case CMD_DIOINWORD:
            procDioinword(found, args); break;
        case CMD_DIOOUTWORD:
            procDiooutword(found, args); break;
        case CMD_DIOINBIT:
            procDioinbit(found, args); break;
        case CMD_DIOOUTBIT:
            procDiooutbit(found, args); break;
        case CMD_DIOSETBIT:
            procDiosetbit(found, args); break;
        case CMD_DIOCLRBIT:
            procDioclrbit(found, args); break;
        case CMD_CTRSHOW:
            procCtrshow(found, args); break;
        case CMD_CTRSETVALUE:
            procCtrsetvalue(found, args); break;
        case CMD_CTRSETRATE:
            procCtrsetrate(found, args); break;
        case CMD_CTRREAD:
            procCtrread(found, args); break;
        case CMD_INTSHOW:
            procIntshow(found, args); break;
        case CMD_INTSTART:
            procIntstart(found, args); break;
        case CMD_INTSTOP:
            procIntstop(found, args); break;
        case CMD_INTCANCEL:
            procIntcancel(found, args); break;
        case CMD_INTPAUSE:
            procIntpause(found, args); break;
        case CMD_INTRESUME:
            procIntresume(found, args); break;
        case CMD_DSHOW:
            procDshow(found, args); break;
        case CMD_INP:
            procInp(found, args); break;
        case CMD_INPW:
            procInpw(found, args); break;
        case CMD_INPL:
            procInpl(found, args); break;
        case CMD_INPWS:
            procInpws(found, args); break;
        case CMD_OUT:
            procOut(found, args); break;
        case CMD_OUTW:
            procOutw(found, args); break;
        case CMD_OUTL:
            procOutl(found, args); break;
        case CMD_OUTWS:
            procOutws(found, args); break;
        case CMD_REGREAD:
            procRegread(found, args); break;
        case CMD_REGWRITE:
            procRegwrite(found, args); break;
        case CMD_TIMEREAD:
            procTimeread(found, args); break;
        case CMD_DUMP:
            procDump(found, args); break;
        case CMD_SHOWTIMER:
            procShowtimer(found, args); break;
        case CMD_SHOWBOARD:
            procShowboard(found, args); break;
        case CMD_THRSHOW:
            procThrshow(found, args); break;
        case CMD_THRSTOP:
            procThrstop(found, args); break;
        case CMD_THRDAEMON:
            procThrdaemon(found, args); break;
        default:
            out.println("Command not fully implemented");
        }

        return true;
    }


/**
 ***************************************************************************
 **
 **  Processes the INIT command
 **
 ***************************************************************************
 */
    private void procInit(int found, Object[] args)
    {
        if (initd) {
            initd = false;
            board.freeBoard();
        }
        if ((found & 0x01) != 0)
            boardType = (byte)(int)(Integer)args[0];
        if ((found & 0x02) != 0)
            baseAddr = (Short)args[1];
        if ((found & 0x04) != 0)
            intLevel = (Byte)args[2];
        if ((found & 0x08) != 0)
            clkFreq0 = (Byte)args[3];
        if ((found & 0x10) != 0)
            clkFreq1 = (Byte)args[4];
        if ((found & 0x20) != 0)
            clkSlct1 = (Byte)args[5];
        dAddress = baseAddr;
        board.initBoard(boardType, baseAddr, intLevel, clkFreq0,
                        clkFreq1, clkSlct1);
        initd = true;
    }


/**
 ***************************************************************************
 **
 **  Processes the SHOW command
 **
 ***************************************************************************
 */
    private void procShow(int found, Object[] args)
    {
        out.println("Initialization parameters:");
        out.println("  Board type       = "
                     + brdTypes.decode((int)boardType));
        out.format("  Base address     = %04x\n", baseAddr);
        out.println("  Interrupt level  = " + intLevel);
        out.format("  Clock0 frequency = %s (%s)\n", clkFreq0,
                   (clkFreq0 == 0) ? "10 MHz" : "1 MHz");
        out.format("  Clock1 frequency = %s (%s)\n", clkFreq1,
                   (clkFreq1 == 0) ? "10 MHz" : "100 KHz");
        out.format("  Clock1 source    = %s (%s)\n", clkSlct1,
                   (clkSlct1 == 0) ? "Internal" : "External");
    }


/**
 ***************************************************************************
 **
 **  Processes the SHOWREGS command
 **
 ***************************************************************************
 */
    private final static String[] heliosDesc = {
        "| AD7  | AD6  | AD5  | AD4  | AD3  | AD2  | AD1  | AD0  |",
        "| AD15 | AD14 | AD13 | AD12 | AD11 | AD10 | AD9  | AD8  |",
        "|  H3  |  H2  |  H1  |  H0  |  L3  |  L2  |  L1  |  L0  |",
        "|ADBUSY|SE/DIF|ADWAIT|DACBSY|  OF  |SCANEN| ADG1 | ADG0 |",
        "|CKSEL1|FRQSL1|FRQSL0|ADCLK |  -   |TINTE |DINTE |AINTE |",
        "|  -   |  -   | FT5  | FT4  | FT3  | FT2  | FT1  | FT0  |",
        "|  -   |  -   | FD5  | FD4  | FD3  | FD2  | FD1  | FD0  |",
        "|  -   | TINT | DINT | AINT |ADCH3 |ADCH2 |ADCH1 |ADCH0 |",
        "|  A7  |  A6  |  A5  |  A4  |  A3  |  A2  |  A1  |  A0  |",
        "|  B7  |  B6  |  B5  |  B4  |  B3  |  B2  |  B1  |  B0  |",
        "|  C7  |  C6  |  C5  |  C4  |  C3  |  C2  |  C1  |  C0  |",
        "|DIOCTR|DAMODE|DASIM | DIRA |DIRCH |  -   | DIRB |DIRCL |",
        "|CTRD7 |CTRD6 |CTRD5 |CTRD4 |CTRD3 |CTRD2 |CTRD1 |CTRD0 |",
        "|CTRD15|CTRD14|CTRD13|CTRD12|CTRD11|CTRD10|CTRD9 |CTRD8 |",
        "|CTRD23|CTRD22|CTRD21|CTRD20|CTRD19|CTRD18|CTRD17|CTRD16|",
        "|                  FPGA Revision Code                   |",
        "|  D7  |  D6  |  D5  |  D4  |  D3  |  D2  |  D1  |  D0  |",
        "|  A7  |  A6  |  A5  |  A4  |  A3  |  A2  |  A1  |  A0  |",
        "|  -   |TDBUSY|EEBUSY|CALMUX|  -   |  -   |  -   |  -   |",
        "|                       Not Used                        |",
        "|  -   |  -   |  -   |  -   |  -   |  -   |  -   |EXFIFO|",
        "| DAUR |DACH1 |DACH0 |  -   |ADPOL |  -   | ADSD |  -   |",
        "|  -   |  -   |  -   |DASIZE|DAPOL |  -   | DAG1 | DAG0 |",
        "|                D/A Simultaneous Update                |",
        "| FD07 | FD06 | FD05 | FD04 | FD03 | FD02 | FD01 | FD00 |",
        "| FD11 | FD10 | FD09 | FD08 |  OF  |  FF  |  HF  |  EF  |",
    };

    private final static String[] pageDesc = {
        "Counter/Timer Access",
        "AutoCal Control",
        "Expanded FIFO and AD/DA Control"
    };

    private void procShowregs(int found, Object[] args)
    {
        if (boardType == Dscud.DSC_HELIOS) {
            byte[] regs = new byte[24];
            for (short r = 0; r < 12; r++)
                regs[r] = board.registerRead(r);
            for (byte p = 0; p < 3; p++) {
                board.registerWrite(Dscud.REG_1, p);
                for (short r = 12; r < 16; r++)
                    regs[4 * p + r] = board.registerRead(r);
            }
            out.println("Main registers:");
            for (short r = 0; r < 12; r++) {
                if ((r == 5 || r == 6) && ((regs[20] & 1) != 0))
                    out.format("  %2s: %02x  %s\n", r, regs[r],
                               heliosDesc[r + 19]);
                else
                    out.format("  %2s: %02x  %s\n", r, regs[r], heliosDesc[r]);
            }
            for (byte p = 0; p < 3; p++) {
                out.format("\nPage %s: %s\n", p, pageDesc[p]);
                for (short r = 12; r < 16; r++) {
                    int ri = 4 * p + r;
                    out.format("  %2s: %02x  %s\n", r, regs[ri],
                               heliosDesc[ri]);
                }
            }
        }

        else
            out.println("Unknown board type (" + boardType + ")");
    }


/**
 ***************************************************************************
 **
 **  Processes the SHOWERROR command
 **
 ***************************************************************************
 */
    private void procShowerror(int found, Object[] args)
    {
        if ((found & 0x01) != 0) {
            byte eCode = (Byte)args[0];
            out.println(eCode + ": " + Dscud.getErrorString(eCode));
        }
        else {
            String[] text = new String[1];
            int eCode = Dscud.getLastError(text);
            out.println(eCode + ": " + text[0]);
        }
    }


/**
 ***************************************************************************
 **
 **  Processes the FREE command
 **
 ***************************************************************************
 */
    private void procFree(int found, Object[] args)
    {
        initd = false;
        board.freeBoard();
    }


/**
 ***************************************************************************
 **
 **  Processes the ADSET command
 **
 ***************************************************************************
 */
    private void procAdset(int found, Object[] args)
    {
        if ((found & 0x01) != 0)
            adChan = (Byte)args[0];
        if ((found & 0x02) != 0)
            adGain = (Byte)args[1];
        if ((found & 0x04) != 0)
            adRange = (Byte)args[2];
        if ((found & 0x08) != 0)
            adPolarity = (Byte)args[3];
        if ((found & 0x10) != 0)
            adLoadCal = (Byte)args[4] != 0;
        if ((found & 0x20) != 0)
            adScanIntv = (Byte)args[5];
        if ((found & 0x40) != 0)
            adDiff = (Byte)args[6];
        board.adSetSettings(adChan, adGain, adRange, adPolarity,
                            adLoadCal, adScanIntv, adDiff);
        adCurr = adLow = adHigh = adChan;
    }


/**
 ***************************************************************************
 **
 **  Processes the ADSETCHAN command
 **
 ***************************************************************************
 */
    private void procAdsetchan(int found, Object[] args)
    {
        short[] value = new short[Dscud.ADCS_MAX_CHANNELS];
        if ((found & 0x01) != 0)
            adLowChan = (Byte)args[0];
        if ((found & 0x02) != 0)
            adHighChan = (Byte)args[1];
        board.adSetChannel(adLowChan, adHighChan);
        adCurr = adLow = adLowChan;
        adHigh = adHighChan;
    }


/**
 ***************************************************************************
 **
 **  Processes the ADSHOW command
 **
 ***************************************************************************
 */
    private void procAdshow(int found, Object[] args)
    {
        out.println("Analog input parameters:");
        out.println("  Sample channel    = " + adChan);
        out.format("  Gain              = %s (%s)\n", adGain,
                   gainTypes.decode(adGain));
        out.format("  Range             = %s (%s)\n", adRange,
                   rngTypes.decode((int)adRange));
        out.format("  Polarity          = %s (%s)\n", adPolarity,
                   polTypes.decode((int)adPolarity));
        out.println("  Load calibration  = " + adLoadCal);
        out.format("  Scan interval     = %s (%s)\n", adScanIntv,
                   sintTypes.decode((int)adScanIntv));
        out.format("  Differential      = %s (%s)\n", adDiff,
                   diffTypes.decode(adDiff));
        out.println("  Low channel reqd  = " + adLowChan);
        out.println("  High channel reqd = " + adHighChan);
        out.println("  Averaging count   = " + adAvgCount);
        out.println("  Intr sample count = " + adSmpCount);
        out.println("  Interrupt rate    = " + adRate);
        out.println("  Intr clock source = " + adClksrc);
        out.println("  Intr FIFO depth   = " + adFifoDepth);
        out.println("  Intr threshold    = " + adThresh);
        out.println("  Current channel   = " + adCurr);
        out.println("  Low channel set   = " + adLow);
        out.println("  High channel set  = " + adHigh);
    }


/**
 ***************************************************************************
 **
 **  Processes the ADSHOWFIFO command
 **
 ***************************************************************************
 */
    private void procAdshowfifo(int found, Object[] args)
    {
        out.print("Analog input FIFO status: ");
        byte reg5 = board.registerRead(Dscud.REG_5);
        byte reg6 = board.registerRead(Dscud.REG_6);
        board.registerWrite(Dscud.REG_1, (byte)2);
        byte reg12 = board.registerRead(Dscud.REG_12);
        if ((reg12 & 1) == 0)
            out.format("mode = basic, threshold = %s, depth = %s.\n",
                       reg5, reg6);
        else {
            String fstatus = fifoStatus[reg6 & 0x0f];
            int depth = (((int)reg6 & 0xf0) << 8) | ((int)reg5 & 0xff);
            out.format("mode = expanded, status = %s, depth = %s.\n",
                       fstatus, depth);
        }
    }


/**
 ***************************************************************************
 **
 **  Processes the ADSAMP command
 **
 ***************************************************************************
 */
    private void procAdsample(int found, Object[] args)
    {
        short value = board.adSample();
        double volts = board.adCodeToVoltage(adPolarity, adGain,
                                             adRange, value);
        out.format("Channel %s: %04x ", adCurr, value);
        if (volts > -1.0 && volts < 1.0)
            out.format("(%.5f)\n", volts);
        else
            out.format("(%.6g)\n", volts);
        if (adCurr == adHigh)
            adCurr = adLow;
        else if (++adCurr >= Dscud.ADCS_MAX_CHANNELS)
            adCurr = 0;
    }


/**
 ***************************************************************************
 **
 **  Processes the ADSAMPAVG command
 **
 ***************************************************************************
 */
    private void procAdsampavg(int found, Object[] args)
    {
        if ((found & 0x01) != 0)
            adAvgCount = (Integer)args[0];
        double value = board.adSampleAvg(adAvgCount);
        double volts = board.adCodeToVoltage(adPolarity, adGain,
                                             adRange,
                                             (short)(value + 0.5));
        out.format("Channel %s: %.1f ", adCurr, value);
        if (volts > -1.0 && volts < 1.0)
            out.format("(%.5f)\n", volts);
        else
            out.format("(%.6g)\n", volts);
        int nchan = adHigh - adLow + 1;
        if (nchan <= 0)
            nchan += Dscud.ADCS_MAX_CHANNELS;
        for (int j = 0; j < adAvgCount % nchan; j++) {
            if (adCurr == adHigh)
                adCurr = adLow;
            else if (++adCurr >= Dscud.ADCS_MAX_CHANNELS)
                adCurr = 0;
        }
    }


/**
 ***************************************************************************
 **
 **  Processes the ADSAMPINT command
 **
 ***************************************************************************
 */
    private void procAdsampint(int found, Object[] args)
    {
        if ((found & 0x01) != 0)
            adSmpCount = (Short)args[0];
        if ((found & 0x02) != 0)
            adLowChan = (Byte)args[1];
        if ((found & 0x04) != 0)
            adHighChan = (Byte)args[2];
        if ((found & 0x08) != 0)
            adRate = (Float)args[3];
        if ((found & 0x10) != 0)
            adClksrc = (Byte)args[4];
        if ((found & 0x20) != 0)
            adFifoDepth = (Short)args[5];
        if ((found & 0x40) != 0)
            adThresh = (Integer)args[6];
        short[] value = new short[adSmpCount];
        board.adSampleInt(adSmpCount, adLowChan, adHighChan, adRate,
                          adClksrc, adFifoDepth, adThresh, value);
        adCurr = adLow = adLowChan;
        adHigh = adHighChan;
        for (int j = 0; j < adSmpCount; j++) {
            if (j > 0)
                out.print(((j & 3) == 0) ? "\n" : " ");
            double volts = board.adCodeToVoltage(adPolarity, adGain,
                                                 adRange, value[j]);
            out.format("%2s: %04x ", adCurr, value[j]);
            if (volts > -1.0 && volts < 1.0)
                out.format("(% -8.5f)", volts);
            else
                out.format("(% -8.6g)", volts);
            if (adCurr == adHigh)
                adCurr = adLow;
            else if (++adCurr >= Dscud.ADCS_MAX_CHANNELS)
                adCurr = 0;
        }
        out.println();
    }


/**
 ***************************************************************************
 **
 **  Processes the ADSCAN command
 **
 ***************************************************************************
 */
    private void procAdscan(int found, Object[] args)
    {
        short[] value = new short[Dscud.ADCS_MAX_CHANNELS];
        if ((found & 0x01) != 0)
            adLowChan = (Byte)args[0];
        if ((found & 0x02) != 0)
            adHighChan = (Byte)args[1];
        board.adScan(adLowChan, adHighChan, value);
        adLow = (adLowChan < adHighChan) ? adLowChan : adHighChan;
        adHigh = (adLowChan > adHighChan) ? adLowChan : adHighChan;
        adCurr = adLow;
        for (int j = 0; j < adHigh - adLow + 1; j++, adCurr++) {
            if (j > 0)
                out.print(((j & 3) == 0) ? "\n" : " ");
            double volts = board.adCodeToVoltage(adPolarity, adGain, adRange,
                                                 value[j]);
            out.format("%2s: %04x ", adCurr, value[j]);
            if (volts > -1.0 && volts < 1.0)
                out.format("(% -8.5f)", volts);
            else
                out.format("(% -8.6g)", volts);
        }
        out.println();
    }


/**
 ***************************************************************************
 **
 **  Processes the ADSCANAVG command
 **
 ***************************************************************************
 */
    private void procAdscanavg(int found, Object[] args)
    {
        double[] value = new double[Dscud.ADCS_MAX_CHANNELS];
        double volts;
        if ((found & 0x01) != 0)
            adLowChan = (Byte)args[0];
        if ((found & 0x02) != 0)
            adHighChan = (Byte)args[1];
        if ((found & 0x04) != 0)
            adAvgCount = (Integer)args[2];
        board.adScanAvg(adLowChan, adHighChan, value, adAvgCount);
        adLow = (adLowChan < adHighChan) ? adLowChan : adHighChan;
        adHigh = (adLowChan > adHighChan) ? adLowChan : adHighChan;
        adCurr = adLow;
        for (int j = 0; j < adHigh - adLow + 1; j++, adCurr++) {
            if (j > 0)
                out.print(((j % 3) == 0) ? "\n" : "    ");
            volts = board.adCodeToVoltage(adPolarity, adGain, adRange,
                                          (short)(value[j] + 0.5));
            out.format("%2s: %7.1f ", adCurr, value[j]);
            if (volts > -1.0 && volts < 1.0)
                out.format("(% -8.5f)", volts);
            else
                out.format("(% -8.6g)", volts);
        }
        out.println();
    }


/**
 ***************************************************************************
 **
 **  Processes the ADSCANINT command
 **
 ***************************************************************************
 */
    private void procAdscanint(int found, Object[] args)
    {
        if ((found & 0x01) != 0)
            adSmpCount = (Short)args[0];
        if ((found & 0x02) != 0)
            adLowChan = (Byte)args[1];
        if ((found & 0x04) != 0)
            adHighChan = (Byte)args[2];
        if ((found & 0x08) != 0)
            adRate = (Float)args[3];
        if ((found & 0x10) != 0)
            adClksrc = (Byte)args[4];
        if ((found & 0x20) != 0)
            adFifoDepth = (Short)args[5];
        if ((found & 0x40) != 0)
            adThresh = (Integer)args[6];
        short[] value = new short[adSmpCount];
        board.adScanInt(adSmpCount, adLowChan, adHighChan, adRate,
                        adClksrc, adFifoDepth, adThresh, value);
        adCurr = adLow = adLowChan;
        adHigh = adHighChan;
        for (int j = 0; j < adSmpCount; j++) {
            if (j > 0)
                out.print(((j & 3) == 0) ? "\n" : " ");
            double volts = board.adCodeToVoltage(adPolarity, adGain,
                                                 adRange, value[j]);
            out.format("%2s: %04x ", adCurr, value[j]);
            if (volts > -1.0 && volts < 1.0)
                out.format("(% -8.5f)", volts);
            else
                out.format("(% -8.6g)", volts);
            if (adCurr == adHigh)
                adCurr = adLow;
            else if (++adCurr >= Dscud.ADCS_MAX_CHANNELS)
                adCurr = 0;
        }
        out.println();
    }


/**
 ***************************************************************************
 **
 **  Processes the ADAUTOCAL command
 **
 ***************************************************************************
 */
    private void procAdautocal(int found, Object[] args)
    {
        byte range, bootRange;
        range = (Byte)args[0];
        if ((found & 0x02) != 0)
            bootRange = (Byte)args[1];
        else
            bootRange = (byte)0;
        board.adAutoCal(range, bootRange);
    }


/**
 ***************************************************************************
 **
 **  Processes the ADCALVERIFY command
 **
 ***************************************************************************
 */
    private void procAdcalverify(int found, Object[] args)
    {
        byte range;
        float[] offset = {0F}, gain = {0F};
        range = (Byte)args[0];
        board.adCalVerify(range, offset, gain);
        out.format("Offset error = %1.4f, gain error = %1.4f\n",
                   offset[0], gain[0]);
    }


/**
 ***************************************************************************
 **
 **  Processes the ADCTOV command
 **
 ***************************************************************************
 */
    private void procAdctov(int found, Object[] args)
    {
        short code = (Short)args[0];
        double volts = board.adCodeToVoltage(adPolarity, adGain,
                                             adRange, code);
        if (volts > -1.0 && volts < 1.0)
            out.format("Voltage = %.5f\n", volts);
        else
            out.format("Voltage = %.6g\n", volts);
    }


/**
 ***************************************************************************
 **
 **  Processes the ADVTOC command
 **
 ***************************************************************************
 */
    private void procAdvtoc(int found, Object[] args)
    {
        double volts = (Double)args[0];
        int code = board.voltageToAdCode(adPolarity, adGain, adRange,
                                         volts);
        out.println("Code = " + code);
    }


/**
 ***************************************************************************
 **
 **  Processes the DASET command
 **
 ***************************************************************************
 */
    private void procDaset(int found, Object[] args)
    {
        if ((found & 0x01) != 0)
            daPolarity = (Byte)args[0];
        if ((found & 0x02) != 0)
            daLoadCal = (Byte)args[1] != 0;
        if ((found & 0x04) != 0)
            daRange = (Byte)args[2] != 0 ? 10.0F : 5.0F;
        if ((found & 0x08) != 0)
            daGain = (Byte)args[3];
        if ((found & 0x10) != 0)
            daSimUpd = (Byte)args[4] != 0;
        if ((found & 0x20) != 0)
            daPolEnab = (Byte)args[5] != 0;
        if ((found & 0x40) != 0)
            daUrSet = (Byte)args[6] != 0;
        if ((found & 0x80) != 0)
            daUrChan = (Byte)args[7];
        board.daSetSettings(daPolarity, daLoadCal, daRange, daGain,
                            daSimUpd, daPolEnab, daUrSet, daUrChan);
    }


/**
 ***************************************************************************
 **
 **  Processes the DASETPOL command
 **
 ***************************************************************************
 */
    private void procDasetpol(int found, Object[] args)
    {
        if ((found & 0x01) != 0)
            daPolarity = (Byte)args[0];
        board.daSetPolarity(daPolarity);
    }


/**
 ***************************************************************************
 **
 **  Processes the DASHOW command
 **
 ***************************************************************************
 */
    private void procDashow(int found, Object[] args)
    {
        out.println("Analog output parameters:");
        out.format("  Polarity            = %s (%s)\n", daPolarity,
                   polTypes.decode(1 - daPolarity));
        out.println("  Load calibration    = " + daLoadCal);
        out.println("  Range               = " + daRange);
        out.format("  Gain                = %s (%s)\n", daGain,
                   gainTypes.decode((int)daGain));
        out.println("  Simultaneous update = " + daSimUpd);
        out.println("  Polarity enable     = " + daPolEnab);
        out.println("  Set unique range    = " + daUrSet);
        out.println("  Unique channel      = " + daUrChan);
    }


/**
 ***************************************************************************
 **
 **  Processes the DACONV command
 **
 ***************************************************************************
 */
    private void procDaconv(int found, Object[] args)
    {
        if ((found & 0x01) != 0)
            daChan = (Byte)args[0];
        if ((found & 0x02) != 0)
            daCode = (Integer)args[1];
        board.daConvert(daChan, daCode);
    }


/**
 ***************************************************************************
 **
 **  Processes the DASCAN command
 **
 ***************************************************************************
 */
    private void procDascan(int found, Object[] args)
    {
        if (found != 0) {
            for (int j = 0; j < daChanEnab.length; j++)
                daChanEnab[j] = 0;
            for (int j = 0; j < 2 * daChanEnab.length; j += 2) {
                if ((found & (1 << j)) != 0) {
                    byte chan = (Byte)args[j];
                    daChanEnab[chan] = 1;
                    if ((found & (1 << (j + 1))) != 0)
                        daCodes[chan] = (Integer)args[j + 1];
                }
            }
        }
        board.daConvertScan(daChanEnab, daCodes);
    }


/**
 ***************************************************************************
 **
 **  Processes the DAAUTOCAL command
 **
 ***************************************************************************
 */
    private void procDaautocal(int found, Object[] args)
    {
        byte range;
        range = (Byte)args[0];
        board.daAutoCal(range);
    }


/**
 ***************************************************************************
 **
 **  Processes the DACALVERIFY command
 **
 ***************************************************************************
 */
    private void procDacalverify(int found, Object[] args)
    {
        byte range;
        float[] offset = {0F}, gain = {0F};
        range = (Byte)args[0];
        board.daCalVerify(range, offset, gain);
        out.format("Offset error = %1.4f, gain error = %1.4f\n", offset[0],
                   gain[0]);
    }


/**
 ***************************************************************************
 **
 **  Processes the DACTOV command
 **
 ***************************************************************************
 */
    private void procDactov(int found, Object[] args)
    {
        int code = (Integer)args[0];
        double volts = board.daCodeToVoltage(daPolarity, daGain, daRange, code);
        out.format("Voltage = %1.5f\n", volts);
    }


/**
 ***************************************************************************
 **
 **  Processes the DAVTOC command
 **
 ***************************************************************************
 */
    private void procDavtoc(int found, Object[] args)
    {
        double volts = (Double)args[0];
        int code = board.voltageToDaCode(daPolarity, daGain, daRange, volts);
        out.println("Code = " + code);
    }


/**
 ***************************************************************************
 **
 **  Processes the DIOSET command
 **
 ***************************************************************************
 */
    private void procDioset(int found, Object[] args)
    {
        if ((found & 0x01) != 0)
            dioConfig[0] = (Byte)args[0];
        if ((found & 0x02) != 0)
            dioConfig[1] = (Byte)args[1];
        board.dioSetConfig(dioConfig);
    }


/**
 ***************************************************************************
 **
 **  Processes the DIOSHOW command
 **
 ***************************************************************************
 */
    private void procDioshow(int found, Object[] args)
    {
        out.println("Digital I/O parameters:");
        out.format("  Configuration = (%02x, %02x)\n", dioConfig[0],
                   dioConfig[1]);
        out.println("  Port          = " + dioPort);
        out.format("  Word output   = %04x\n", dioWord);
        out.format("  Byte output   = %02x\n", dioByte);
        out.println("  Bit number    = " + dioBit);
    }


/**
 ***************************************************************************
 **
 **  Processes the DIOINBYTE command
 **
 ***************************************************************************
 */
    private void procDioinbyte(int found, Object[] args)
    {
        if ((found & 0x01) != 0)
            dioPort = (Byte)args[0];
        byte value = board.dioInputByte(dioPort);
        out.format("Value = 0x%02x\n", value);
    }


/**
 ***************************************************************************
 **
 **  Processes the DIOOUTBYTE command
 **
 ***************************************************************************
 */
    private void procDiooutbyte(int found, Object[] args)
    {
        if ((found & 0x01) != 0)
            dioPort = (Byte)args[0];
        if ((found & 0x02) != 0)
            dioByte = (Byte)args[1];
        board.dioOutputByte(dioPort, dioByte);
    }


/**
 ***************************************************************************
 **
 **  Processes the DIOINWORD command
 **
 ***************************************************************************
 */
    private void procDioinword(int found, Object[] args)
    {
        if ((found & 0x01) != 0)
            dioPort = (Byte)args[0];
        short value = board.dioInputWord(dioPort);
        out.format("Value = 0x%04x\n", value);
    }


/**
 ***************************************************************************
 **
 **  Processes the DIOOUTWORD command
 **
 ***************************************************************************
 */
    private void procDiooutword(int found, Object[] args)
    {
        if ((found & 0x01) != 0)
            dioPort = (Byte)args[0];
        if ((found & 0x02) != 0)
            dioWord = (Short)args[1];
        board.dioOutputWord(dioPort, dioWord);
    }


/**
 ***************************************************************************
 **
 **  Processes the DIOINBIT command
 **
 ***************************************************************************
 */
    private void procDioinbit(int found, Object[] args)
    {
        if ((found & 0x01) != 0)
            dioPort = (Byte)args[0];
        if ((found & 0x02) != 0)
            dioBit = (Byte)args[1];
        byte value = board.dioInputBit(dioPort, dioBit);
        out.println("Value = " + value);
    }


/**
 ***************************************************************************
 **
 **  Processes the DIOOUTBIT command
 **
 ***************************************************************************
 */
    private void procDiooutbit(int found, Object[] args)
    {
        if ((found & 0x01) != 0)
            dioPort = (Byte)args[0];
        if ((found & 0x02) != 0)
            dioBit = (Byte)args[1];
        if ((found & 0x04) != 0)
            dioByte = (Byte)args[2];
        board.dioOutputBit(dioPort, dioBit, dioByte);
    }


/**
 ***************************************************************************
 **
 **  Processes the DIOSETBIT command
 **
 ***************************************************************************
 */
    private void procDiosetbit(int found, Object[] args)
    {
        if ((found & 0x01) != 0)
            dioPort = (Byte)args[0];
        if ((found & 0x02) != 0)
            dioBit = (Byte)args[1];
        board.dioSetBit(dioPort, dioBit);
    }


/**
 ***************************************************************************
 **
 **  Processes the DIOCLEARBIT command
 **
 ***************************************************************************
 */
    private void procDioclrbit(int found, Object[] args)
    {
        if ((found & 0x01) != 0)
            dioPort = (Byte)args[0];
        if ((found & 0x02) != 0)
            dioBit = (Byte)args[1];
        board.dioClearBit(dioPort, dioBit);
    }


/**
 ***************************************************************************
 **
 **  Processes the CTRSHOW command
 **
 ***************************************************************************
 */
    private void procCtrshow(int found, Object[] args)
    {
        out.println("Counter Parameters:");
        out.println("  Counter      = " + ctrSelect);
        out.println("  Direct value = " + ctrValue);
        out.format ("  Direct code  = %02x\n", ctrCode);
        out.println("  Rate         = " + ctrRate);
    }


/**
 ***************************************************************************
 **
 **  Processes the CTRSETVALUE command
 **
 ***************************************************************************
 */
    private void procCtrsetvalue(int found, Object[] args)
    {
        if ((found & 0x01) != 0)
            ctrValue = (Integer)args[0];
        if ((found & 0x02) != 0)
            ctrSelect = (Byte)args[1];
        if ((found & 0x04) != 0)
            ctrCode = (Byte)args[2];
        board.counterDirectSet(ctrCode, ctrValue, ctrSelect);
    }


/**
 ***************************************************************************
 **
 **  Processes the CTRSETRATE command
 **
 ***************************************************************************
 */
    private void procCtrsetrate(int found, Object[] args)
    {
        if ((found & 0x01) != 0)
            ctrRate = (Float)args[0];
        if ((found & 0x02) != 0) {
            byte ctr = (Byte)args[1];
            board.counterSetRateSingle(ctrRate, ctr);
        }
        else
            board.counterSetRate(ctrRate);
    }


/**
 ***************************************************************************
 **
 **  Processes the CTRREAD command
 **
 ***************************************************************************
 */
    private void procCtrread(int found, Object[] args)
    {
        if ((found & 0x01) != 0)
            ctrSelect = (Byte)args[0];
        int value = board.counterRead(ctrSelect);
        out.println("Counter = " + value);
    }


/**
 ***************************************************************************
 **
 **  Processes the INTSHOW command
 **
 ***************************************************************************
 */
    private void procIntshow(int found, Object[] args)
    {
        boolean times = false, counts = false, comb = false;
        if ((found & 0x01) != 0) {
            byte option = (Byte)args[0];
            if ((option & 0x01) != 0) times = true;
            if ((option & 0x02) != 0) counts = true;
            if ((option & 0x04) != 0) comb = true;
        }
        long intvl = intStop;
        float rate = 0.0F;
        int count = intCount;
        if (intvl == 0)
            intvl = System.nanoTime();
        intvl -= intStart;
        if (intvl != 0)
            rate = 1.0e9F * count / intvl;
        out.println("Interrupt Parameters:");
        out.format("  Type         = %s (%s)\n", intType,
                   itypTypes.decode(intType));
        out.println("  Req. Rate    = " + intRate);
        out.println("  Clock        = " + intClock);
        out.println("  Source       = " + intSrce);
        out.println("  Count        = " + intCount);
        out.println("  Actual Rate  = " + rate);
        if (times) {
            out.print("  Interval samples (nsec):");
            long prev = intSavst;
            for (int j = 0; j < intSaved; j++) {
                if ((j & 3) == 0) out.println();
                out.format("%14s", intTimes[j] - prev);
                prev = intTimes[j];
            }
            out.println();
        }
        if (counts) {
            out.print("  Counter samples:");
            for (int j = 0; j < intSaved; j++) {
                if ((j & 7) == 0) out.println();
                out.format("%7s", intCounts[j]);
            }
            out.println();
        }
        if (comb) {
            out.println("  Combined samples:");
            for (int j = 0; j < 2; j++)
                out.format("%13s %5s %8s %8s", "Intvl(ns)", "Cntr",
                           "Dvn(ns)", "Dly(ns)");
            out.println();
            for (int j = 0; j < 2; j++)
                out.format("%13s %5s %8s %8s", "---------", "----",
                           "-------", "-------");
            long prev = intSavst;
            int expTime = (int)(1000000000.0 / intRate);
            int mult = (intClock == 0 ? 10000 : 100);
            int cntMax = expTime / mult;
            if (cntMax >= 0x10000) {
                cntMax -= 0x10000;
                expTime = cntMax * mult;
            }
            for (int j = 0; j < intSaved; j++) {
                if ((j & 1) == 0) out.println();
                int period = (int)(intTimes[j] - prev);
                int delay = cntMax - intCounts[j];
                if (delay < 0) delay += 0x10000;
                out.format("%13s %5s %8s %8s", period, intCounts[j],
                           period - expTime, mult * delay);
                prev = intTimes[j];
            }
            out.println();
        }
        if (times || counts || comb) {
            intSaved = 0;
            intSavst = intLast;
        }
    }


/**
 ***************************************************************************
 **
 **  Processes the INTSTART command
 **
 ***************************************************************************
 */
    private void procIntstart(int found, Object[] args)
    {
        if ((found & 0x01) != 0)
            intRate = (Float)args[0];
        if ((found & 0x02) != 0)
            intClock = (Byte)args[1];
        if ((found & 0x04) != 0)
            intSrce = (Byte)args[2];
        if ((found & 0x08) != 0)
            intDsc = (Byte)args[3] != 0;
        byte source = (byte)(intSrce >> 1),
             counter = (byte)(intSrce & 1);
        intSavst = intStart = intLast = System.nanoTime();
        intStop = 0;
        intCount = 0;
        intSaved = 0;
        int type = board.userInt(source, counter, intClock, intRate,
                                 this, "intProc", intDsc);
        if (type != 0) {
            intType = type;
            out.format("Interrupt type = %s (%s)\n", intType,
                       itypTypes.decode(intType));
        }
    }


/**
 ***************************************************************************
 **
 **  Processes the INTSTOP command
 **
 ***************************************************************************
 */
    private void procIntstop(int found, Object[] args)
    {
        board.cancelOpType(intType);
        intType = 0;
        intStop = System.nanoTime();
    }


/**
 ***************************************************************************
 **
 **  Processes the INTCANCEL command
 **
 ***************************************************************************
 */
    private void procIntcancel(int found, Object[] args)
    {
        int type = (Integer)args[0];
        if (type == INT_TYPE_ALL)
            board.cancelOp();
        else
            board.cancelOpType(type);
    }


/**
 ***************************************************************************
 **
 **  Processes the INTPAUSE command
 **
 ***************************************************************************
 */
    private void procIntpause(int found, Object[] args)
    {
        board.pauseOp();
    }


/**
 ***************************************************************************
 **
 **  Processes the INTRESUME command
 **
 ***************************************************************************
 */
    private void procIntresume(int found, Object[] args)
    {
        board.resumeOp();
    }


/**
 ***************************************************************************
 **
 **  Processes the DSHOW command
 **
 ***************************************************************************
 */
    private void procDshow(int found, Object[] args)
    {
        out.println("Direct I/O  Parameters:");
        out.println("  Register     = " + dReg);
        out.format ("  Address      = %08x\n", dAddress);
        out.format ("  Byte value   = %02x\n", dByte);
        out.format ("  Word value   = %04x\n", dWord);
        out.format ("  Long value   = %08x\n", dLong);
        out.println("  Repeat count = " + dCount);
        out.println("  Word size    = " + dWidth);
        out.print  ("  Word array   =");
        for (short j = 0; j < dWords.length; j++)
            out.format(" %04x", dWords[j]);
        out.println();
    }


/**
 ***************************************************************************
 **
 **  Processes the INP command
 **
 ***************************************************************************
 */
    private void procInp(int found, Object[] args)
    {
        if ((found & 0x01) != 0)
            dAddress = (Integer)args[0];
        byte value = Dscud.inp(dAddress);
        out.format("Value = 0x%02x\n", value);
    }


/**
 ***************************************************************************
 **
 **  Processes the INPW command
 **
 ***************************************************************************
 */
    private void procInpw(int found, Object[] args)
    {
        if ((found & 0x01) != 0)
            dAddress = (Integer)args[0];
        short value = Dscud.inpw(dAddress);
        out.format("Value = 0x%04x\n", value);
    }


/**
 ***************************************************************************
 **
 **  Processes the INPL command
 **
 ***************************************************************************
 */
    private void procInpl(int found, Object[] args)
    {
        if ((found & 0x01) != 0)
            dAddress = (Integer)args[0];
        int value = Dscud.inpl(dAddress);
        out.format("Value = 0x%08x\n", value);
    }


/**
 ***************************************************************************
 **
 **  Processes the INPWS command
 **
 ***************************************************************************
 */
    private void procInpws(int found, Object[] args)
    {
        if ((found & 0x01) != 0)
            dAddress = (Integer)args[0];
        if ((found & 0x02) != 0)
            dCount = (Integer)args[1];
        if (dCount > Short.MAX_VALUE)
            dCount = Short.MAX_VALUE;
        short value[] = new short[dCount];
        Dscud.inpws(dAddress, value, (short)dCount);
        out.print("Values =");
        for (int j = 0; j < dCount; j++) {
            if (j > 0 && (j & 7) == 0)
                out.print("\n        ");
            out.format(" %04x", value[j]);
        }
        out.println();
    }


/**
 ***************************************************************************
 **
 **  Processes the OUT command
 **
 ***************************************************************************
 */
    private void procOut(int found, Object[] args)
    {
        if ((found & 0x01) != 0)
            dAddress = (Integer)args[0];
        if ((found & 0x02) != 0)
            dByte = (Byte)args[1];
        Dscud.outp(dAddress, dByte);
    }


/**
 ***************************************************************************
 **
 **  Processes the OUTW command
 **
 ***************************************************************************
 */
    private void procOutw(int found, Object[] args)
    {
        if ((found & 0x01) != 0)
            dAddress = (Integer)args[0];
        if ((found & 0x02) != 0)
            dWord = (Short)args[1];
        Dscud.outpw(dAddress, dWord);
    }


/**
 ***************************************************************************
 **
 **  Processes the OUTL command
 **
 ***************************************************************************
 */
    private void procOutl(int found, Object[] args)
    {
        if ((found & 0x01) != 0)
            dAddress = (Integer)args[0];
        if ((found & 0x02) != 0)
            dLong = (Integer)args[1];
        Dscud.outpl(dAddress, dLong);
    }


/**
 ***************************************************************************
 **
 **  Processes the OUTWS command
 **
 ***************************************************************************
 */
    private void procOutws(int found, Object[] args)
    {
        if ((found & 0x01) != 0)
            dAddress = (Integer)args[0];
        if ((found & 0x02) != 0)
            dCount = (Integer)args[1];
        if (dCount > dWords.length)
            dCount = dWords.length;
        for (int j = 0; j < dCount; j++) {
            if ((found & (1 << (j + 2))) != 0)
                dWords[j] = (Short)args[j + 2];
        }
        Dscud.outpws(dAddress, dWords, (short)dCount);
    }


/**
 ***************************************************************************
 **
 **  Processes the REGREAD command
 **
 ***************************************************************************
 */
    private void procRegread(int found, Object[] args)
    {
        if ((found & 0x01) != 0)
            dReg = (Short)args[0];
        byte value = board.registerRead(dReg);
        out.format("Value = 0x%02x\n", value);
    }


/**
 ***************************************************************************
 **
 **  Processes the REGWRITE command
 **
 ***************************************************************************
 */
    private void procRegwrite(int found, Object[] args)
    {
        if ((found & 0x01) != 0)
            dReg = (Short)args[0];
        if ((found & 0x02) != 0)
            dByte = (Byte)args[1];
        board.registerWrite(dReg, dByte);
    }


/**
 ***************************************************************************
 **
 **  Processes the TIMEREAD command
 **
 ***************************************************************************
 */
    private void procTimeread(int found, Object[] args)
    {
        if ((found & 0x01) != 0)
            dCount = (Integer)args[0];
        if ((found & 0x02) != 0)
            dAddress = (Integer)args[1];
        if ((found & 0x04) != 0)
            dWidth = (Byte)args[2];
        long msec = System.currentTimeMillis();
        if (dWidth == 4) {
            for (int j = 0; j < dCount; j++)
                Dscud.inpl(dAddress);
        }
        else if (dWidth == 2) {
            for (int j = 0; j < dCount; j++)
                Dscud.inpw(dAddress);
        }
        else {
            for (int j = 0; j < dCount; j++)
                Dscud.inp(dAddress);
        }
        msec = System.currentTimeMillis() - msec;
        out.println("Performed " + dCount + " operations in " + msec
                     + " msec");
    }


/**
 ***************************************************************************
 **
 **  Processes the DUMP command
 **
 ***************************************************************************
 */
    private void procDump(int found, Object[] args)
    {
        int mask;
        String frmt;
        if ((found & 0x01) != 0)
            dmpAddr = (Integer)args[0];
        if ((found & 0x02) != 0)
            dmpCount = (Integer)args[1];
        if ((found & 0x04) != 0)
            dmpWidth = (Integer)args[2];
        mask = (dmpWidth == 1) ? 0x0f : (dmpWidth == 2) ? 0x07 : 0x03;
        frmt = (dmpWidth == 1) ? "2" : (dmpWidth == 2) ? "4" : "8";
        frmt = " %0" + frmt + "x";
        for (int j = 0; j < dmpCount; j++) {
            if ((j & mask) == 0) {
                if (j != 0) out.println();
                out.format("%08x:", dmpAddr);
            }
            out.format(frmt, Dscud.peek(dmpAddr, dmpWidth));
            dmpAddr += dmpWidth;
        }
        if (dmpCount > 0) out.println();
    }


/**
 ***************************************************************************
 **
 **  Processes the SHOWTIMER command
 **
 ***************************************************************************
 */
    private void procShowtimer(int found, Object[] args)
    {
        out.println("Timer value = " + Dscud.getTime() + " ms");
    }


/**
 ***************************************************************************
 **
 **  Processes the SHOWBOARD command
 **
 ***************************************************************************
 */
    private void procShowboard(int found, Object[] args)
    {
        out.println("Board type = " + Dscud.getBoardMacro((String)args[0]));
    }


/**
 ***************************************************************************
 **
 **  Processes the THRSHOW command
 **
 ***************************************************************************
 */
    private void procThrshow(int found, Object[] args)
    {
        Thread[] threads = new Thread[Thread.activeCount()];
        int nThread = Thread.enumerate(threads);
        out.println("Thread information:");
        for (int j = 0; j < nThread; j++)
            out.println("  " + threads[j]);
    }


/**
 ***************************************************************************
 **
 **  Processes the THRSTOP command
 **
 ***************************************************************************
 */
    private void procThrstop(int found, Object[] args)
    {
        String name = (String)args[0];
        Thread[] threads = new Thread[Thread.activeCount()];
        int nThread = Thread.enumerate(threads);
        for (int j = 0; j < nThread; j++) {
            if (threads[j].getName().equals(name)) {
                threads[j].stop();
                return;
            }
        }
        out.println("Thread " + name + " not found");
    }


/**
 ***************************************************************************
 **
 **  Processes the THRDAEMON command
 **
 ***************************************************************************
 */
    private void procThrdaemon(int found, Object[] args)
    {
        int oper = -1;
        String name = (String)args[0];
        if ((found & 0x02) != 0)
            oper = (Integer)args[1];
        Thread[] threads = new Thread[Thread.activeCount()];
        int nThread = Thread.enumerate(threads), j;
        for (j = 0; j < nThread; j++)
            if (threads[j].getName().equals(name)) break;
        if (j >= nThread)
            out.println("Thread " + name + " not found");
        else if (oper < 0)
            out.format("Thread %s is%s a daemon\n", name,
                       threads[j].isDaemon() ? "" : " not");
        else
            threads[j].setDaemon(oper > 0);
    }


/**
 ***************************************************************************
 **
 **  Performs any necessary cleanup
 **
 ***************************************************************************
 */
    public void cleanup()
    {
        try {
            board.freeBoard();
        }
        catch (DscudException e) {
        }
    }


/**
 ***************************************************************************
 **
 **  Processes user interrupts
 **
 ***************************************************************************
 */
    private void intProc()
    {
        intLast = System.nanoTime();
        if (intSaved < MAX_INT_SAVED) {
            intTimes[intSaved] = intLast;
            board.registerWrite(Dscud.REG_1, (byte)0);
            board.registerWrite(Dscud.REG_15, (byte)0xc0);
            intCounts[intSaved] = (board.registerRead(Dscud.REG_12) & 0x00ff)
                                    | ((board.registerRead(Dscud.REG_13) << 8)
                                          & 0xff00);
            intSaved++;
        }
        intCount++;
    }

}
