package org.lsst.ccs.drivers.dscud;

/**
 ***************************************************************************
 **
 **  \file  Dscud.java
 **
 **  Java interface to the DSC universal driver routines
 **
 **  \author Owen Saxton
 **
 ***************************************************************************
 */
public class Dscud {

/**
 ***************************************************************************
 **
 **  Various constants
 **
 ***************************************************************************
 */
    public static final byte
        DSC_DMM16       = 0,
        DSC_RMM         = 1,
        DSC_TMM         = 2,
        DSC_OPMM        = 3,
        DSC_DMM         = 4,
        DSC_SMM         = 5,
        DSC_GMM         = 6,
        DSC_QMM         = 7,
        DSC_ZMM         = 8,
        DSC_PMM         = 9,
        DSC_OMM         = 10,
        DSC_RMM416      = 11,
        DSC_DMM32       = 12,
        DSC_DMM32AT     = 12,
        DSC_EMMDIO      = 13,
        DSC_RMM1612     = 14,
        DSC_DMMAT       = 15,
        DSC_DMM16AT     = 16,
        DSC_IR104       = 17,
        DSC_EMM8        = 18,
        DSC_PROM        = 19,
        DSC_HERCEBX     = 20,
        DSC_CPT         = 21,
        DSC_DMM48       = 22,
        DSC_DMM48AT     = 22,
        DSC_OMMDIO      = 23,
        DSC_DIO82C55    = 24,
        DSC_MRC         = 24,
        DSC_EMMOPTO     = 24,
        DSC_ATHENA      = 25,
        DSC_METIS       = 26,
        DSC_DMM32X      = 27,
        DSC_DMM32XAT    = 27,
        DSC_ELEKTRA     = 28,
        DSC_GPIO11_9513 = 29,
        DSC_GPIO11_DIO  = 30,
        DSC_GPIO21      = 31,
        DSC_PSD         = 32,
        DSC_ATHENAII    = 33,
        DSC_DMM32DX     = 34,
        DSC_DMM32DXAT   = 34,
        DSC_HELIOS      = 35,
        DSC_NEPTUNE     = 36,
        DSC_TEST        = 126,
        DSC_RAW         = 127,
        DSC_DRVR        = -1;

    public static final int
        DE_NONE                    = 0,  // No error
        DE_HW_FAILURE              = 1,  // Hardware failed
        DE_SW_FAILURE              = 2,  // Software failed
        DE_HW_NOT_SUPPORTED        = 3,  // Could not perform operation on this
                                         //  board
        DE_SW_NOT_SUPPORTED        = 4,  // Software does not yet support this
                                         //  operation
        DE_INVALID_PARM            = 5,  // A parameter to the function is
                                         //  invalid
        DE_ALTERNATE_IN_PROGRESS   = 6,  // Alternate interrupt function in
                                         //  progress
        DE_NONE_IN_PROGRESS        = 7,  // No interrupt function in progress
        DE_BUFFER_ROLLOVER         = 8,  // Buffer would roll over a segment
        DE_INVALID_ERROR_CODE      = 9,  // We used an invalid error code
                                         //  somewhere
        DE_ALREADY_PAUSED          = 10, // Can't pause an operation--it's
                                         //  already paused
        DE_OVERFLOW                = 11, // Am9513A counter function overflowed
        DE_INVALID_FUNC            = 12, // Function number not recognized by
                                         //  board type
        DE_DSCUDH_INVALID          = 13, // Header - Library version mismatch
        DE_INVALID_BOARD           = 14, // Invalid board-type constant
        DE_BOARD_LIMIT_REACHED     = 15, // Tried to initialize too many boards
                                         //  at once
        DE_BOARD_BUSY              = 16, // Attempt to acquire the board timed
                                         //  out
        DE_INVALID_WINDRVR_HANDLE  = 17, // Windriver initialization failed
        DE_INVALID_WINDRVR_VERSION = 18, // Windriver version conflict
        DE_BAD_WINDRVR_BOARD_INIT  = 19, // Windriver could not initialize the
                                         //  board
        DE_OPERATION_TIMED_OUT     = 20, // Operation timed out
        DE_INVALID_WINDRVR_KP      = 21, // Windriver kernel plugin
                                         //  initialization failed
        DE_THREAD_FAILURE          = 22; // Windriver failed to initialize
                                         //  thread

    public static final byte
        ADCS_MAX_CHANNELS = 16,
        DACS_MAX_CHANNELS = 16,
        RANGE_5   = 0,
        RANGE_10  = 1,
        BIPOLAR   = 0,
        UNIPOLAR  = 1,
        GAIN_1    = 0,
        GAIN_2    = 1,
        GAIN_4    = 2,
        GAIN_8    = 3,
        GAIN_10   = 1,
        GAIN_100  = 2,
        GAIN_1000 = 3,
        SCAN_INTERVAL_4  = 5,
        SCAN_INTERVAL_5  = 3,
        SCAN_INTERVAL_9  = 4,
        SCAN_INTERVAL_10 = 2,
        SCAN_INTERVAL_15 = 1,
        SCAN_INTERVAL_20 = 0,
        SINGLE_ENDED = 0,
        DIFFERENTIAL = 1;

    public static final byte
        COUNTER_0  = 0,
        COUNTER_1  = 1,
        FRQ_100KHZ = 0,
        FRQ_1MHZ   = 0,
        FRQ_10MHZ  = 1;

    public static final byte
        INT_TYPE_AD       = 0x01,
        INT_TYPE_DA       = 0x02,
        INT_TYPE_DIOIN    = 0x04,
        INT_TYPE_USER     = 0x08,
        INT_TYPE_COUNTER  = 0x10,
        INT_TYPE_DIOREAD  = 0x20,
        INT_TYPE_OPTO     = 0x40,
        INT_TYPE_DIOWRITE = (byte)0x80,
        USER_INT_CANCEL  = 0,
        USER_INT_AFTER   = 1,
        USER_INT_INSTEAD = 2,
        USER_INT_SOURCE_EXTERNAL = 1,
        USER_INT_SOURCE_INTERNAL = 0;

    public static final short
        REG_0  = 0,
        REG_1  = 1,
        REG_2  = 2,
        REG_3  = 3,
        REG_4  = 4,
        REG_5  = 5,
        REG_6  = 6,
        REG_7  = 7,
        REG_8  = 8,
        REG_9  = 9,
        REG_10 = 10,
        REG_11 = 11,
        REG_12 = 12,
        REG_13 = 13,
        REG_14 = 14,
        REG_15 = 15;

/**
 ***************************************************************************
 **
 **  Private data
 **
 ***************************************************************************
 */
    private short brdAddr, board = -1, ctrIntIx = -1, dioIntIx = -1;
    private byte brdType, intLevel;


/**
 ***************************************************************************
 **
 **  Static initializer
 **
 ***************************************************************************
 */
    static {
        System.loadLibrary("Dscud");
        init();
    }


/**
 ***************************************************************************
 **
 **  Main constructor
 **
 ***************************************************************************
 */
    public Dscud() {}


/**
 ***************************************************************************
 **
 **  Initialize the library (private and static)
 **
 ***************************************************************************
 */
    private native static void init();


/**
 ***************************************************************************
 **
 **  Get information about the last error (static)
 **
 ***************************************************************************
 */
    public native static int getLastError(String[] errstr);


/**
 ***************************************************************************
 **
 **  Get the string corresponding to an error code (static)
 **
 ***************************************************************************
 */
    public native static String getErrorString(int error_code);


/**
 ***************************************************************************
 **
 **  Read a byte from I/O space (static)
 **
 ***************************************************************************
 */
    public native static byte inp(int address) throws DscudException;


/**
 ***************************************************************************
 **
 **  Read a short integer from I/O space (static)
 **
 ***************************************************************************
 */
    public native static short inpw(int address) throws DscudException;


/**
 ***************************************************************************
 **
 **  Read an integer from I/O space (static)
 **
 ***************************************************************************
 */
    public native static int inpl(int address) throws DscudException;


/**
 ***************************************************************************
 **
 **  Read a short integer repeatedly from I/O space FIFO (static)
 **
 ***************************************************************************
 */
    public native static void inpws(int address, short[] buffer, short n)
                                throws DscudException;


/**
 ***************************************************************************
 **
 **  Write a byte to I/O space (static)
 **
 ***************************************************************************
 */
    public native static void outp(int address, byte value)
                                throws DscudException;


/**
 ***************************************************************************
 **
 **  Write a short integer to I/O space (static)
 **
 ***************************************************************************
 */
    public native static void outpw(int address, short value)
                                throws DscudException;


/**
 ***************************************************************************
 **
 **  Write an integer to I/O space (static)
 **
 ***************************************************************************
 */
    public native static void outpl(int address, int value)
                                throws DscudException;


/**
 ***************************************************************************
 **
 **  Write a short integer array to I/O space FIFO (static)
 **
 ***************************************************************************
 */
    public native static void outpws(int address, short[] buffer, short n)
                                throws DscudException;


/**
 ***************************************************************************
 **
 **  Pause execution for a time (static)
 **
 ***************************************************************************
 */
    public native static void sleep(int ms) throws DscudException;


/**
 ***************************************************************************
 **
 **  Get the current time (static)
 **
 ***************************************************************************
 */
    public native static int getTime() throws DscudException;


/**
 ***************************************************************************
 **
 **  Get a board's type given its name (static)
 **
 ***************************************************************************
 */
    public native static byte getBoardMacro(String boardtype)
                                throws DscudException;


/**
 ***************************************************************************
 **
 **  Set the execution priority (static)
 **
 ***************************************************************************
 */
    public native static void setSystemPriority(int priority)
                                throws DscudException;


/**
 ***************************************************************************
 **
 **  Get the value at a given mamory address (static)
 **
 ***************************************************************************
 */
    public native static int peek(int address, int width);


/**
 ***************************************************************************
 **
 **  Initialize a board
 **
 ***************************************************************************
 */
    public native void initBoard(byte type, short baseAddr, byte intLevel,
                                 byte clkFrq0, byte clkFrq1, byte clkSel1)
                         throws DscudException;


/**
 ***************************************************************************
 **
 **  Initialize a board, short version
 **
 ***************************************************************************
 */
    public void initBoard(byte type, short baseAddr, byte intLevel)
                  throws DscudException
    {
        initBoard(type, baseAddr, intLevel, (byte)0, (byte)0, (byte)0);
    }


/**
 ***************************************************************************
 **
 **  Initialize a board on the PCI bus
 **
 ***************************************************************************
 */
    public native void pciInitBoard(byte type, byte pciSlot)
                         throws DscudException;


/**
 ***************************************************************************
 **
 **  Free a board
 **
 ***************************************************************************
 */
    public native void freeBoard() throws DscudException;


/**
 ***************************************************************************
 **
 **  Set ADC parameters
 **
 ***************************************************************************
 */
    public native void adSetSettings(byte channel, byte gain, byte range,
                                     byte polarity, boolean loadCal,
                                     byte scanIntvl, byte diff)
                         throws DscudException;


/**
 ***************************************************************************
 **
 **  Set ADC sample channel range
 **
 ***************************************************************************
 */
    public native void adSetChannel(byte lowChan, byte highChan)
                         throws DscudException;


/**
 ***************************************************************************
 **
 **  Read the next ADC sample
 **
 ***************************************************************************
 */
    public native short adSample() throws DscudException;


/**
 ***************************************************************************
 **
 **  Read and average values from successive ADC samples
 **
 ***************************************************************************
 */
    public native double adSampleAvg(int count) throws DscudException;


/**
 ***************************************************************************
 **
 **  Read ADC values in sample mode using interrupts
 **
 ***************************************************************************
 */
    public native void adSampleInt(int nConv, byte lowChan, byte highChan,
                                   float rate, byte clkSrc, short fifoDepth,
                                   int thresh, short[] values)
                         throws DscudException;


/**
 ***************************************************************************
 **
 **  Read a range of ADC values in scan mode
 **
 ***************************************************************************
 */
    public native void adScan(byte lowChan, byte highChan, short[] values)
                         throws DscudException;


/**
 ***************************************************************************
 **
 **  Read and average a range of ADC values in scan mode
 **
 ***************************************************************************
 */
    public native void adScanAvg(byte lowChan, byte highChan, double[] averages,
                                 int count) throws DscudException;


/**
 ***************************************************************************
 **
 **  Read ADC values in scan mode using interrupts
 **
 ***************************************************************************
 */
    public native void adScanInt(int nConv, byte lowChan, byte highChan,
                                 float rate, byte clkSrc, short fifoDepth,
                                 int thresh, short[] values)
                         throws DscudException;


/**
 ***************************************************************************
 **
 **  Calibarate one ADC range or all of them
 **
 ***************************************************************************
 */
    public native void adAutoCal(byte range, byte bootRange)
                         throws DscudException;


/**
 ***************************************************************************
 **
 **  Verify an ADC calibration
 **
 ***************************************************************************
 */
    public native void adCalVerify(byte range, float[] offsetError,
                                   float[] gainError) throws DscudException;


/**
 ***************************************************************************
 **
 **  Convert an ADC value to a voltage
 **
 ***************************************************************************
 */
    public native double adCodeToVoltage(byte polarity, byte gain, byte range,
                                         short code) throws DscudException;


/**
 ***************************************************************************
 **
 **  Convert a voltage to an ADC value
 **
 ***************************************************************************
 */
    public native short voltageToAdCode(byte polarity, byte gain, byte range,
                                        double voltage) throws DscudException;


/**
 ***************************************************************************
 **
 **  Set the DAC polarity
 **
 ***************************************************************************
 */
    public native void daSetPolarity(byte polarity) throws DscudException;


/**
 ***************************************************************************
 **
 **  Set the DAC parameters
 **
 ***************************************************************************
 */
    public native void daSetSettings(byte polarity, boolean loadCal,
                                     float range, byte gain, boolean simUpdate,
                                     boolean polEnab, boolean urSet,
                                     byte urChan) throws DscudException;


/**
 ***************************************************************************
 **
 **  Write a value to a DAC
 **
 ***************************************************************************
 */
    public native void daConvert(byte channel, int code) throws DscudException;


/**
 ***************************************************************************
 **
 **  Write an array of values to a set of DACs
 **
 ***************************************************************************
 */
    public native void daConvertScan(int[] chanEnab, int[] codes)
                         throws DscudException;


/**
 ***************************************************************************
 **
 **  Calibarate one DAC range
 **
 ***************************************************************************
 */
    public native void daAutoCal(byte range) throws DscudException;


/**
 ***************************************************************************
 **
 **  Verify a DAC calibration
 **
 ***************************************************************************
 */
    public native void daCalVerify(byte range, float[] offsetError,
                                   float[] gainError) throws DscudException;


/**
 ***************************************************************************
 **
 **  Convert a DAC value to a voltage
 **
 ***************************************************************************
 */
    public native double daCodeToVoltage(byte polarity, byte gain, float range,
                                         int code) throws DscudException;


/**
 ***************************************************************************
 **
 **  Convert a voltage to a DAC value
 **
 ***************************************************************************
 */
    public native int voltageToDaCode(byte polarity, byte gain, float range,
                                      double voltage) throws DscudException;


/**
 ***************************************************************************
 **
 **  Configure the digital I/O ports
 **
 ***************************************************************************
 */
    public native void dioSetConfig(byte[] config_bytes) throws DscudException;


/**
 ***************************************************************************
 **
 **  Read a byte from a digital I/O port
 **
 ***************************************************************************
 */
    public native byte dioInputByte(byte port) throws DscudException;


/**
 ***************************************************************************
 **
 **  Read a short integer from a digital I/O port
 **
 ***************************************************************************
 */
    public native short dioInputWord(byte port) throws DscudException;


/**
 ***************************************************************************
 **
 **  Write a byte to a digital I/O port
 **
 ***************************************************************************
 */
    public native void dioOutputByte(byte port, byte digital_value)
                         throws DscudException;


/**
 ***************************************************************************
 **
 **  Write a short integer to a digital I/O port
 **
 ***************************************************************************
 */
    public native void dioOutputWord(byte port, short digital_value)
                         throws DscudException;


/**
 ***************************************************************************
 **
 **  Read a bit from a digital I/O port
 **
 ***************************************************************************
 */
    public native byte dioInputBit(byte port, byte bit) throws DscudException;


/**
 ***************************************************************************
 **
 **  Write a bit to a digital I/O port
 **
 ***************************************************************************
 */
    public native void dioOutputBit(byte port, byte bit, byte digital_value)
                         throws DscudException;


/**
 ***************************************************************************
 **
 **  Set a bit in a digital I/O port
 **
 ***************************************************************************
 */
    public native void dioSetBit(byte port, byte bit) throws DscudException;


/**
 ***************************************************************************
 **
 **  Clear a bit in a digital I/O port
 **
 ***************************************************************************
 */
    public native void dioClearBit(byte port, byte bit) throws DscudException;


/**
 ***************************************************************************
 **
 **  Set the rate for counter
 **
 **  This routine uses the current frequency set for the counter to
 **  determine the value to be loaded into the counter.  No check is made
 **  on this value: only the appropriate number of low-order bytes is used.
 **
 ***************************************************************************
 */
    public native void counterSetRateSingle(float hertz, byte ctr)
                         throws DscudException;


/**
 ***************************************************************************
 **
 **  Set the rate for all counters
 **
 ***************************************************************************
 */
    public native void counterSetRate(float hertz) throws DscudException;


/**
 ***************************************************************************
 **
 **  Perform an operation on a counter directly
 **
 ***************************************************************************
 */
    public native void counterDirectSet(byte code, int value, byte ctr)
                         throws DscudException;


/**
 ***************************************************************************
 **
 **  Read the value of a counter
 **
 ***************************************************************************
 */
    public native int counterRead(byte ctr) throws DscudException;


/**
 ***************************************************************************
 **
 **  Read the value of a board register
 **
 ***************************************************************************
 */
    public native byte registerRead(short address) throws DscudException;


/**
 ***************************************************************************
 **
 **  Write a value to a board register
 **
 ***************************************************************************
 */
    public native void registerWrite(short address, byte data)
                         throws DscudException;


/**
 ***************************************************************************
 **
 **  Cancel all interrupt activity
 **
 ***************************************************************************
 */
    public native void cancelOp() throws DscudException;


/**
 ***************************************************************************
 **
 **  Cancel one type of interrupt activity
 **
 ***************************************************************************
 */
    public native void cancelOpType(int int_type) throws DscudException;


/**
 ***************************************************************************
 **
 **  Pause all interrupt activity
 **
 ***************************************************************************
 */
    public native void pauseOp() throws DscudException;


/**
 ***************************************************************************
 **
 **  Resume all interrupt activity
 **
 ***************************************************************************
 */
    public native void resumeOp() throws DscudException;


/**
 ***************************************************************************
 **
 **  Start user interrupt processing on counter or digital I/O interrupt
 **
 ***************************************************************************
 */
    public native int userInt(byte intSrc, byte counter, byte clkSrc,
                              float rate, Object cbObj, String cbMeth,
                              boolean useDsc)
                        throws DscudException;


/**
 ***************************************************************************
 **
 **  Start user interrupt processing on counter or digital interrupt
 **  (default version)
 **
 ***************************************************************************
 */
    public int userInt(byte intSrc, byte counter, byte clkSrc,
                       float rate, Object cbObj, String cbMeth)
                 throws DscudException
    {
        return userInt(intSrc, counter, clkSrc, rate, cbObj, cbMeth, false);
    }

}


/**
 ***************************************************************************
 **
 **  Unimplemented routines
 **
 ***************************************************************************
 */
//  private native static void free();

//  public native void initBoard(byte type, short baseAddr, byte intLevel,
//                               boolean extTrig, boolean extTrigC3,
//                               short ioAddr[8], short interrupt[8],
//                               byte clkFreq0, byte clkFreq1, byte clkSlct,
//                               short addrSpace, byte dacConfig);

//  public native int daConvertScanInt(DSCAIOINT *dscaioint);

//  public native int  dioInputInt(DSCDIOINT *dscdioint);
//  public native int  dioOutputInt(DSCDIOINT *dscdioint);

//  public native int  dmm32DIOOutputByte(byte port, byte digital_value);
//  public native int  dmm32DIOInputByte(byte port, byte[] digital_value);
//  public native int  optoInputByte(byte port, byte[] optoValue);
//  public native int  optoInputBit(byte port, byte bit, byte[] optoValue);
//  public native int  optoGetPolarity(byte[] polarity);
//  public native int  optoGetState(DSCOPTOSTATE *state);
//  public native int  optoSetState(DSCOPTOSTATE *state);

//  public native int  setCalMux(BOOL on);
//  public native int  getReferenceVoltages(double[] refs);
//  public native int  setReferenceVoltages(double[] refs);
//  public native int  daGetOffsets(double[] offsets, int count);
//  public native int  daSetOffsets(double[] offsets, int count);
//  public native int  setRelay(byte relay, byte value);
//  public native int  getRelay(byte relay, byte[] value);
//  public native int  setRelayMulti(byte relayGroup, byte value);
//  public native int  getRelayMulti(byte relayGroup, byte[] value);
//  public native int  watchdogEnable(short timer16, byte timer8, int options);
//  public native int  watchdogDisable();
//  public native int  watchdogTrigger();
//  public native int  pwmLoad(byte pwm_circuit, byte counter, int value);
//  public native int  pwmConfig(byte[] config_byte);
//  public native int  pwmClear(byte pwm_circuit);
//  public native int  pwmFunction(DSCPWM *dscpwm);

//  public native byte getEEPROM(int address);
//  public native void setEEPROM(int address, byte data);
//  public native void setTrimDac(int trimDac, byte value);
//  public native int  getStatus(DSCS *status);

//  public native int  setUserInterruptFunction(DSCUSERINTFUNCTION *dscuserintfunc);
//  public native int  setUserInterruptFunctionType(DSCUSERINTFUNCTION *dscuserintfunc, int int_type);

//  public native int  userIntVB(DSCUSERINT *dscuserint);
//  public native int  clearUserInterruptFunction();
//  public native int  clearUserInterruptFunctionType(int int_type);

//  public native int  getFPGARev(short[] fpga);
//  public native int  ir104ClearRelay(byte relay);
//  public native int  ir104SetRelay(byte relay);
//  public native int  ir104RelayInput(byte relay, byte[] value);
//  public native int  ir104OptoInput(byte opto, byte[] value);
//  public native int  emmDIOGetState(DSCEMMDIO *state);
//  public native int  emmDIOSetState(DSCEMMDIO *state);
//  public native int  emmDIOResetInt(DSCEMMDIORESETINT *edge);
//  public native int  enhancedFeaturesEnable(BOOL enable);
//  public native int  picOutp(int address, byte value);
//  public native int  picInp(int address, byte[] value);
//  public native int  picProgEnable(BOOL enable);
//  public native int  picProgSet(BOOL dout, BOOL dstate, BOOL cstate);
//  public native int  picProgGet(BOOL[] din);
//  public native int  aacCommand(int cmd);
//  public native int  aacGetStatus(DSCAACSTATUS *status);
//  public native int  wgCommand(int cmd);
//  public native int  wgConfigSet(DSCWGCONFIG *config);
//  public native int  wgBufferSet(int address, DSCDACODE value, int channel, BOOL simul);
