package org.lsst.ccs.drivers.dmmio;

import org.apache.log4j.Logger;
import org.lsst.ccs.drivers.commons.IOBoardDriver;

/** Driver for Diamond board */

public class DMMIOBoardDriver extends IOBoardDriver {

	static Logger log = Logger.getLogger(DMMIOBoardDriver.class);

	public DMMIOBoardDriver() {
	}

	public DMMIOBoardDriver(int baseAddress) {
		setBaseAddress(baseAddress);
	}

	public void init() {
		if (inited)
			return;
		inited = true;
		if (baseAddress < 0) {
			baseAddress = 0x300;
		}
		try {
			byte status = PCIIOHandler.readByte(baseAddress + 8);
			differential = (status & 0x20) == 0;
			biPolar = (status & 0x40) == 0;
			setDigitalOutChannelsByte((byte) 0); // clear and sync
		} catch (Throwable e) {
			log.error("***** Error loading driver, running dummy driver");
			baseAddress = -1;
		}
	}

	int baseAddress = -1;

	boolean inited = false;

	boolean differential = false;

	boolean biPolar = true;

	double gain = 1; // standard values : .5, 1, 2, 5 10

	// unipolar full scale gain 1 = 0-10 V
	// bipolar full scale gain 1 = +- 5 V
	// factory setting bipolar gain .5, +- 10V

	double fullScaleOut = 5.; // full scale volts

	byte digitalOut = 0;

	public void setBaseAddress(int baseAddress) {
		this.baseAddress = baseAddress;
		inited = false;
		init();
	}

	public int getBaseAddress() {
		return baseAddress;
	}

	public boolean isBiPolar() {
		return biPolar;
	}

	public boolean isDifferential() {
		return differential;
	}

	public void setGain(double gain) {
		this.gain = gain;
	}

	public double getGain() {
		return gain;
	}

	public void setFullScaleOut(double fullScaleOut) {
		this.fullScaleOut = fullScaleOut;
	}

	public double getFullScaleOut() {
		return fullScaleOut;
	}

	public final void setAnalogChannelRange(byte low, byte high) {
		byte value = (byte) ((high << 4) | low);
		if (baseAddress > 0) {
			PCIIOHandler.writeByte(baseAddress + 2, value);
		}
	}

	protected final void triggerADC() {
		if (baseAddress > 0) {
			PCIIOHandler.writeByte(baseAddress + 0, (byte) 0);
		}
	}

	// we could do some sleep, but usually loops one or two times only, sleep is
	// a few ms
	protected final void waitWhileBusy() {
		if (baseAddress > 0) {
			while ((PCIIOHandler.readByte(baseAddress + 8) & 0x80) != 0) {
				;
			}
		}

	}

	protected final synchronized short readData() {
		if (baseAddress < 0)
			return 0;
		byte b1 = PCIIOHandler.readByte(baseAddress + 0);
		byte b2 = PCIIOHandler.readByte(baseAddress + 1);
		return (short) (((b1 >>> 4) & 0x0F) | (b2 << 4));
	}

	protected double convertToVolts(short value) {
		if (biPolar)
			return 5. * (value / 2048. - 1.) / gain;
		else
			return 10. * value / 4096. / gain;
	}

	protected short convertFromVolts(double value) {
		return (short) (value / fullScaleOut * 4096);
	}

	@Override
	public int getAnalogInChannels() {
		return differential ? 8 : 16;
	}

	@Override
	public int getAnalogOutChannels() {
		return 2;
	}

	@Override
	public int getDigitalInChannels() {
		return 8;
	}

	@Override
	public int getDigitalOutChannels() {
		return 8;
	}

	@Override
	public synchronized double getAnalogInChannel(int n) {
		setAnalogChannelRange((byte) n, (byte) n);
		triggerADC();
		waitWhileBusy();
		return convertToVolts(readData());
	}

	public synchronized double getNextAnalogInChannel() {
		triggerADC();
		waitWhileBusy();
		return convertToVolts(readData());
	}

	@Override
	public synchronized void setAnalogOutChannel(int n, double value) {
		if (baseAddress < 0)
			return;
		if (n < 0 || n > 1)
			throw new IllegalArgumentException("Analog out channel");
		short adu = convertFromVolts(value);
		byte lsb = (byte) (adu & 0xFF);
		byte msb = (byte) (adu / 256);
		PCIIOHandler.writeByte(baseAddress + 4 + 2 * n, lsb);
		PCIIOHandler.writeByte(baseAddress + 5 + 2 * n, msb);
	}

	public byte getDigitalInChannelsByte() {
		if (baseAddress < 0)
			return 0;
		return PCIIOHandler.readByte(baseAddress + 3);
	}

	public void setDigitalOutChannelsByte(byte b) {
		if (baseAddress < 0)
			return;
		PCIIOHandler.writeByte(baseAddress + 3, b);
		digitalOut = b;
	}

	@Override
	public int getDigitalInChannel(int n) {
		return ((getDigitalInChannelsByte() >> n) & 1);
	}

	@Override
	public void setDigitalOutChannel(int n, int value) {
		byte mask = (byte) (1 << n);
		digitalOut &= ~mask;
		if (value != 0) {
			digitalOut |= mask;
		}
		setDigitalOutChannelsByte(digitalOut);
	}

	public static void main(String[] args) {
		DMMIOBoardDriver b = new DMMIOBoardDriver();
		b.init();
		byte status = PCIIOHandler.readByte(b.baseAddress + 8);
		System.out.print("status " + status + " : ");
		System.out.print(" STS " + (status >> 7));
		System.out.print(" U/B " + ((status & 0x40) >> 6));
		System.out.print(" S/D " + ((status & 0x20) >> 5));
		System.out.print(" INT " + ((status & 0x10) >> 6));
		System.out.println(" ADC " + (status & 0x0F));
		System.out.println("differential : " + b.isDifferential());
		System.out.println("bipolar : " + b.isBiPolar());
		byte value = Byte.parseByte(args[0]);
		b.setDigitalOutChannelsByte(value);
		b.setGain(2.);
		value = b.getDigitalInChannelsByte();
		System.out.println("in byte " + value);
		int nch = b.getAnalogInChannels();
		b.setAnalogOutChannel(0, nch - 1);
		status = PCIIOHandler.readByte(b.baseAddress + 8);
		int nmes = 100;
		double sum = 0;
		for (int i = 0; i < nmes; i++) {
			double mes = b.getAnalogInChannel(0);
			System.out.println("analog in " + i + " " + mes);
			sum += mes;
		}
		System.out.println("average " + (sum / nmes));
		status = PCIIOHandler.readByte(b.baseAddress + 8);
	}

}
