package org.lsst.ccs.subsystem.demo.jni;

/**
 * Demo of a jni interface to an asynchronous native service, that is able to
 * preserve some user data between the call and the call back. This data will be
 * used to select also the listener that will be called.
 * 
 * @author aubourg
 *
 */
public class DemoCallbackWithUserData {

    public DemoCallbackWithUserData() {
    }

    static {
        System.out.println("**** Loading the library ");
        try {
            System.loadLibrary("jnidemo");
        } catch (Exception e) {
            e.printStackTrace();
        }
        System.out.println("**** Done Loading the library ");
    }

    /**
     * Native call to the sample API.
     * 
     * @param minDelay min sleep time in ms
     * @param maxDelay max sleep time in ms
     * @param userData user data transmitted to the listener
     * @param returnData value returned by external API (which happens to be twice the
     *                   sleep time. Content will be filled at the time listener is called
     * @param listener the java listener that will be called upon completion 
     */
    public native void callNative(long minDelay, long maxDelay, Object userData, ReturnData returnData, CallbackListener listener);

    /**
     * Callback listener called after completion of the native API function.
     * The return data will have been filled.
     *
     */
    public interface CallbackListener {
        public void callback(Object userData);
    }

    static public class ReturnData {
        public long x;
    }


    /**
     * Synchronous adapter to the external API
     * 
     * @param minDelay
     *            min sleep delay
     * @param maxDelay
     *            max sleep delay
     * @return the data returned by the API, the function will wait until the
     *         API signals the operation is complete and will recover and return
     *         the result.
     */
    public ReturnData callSync(long minDelay, long maxDelay) {
        ReturnData r = new ReturnData();
        CallbackListener l = new CallbackListener() {
            @Override
            public void callback(Object userData) {
                synchronized(r) {
                    r.notify();
                }
            }       
        };
        synchronized (r) {
            callNative(minDelay, maxDelay, "sync adapter", r, l);
            try {
                r.wait();
            } catch (InterruptedException e) {
            }
        }
        return r;
    }

    public static void main(String[] args) {
        DemoCallbackWithUserData d = new DemoCallbackWithUserData();
        ReturnData r = new ReturnData();
        CallbackListener l = new CallbackListener() {

            @Override
            public void callback(Object userData) {
                System.out.println(">> callback " + userData + " : " + r.x);
            }

        };

        System.out.println("calling native");
        d.callNative(10, 20, "call1", r, l);
        System.out.println("called native");
        d.callNative(0, 5, "call2", r, l);
        System.out.println("called native 2");
        
        ReturnData r2 = d.callSync(30, 40);
        System.out.println("returned from sync "+r2.x);

        // without the sync call, 
        // need to sleep, because dummy external API does not know about java
        // and does
        // not attach its thread to the JVM
        try {
            Thread.sleep(1000);
        } catch (InterruptedException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }

    }

}
