package org.lsst.ccs.subsystem.demo.jni;

/**
 * Demo of a jni interface to an asynchronous native service, that is NOT able
 * to preserve some user data between the call and the call back.
 * 
 * @author aubourg
 *
 */
public class DemoCallbackNoUserData {

    public DemoCallbackNoUserData() {
    }

    static {
        System.out.println("**** Loading the library ");
        try {
            System.loadLibrary("jnidemo");
        } catch (Exception e) {
            e.printStackTrace();
        }
        System.out.println("**** Done Loading the library ");
    }

    /**
     * Native call to the sample API.
     * 
     * the synchronization is needed because of the global data structure in C
     * code
     * 
     * 
     * @param minDelay
     *            min sleep time in ms
     * @param maxDelay
     *            max sleep time in ms
     * @param returnData
     *            value returned by external API (which happens to be twice the
     *            sleep time. Content will be filled at the time listener is
     *            called
     * @param listener
     *            the java listener that will be called upon completion
     * 
     */
    public static synchronized native void callNative(long minDelay, long maxDelay, ReturnData returnData, CallbackListener listener);

    /**
     * to be used in case of timeout. Native call is not reentrant.
     * 
     */
    public static synchronized native void clearCallBack();

    /**
     * Callback listener called after completion of the native API function. The
     * return data will have been filled.
     *
     */
    public interface CallbackListener {
        public void callback();
    }

    static public class ReturnData {
        public long x;
    }

    /**
     * Synchronous adapter to the external API
     * 
     * @param minDelay
     *            min sleep delay
     * @param maxDelay
     *            max sleep delay
     * @return the data returned by the API, the function will wait until the
     *         API signals the operation is complete and will recover and return
     *         the result.
     */

    public static ReturnData callSync(long minDelay, long maxDelay) {
        ReturnData r = new ReturnData();
        CallbackListener l = new CallbackListener() {
            @Override
            public void callback() {
                synchronized (r) {
                    r.notify();
                }
            }
        };
        synchronized (r) {
            callNative(minDelay, maxDelay, r, l);
            try {
                r.wait();
                // in case of implementation with a timeout,
                // clearCallBack should be called.
            } catch (InterruptedException e) {
                clearCallBack();
            }
        }
        return r;
    }

    public static void main(String[] args) {
        // DemoCallbackNoUserData d = new DemoCallbackNoUserData();
        ReturnData r = new ReturnData();
        CallbackListener l = new CallbackListener() {

            @Override
            public void callback() {
                System.out.println(">> callback ");
            }

        };

        System.out.println("calling native and wait");
        callNative(10, 20, r, l);
        try {
            Thread.sleep(1000);
        } catch (InterruptedException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }

        ReturnData r2 = callSync(5, 20);
        System.out.println("sync call return " + r2.x);
        r2 = callSync(3, 10);
        System.out.println("sync call return " + r2.x);

        System.out.println("called native twice in a row, should fail");
        callNative(10, 20, r, l);
        callNative(0, 5, r, l);
        // need to sleep, because dummy external API does not know about java
        // and does
        // not attach its thread to the JVM
        try {
            Thread.sleep(1000);
        } catch (InterruptedException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }

    }

}
