package org.lsst.ccs.drivers.demo;

import java.util.Random;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;

/**
 **************************************************************************
 **
 ** General access routines for this device
 **
 ** Driver methods are annotated for use by CCS command dictionary and shell
 ** system. Exceptions are passed through to next level where either subsystem
 ** or shell code handles them.
 **
 ** @author Your Name
 **
 ***************************************************************************
 */


public class DemoDriver {

    /**
     **************************************************************************
     **
     ** Public constants
     **
     ***************************************************************************
     */
    /** The First of our devices */
    public final static int DEVID0 = 0;

    /** The number of known device identifiers */
    private final static int N_DIDS = 1;

    /**
     **************************************************************************
     **
     ** Private constants
     **
     ***************************************************************************
     */
    private final static int[] pids = new int[N_DIDS];

    static { // array of valid device ids
        pids[DEVID0] = DEVID0;
    }

    /**
     **************************************************************************
     **
     ** Private fields
     **
     ***************************************************************************
     */
    private int devId = -1;

    /**
     **************************************************************************
     **
     ** Constructor
     **
     ** Generally not needed unless some initialization required at creation.
     ** See Ad7794Eval.java and TempControl24C.java for examples.
     **
     ***************************************************************************
     */
    public DemoDriver() throws DriverException {
    }

    /**
     **************************************************************************
     **
     ** Gets the device ID
     **
     ** @return The ID of the current open device, or -1 if no device open.
     **
     ***************************************************************************
     */
    public int getDevId() {
        return devId;
    }

    /**
     **************************************************************************
     **
     ** Open the device
     **
     ** @param did The device identifier is index from our list of devs
     **
     **
     ***************************************************************************
     */
    @Command(name = "open", description = "Open a device for use")
    public void open(
            @Argument(name = "devId", description = "The device ID of the device to open") int did
    ) throws DriverException {
        if (devId >= 0) {
            throw new DriverException("Device already open");
        }

        if (did < 0 || did >= pids.length) {
            throw new DriverException("Invalid device identifier");
        }

        devId = did;
    }

    /**
     **************************************************************************
     **
     ** Close the device
     **
     ** @param did The device identifier is index from our list of devs
     **
     ***************************************************************************
     */
    @Command(name = "close", description = "Close the device")
    public void close() throws DriverException {
        if (devId < 0) {
            throw new DriverException("No device open");
        }

        devId = -1;
    }

    /**
     **************************************************************************
     **
     ** Read the device
     **
     ** @param count The number of bytes to read.
     ** @param data An array of sufficient size to contain the data.
     ** @return The number of bytes read
     **
     ** @throws DriverException
     **
     ***************************************************************************
     */
    @Command(name = "read", description = "Read <count> bytes from device and print in hex format\n\tread <N> ")
    public String read(
            @Argument(name = "count", description = "The number of bytes to read") int count
    ) throws Exception {
        byte[] data = new byte[count];
        int bytesRead = 0;
        bytesRead = read(count, data);
        if (bytesRead != count) {
            throw new DriverException("bad count returned by read()");
        }
        return toHex(data);
    }

    private synchronized int read(int count, byte[] data) throws DriverException {
        if (devId < 0) {
            throw new DriverException("No device open");
        }
        if (data.length < count) {
            throw new DriverException("buffer size less than count given");
        }

        Random rnd = new Random(System.currentTimeMillis());
        byte[] tmpBuf = new byte[count];
        rnd.nextBytes(tmpBuf);
        System.arraycopy(tmpBuf, 0, data, 0, count);

        return count;
    }

    /**
     **************************************************************************
     **
     ** Write to the device (would add address, etc. for real device)
     **
     ** @param count The number of bytes to write.
     ** @param buf An array of bytes with the data to write
     ** @return The number of bytes written
     **
     ** @throws DriverException
     **
     ***************************************************************************
     */
    @Command(name = "write", description = "write the input string to the device\n\twrite <input-string>")
    public void write(
            @Argument(name = "input-string", description = "The input string") String input
    ) throws DriverException {
        byte[] data = input.getBytes();
        write(data.length, data);
    }

    private synchronized int write(int count, byte[] data) throws DriverException {
        if (devId < 0) {
            throw new DriverException("No device open");
        }
        byte[] tmpBuf = new byte[count];
        if (data.length < count) {
            throw new DriverException("buffer size less than count given");
        }
        System.arraycopy(data, 0, tmpBuf, 0, count);

        return count;
    }

	// Stolen from http://stackoverflow.com/questions/332079
    private static String toHex(byte[] digest) {
        String digits = "0123456789abcdef";
        StringBuilder sb = new StringBuilder(digest.length * 2);
        for (byte b : digest) {
            int bi = b & 0xff;
            sb.append(digits.charAt(bi >> 4));
            sb.append(digits.charAt(bi & 0xf));
        }
        return sb.toString();
    }

}
