package org.lsst.ccs.drivers.dataforth;

import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Routines for controlling a Dataforth MAQ20 discrete input module.
 *
 *  @author  Owen Saxton
 */
public class Maq20DiscreteIn extends Maq20Discrete {

    static final int
        READ_ALL_ADDR  = 1035,
        LOGIC_POL_ADDR = 1041;


    /**
     *  Constructor.
     *
     *  @param  maq    The underlying Maq20 object
     *  @param  modId  The module registration ID
     *  @throws DriverException
     */
    Maq20DiscreteIn(Maq20 maq, int modId) throws DriverException
    {
        super(maq, modId);
    }


    /**
     *  Reads all channels.
     *
     *  @return  A bit mask of active channels
     *  @throws  DriverException
     */
    public int readDiscAll() throws DriverException
    {
        setStraightLogic();
        return maq.readRegisterPair((short)(module.baseAddr + READ_ALL_ADDR));
    }


    /**
     *  Reads a set of input channels.
     *
     *  Overrides superclass to set inverted logic
     *
     *  @param  chan   The first channel number
     *  @param  count  The number of channels
     *  @return  The array of data values
     *  @throws  DriverException
     */
    @Override
    public int[] readDiscIn(int chan, int count) throws DriverException
    {
        checkInputChannels(chan, count);
        setStraightLogic();
        return readDiscData(CHAN_DATA_ADDR, chan, count, false);
    }


    /**
     *  Sets straight logic.
     * 
     *  This ensures that the device always behaves as expected
     *
     *  @throws  DriverException
     */
    void setStraightLogic() throws DriverException
    {
        maq.writeRegister((short)(module.baseAddr + LOGIC_POL_ADDR), Maq20.ZERO);
    }

}
