package org.lsst.ccs.drivers.dataforth;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Code for implementing MAQ20 simulation.
 * 
 *  @author saxton
 */
public class SimMaq20 {

    static class ModuleSpec {

        short numInputChans;
        short numOutputChans;
        RangeSpec[] rangeSpecs;

        ModuleSpec(int numInputChans, int numOutputChans, RangeSpec[] rangeSpecs)
        {
            this.numInputChans = (short)numInputChans;
            this.numOutputChans = (short)numOutputChans;
            this.rangeSpecs = rangeSpecs;
        }

    }

    static class RangeSpec {

        short engFsPower;
        short engNegFs;
        short engPosFs;

        RangeSpec(int engFsPower, int engNegFs, int engPosFs)
        {
            this.engFsPower = (short)engFsPower;
            this.engNegFs = (short)engNegFs;
            this.engPosFs = (short)engPosFs;
        }

    }

    private static final RangeSpec[] idRangeSpecs = new RangeSpec[]{new RangeSpec(-3, 0, 20), new RangeSpec(-3, 4, 20)};
    private static final RangeSpec[] ioRangeSpecs = new RangeSpec[]{new RangeSpec(0, 0, 20), new RangeSpec(0, 4, 20)};
    private static final RangeSpec[] vdRangeSpecs = new RangeSpec[]{new RangeSpec(0, -60, 60), new RangeSpec(0, -40, 40),
                                                                    new RangeSpec(0, -20, 20), new RangeSpec(0, -10, 10),
                                                                    new RangeSpec(0, -5, 5)};
    private static final RangeSpec[] mvdRangeSpecs = new RangeSpec[]{new RangeSpec(0, -2, 2), new RangeSpec(0, -1, 1),
                                                                     new RangeSpec(-3, -250, 250), new RangeSpec(-3, -100, 100),
                                                                     new RangeSpec(-3, -50, 50)};
    private static final RangeSpec[] voRangeSpecs = new RangeSpec[]{new RangeSpec(0, -10, 10), new RangeSpec(0, -5, 5),
                                                                    new RangeSpec(-1, -25, 25), new RangeSpec(0, 0, 10),
                                                                    new RangeSpec(0, 0, 5), new RangeSpec(-1, 0, 25)};
    private static final RangeSpec[] tctRangeSpecs = new RangeSpec[]{new RangeSpec(0, -100, 400), new RangeSpec(0, -100, 220)};
    private static final RangeSpec[] tcjRangeSpecs = new RangeSpec[]{new RangeSpec(0, -100, 760), new RangeSpec(0, -100, 393),
                                                                     new RangeSpec(0, -100, 199)};
    private static final RangeSpec[] tckRangeSpecs = new RangeSpec[]{new RangeSpec(0, -100, 1350), new RangeSpec(0, -100, 651),
                                                                     new RangeSpec(0, -100, 332)};
    private static final RangeSpec[] tcrsRangeSpecs = new RangeSpec[]{new RangeSpec(0, 0, 1750), new RangeSpec(0, 0, 990),
                                                                      new RangeSpec(0, 0, 1750), new RangeSpec(0, 0, 970)};
    private static final RangeSpec[] rtdRangeSpecs = new RangeSpec[]{new RangeSpec(0, -200, 850), new RangeSpec(0, -200, 200),
                                                                     new RangeSpec(0, -100, 100)};
    private static final RangeSpec[] emptyRangeSpecs = new RangeSpec[0];
    private static final Map<Maq20.ModuleType, String> typeNameMap = new HashMap<>();
    static {
        for (Map.Entry e : Maq20.typeMap.entrySet()) {
            typeNameMap.put((Maq20.ModuleType)e.getValue(), (String)e.getKey());
        }
    }
    private static final Map<Maq20.ModuleType, ModuleSpec> moduleSpecMap = new HashMap<>();
    static {
        moduleSpecMap.put(Maq20.ModuleType.DINP, new ModuleSpec(20, 0, emptyRangeSpecs));
        moduleSpecMap.put(Maq20.ModuleType.DIOH, new ModuleSpec(5, 5, emptyRangeSpecs));
        moduleSpecMap.put(Maq20.ModuleType.DIOL, new ModuleSpec(5, 5, emptyRangeSpecs));
        moduleSpecMap.put(Maq20.ModuleType.DOUT, new ModuleSpec(0, 20, emptyRangeSpecs));
        moduleSpecMap.put(Maq20.ModuleType.ID, new ModuleSpec(8, 0, idRangeSpecs));
        moduleSpecMap.put(Maq20.ModuleType.IO, new ModuleSpec(0, 8, ioRangeSpecs));
        moduleSpecMap.put(Maq20.ModuleType.IS, new ModuleSpec(16, 0, idRangeSpecs));
        moduleSpecMap.put(Maq20.ModuleType.JTC, new ModuleSpec(8, 0, tcjRangeSpecs));
        moduleSpecMap.put(Maq20.ModuleType.KTC, new ModuleSpec(8, 0, tckRangeSpecs));
        moduleSpecMap.put(Maq20.ModuleType.MVD, new ModuleSpec(8, 0, mvdRangeSpecs));
        moduleSpecMap.put(Maq20.ModuleType.RSTC, new ModuleSpec(8, 0, tcrsRangeSpecs));
        moduleSpecMap.put(Maq20.ModuleType.RTD, new ModuleSpec(6, 0, rtdRangeSpecs));
        moduleSpecMap.put(Maq20.ModuleType.TTC, new ModuleSpec(8, 0, tctRangeSpecs));
        moduleSpecMap.put(Maq20.ModuleType.VD, new ModuleSpec(8, 0, vdRangeSpecs));
        moduleSpecMap.put(Maq20.ModuleType.VO, new ModuleSpec(0, 8, voRangeSpecs));
        moduleSpecMap.put(Maq20.ModuleType.VS, new ModuleSpec(16, 0, vdRangeSpecs));
    }

    private final Maq20 maq20;
    private final Map<Integer, Short> registerData = new HashMap<>();

    public SimMaq20(Maq20 maq20)
    {
        this.maq20 = maq20;
    }

    /**
     *  Initializes needed simulated registers.
     * 
     *  @param  modTypes  Array of module types
     *  @param  serials  Array of serial numbers 
     */
    public void initialize(Maq20.ModuleType[] modTypes, String[] serials)
    {
        int nModule = Math.min(modTypes.length, serials.length);
        for (int id = 1; id <= nModule; id++) {
            Maq20.ModuleType modType = modTypes[id - 1];
            int baseAddr = Maq20.MODU_ADDR_INCR * id;
            setModuleName(id, "MAQ20-" + typeNameMap.get(modType));
            setSerialNum(id, serials[id - 1]);
            writeRegister((short)(Maq20.MODU_STATUS_ADDR + id), Maq20.ONE);
            ModuleSpec modSpec = moduleSpecMap.get(modType);
            writeRegister((short)(baseAddr + Maq20.NUM_INP_CHANS_ADDR), modSpec.numInputChans);
            writeRegister((short)(baseAddr + Maq20.NUM_OUT_CHANS_ADDR), modSpec.numOutputChans);
            writeRegister((short)(baseAddr + Maq20.RANGE_COUNT_ADDR), (short)modSpec.rangeSpecs.length);
            for (int j = 0; j < modSpec.rangeSpecs.length; j++) {
                int rangeBaseAddr = baseAddr + Maq20.RANGE_DATA_ADDR + Maq20.RANGE_DATA_INCR * j;
                RangeSpec rangeSpec = modSpec.rangeSpecs[j];
                writeRegister((short)(rangeBaseAddr + Maq20.CNT_NFS_OFFS), Maq20.ZERO);
                writeRegister((short)(rangeBaseAddr + Maq20.CNT_PFS_OFFS), (short)8000);
                writeRegister((short)(rangeBaseAddr + Maq20.ENG_NFS_OFFS), rangeSpec.engNegFs);
                writeRegister((short)(rangeBaseAddr + Maq20.ENG_PFS_OFFS), rangeSpec.engPosFs);
                writeRegister((short)(rangeBaseAddr + Maq20.ENG_FS_PWR_OFFS), rangeSpec.engFsPower);
            }
        }
    }

    /**
     *  Sets the value for an input channel.
     * 
     *  @param  modId  The module ID
     *  @param  chan   The channel number
     *  @param  value  The value to set
     *  @throws DriverException 
     */
    void setChannelValue(int modId, int chan, double value) throws DriverException
    {
        Maq20.ModuleData modData = maq20.getModuleData(modId, false);
        int range = modData.range[chan];
        short raw = (short)((value + modData.offset[range]) / modData.scale[range] + 0.5);
        writeRegister((short)(modData.baseAddr + Maq20Analog.CHAN_DATA_ADDR + chan), raw);
    }

    /**
     *  Reads a set of registers.
     *
     *  @param  addr   The first register address
     *  @param  count  The number of registers to read
     *  @return  An array of register values
     *  @throws DriverException
     */
    short[] readRegisters(short addr, short count)
    {
        short[] values = new short[count];
        for (int j = 0; j < count; j++) {
            Short value = registerData.get(addr + j);
            values[j] = value == null ? 0 : value;
        }
        return values;
    }

    /**
     *  Writes a set of registers.
     *
     *  @param  addr   The first register address
     *  @param  value  The array of values to write
     */
    void writeRegisters(short addr, short[] value)
    {
        for (int j = 0; j < value.length; j++) {
            registerData.put(addr + j, value[j]);
        }
    }

    /**
     *  Writes a register.
     *
     *  @param  addr   The register address
     *  @param  value  The value to write
     */
    void writeRegister(short addr, short value)
    {
        registerData.put((int)addr, value);
    }

    /**
     *  Sets the module name.
     *
     *  @param  modId  The module registration ID
     *  @param  modName  The module name
     */
    private void setModuleName(int modId, String modName)
    {
        short addr = (short)(Maq20.MODU_ADDR_INCR * modId + Maq20.DEV_NAME_ADDR);
        writeRegisters(addr, makeShorts(modName));
    }

    /**
     *  Sets the serial number.
     *
     *  @param  modId  The module registration ID
     *  @param  serial  The serial number
     */
    private void setSerialNum(int modId, String serial)
    {
        short addr = (short)(Maq20.MODU_ADDR_INCR * modId + Maq20.SER_NUM_ADDR);
        writeRegisters(addr, makeShorts(serial));
    }

    /**
     *  Make an array of shorts from a string.
     * 
     *  @param text
     *  @return 
     */
    private short[] makeShorts(String text)
    {
        short[] values = new short[text.length()];
        for (int j = 0; j < values.length; j++) {
            values[j] = (short)text.charAt(j);
        }
        return values;
    }

}
