package org.lsst.ccs.drivers.dataforth;

import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Control a Dataforth MAQ20 discrete I/O module PWM generator.
 *
 *  @author  Owen Saxton
 */
public class Maq20DiscretePWM extends Maq20DiscreteFunc {

    /**
     *  Private data.
     */
    private static final short
        TIMEBASE_ADDR = 3,
        OUTP2_ENAB_ADDR = 4,
        PERIOD_ADDR = 6,
        OUTP1_LOW_ADDR = 8,
        OUTP2_LOW_ADDR = 10;


    /**
     *  Constructor.
     *
     *  @param  maq    The underlying Maq20 object
     *  @param  modId  The module registration ID
     *  @throws DriverException
     */
    Maq20DiscretePWM(Maq20 maq, int modId) throws DriverException
    {
        super(maq, modId);
    }


    /**
     *  Enables the PWM generator.
     *
     *  @param  chan  The function channel (0 or 1)
     *  @throws  DriverException
     */
    public void enable(int chan) throws DriverException 
    {
        setFunction(chan, FUNC_PWM_GEN);
    }


    /**
     *  Gets whether the PWM generator is enabled.
     *
     *  @param  chan  The function channel (0 or 1)
     *  @return  Whether anabled
     *  @throws  DriverException
     */
    public boolean isEnabled(int chan) throws DriverException 
    {
        return getFunction(chan) == FUNC_PWM_GEN;
    }


    /**
     *  Sets the timebase.
     *
     *  @param  chan   The function channel (0 or 1)
     *  @param  base   The encoded timebase value
     *  @throws  DriverException
     */
    public void setTimebase(int chan, int base) throws DriverException 
    {
        maq.writeRegister(getRegAddr(chan, FUNC_PWM_GEN, TIMEBASE_ADDR), (short)base);
    }


    /**
     *  Gets the timebase.
     *
     *  @param  chan  The function channel (0 or 1)
     *  @return  The encoded timebase
     *  @throws  DriverException
     */
    public int getTimebase(int chan) throws DriverException 
    {
        return maq.readRegister(getRegAddr(chan, FUNC_PWM_GEN, TIMEBASE_ADDR));
    }


    /**
     *  Sets the output 2 enabled state.
     *
     *  @param  chan    The function channel (0 or 1)
     *  @param  enable  Whether to enable output 2
     *  @throws  DriverException
     */
    public void enableOutput2(int chan, boolean enable) throws DriverException 
    {
        maq.writeRegister(getRegAddr(chan, FUNC_PWM_GEN, OUTP2_ENAB_ADDR), enable ? Maq20.ONE : Maq20.ZERO);
    }


    /**
     *  Gets the output 2 enabled state.
     *
     *  @param  chan  The function channel (0 or 1)
     *  @return  Whether output 2 is enabled
     *  @throws  DriverException
     */
    public boolean isOutput2Enabled(int chan) throws DriverException 
    {
        return maq.readRegister(getRegAddr(chan, FUNC_PWM_GEN, OUTP2_ENAB_ADDR)) != 0;
    }


    /**
     *  Sets the period.
     *
     *  @param  chan    The function channel (0 or 1)
     *  @param  period  The period
     *  @throws  DriverException
     */
    public void setPeriod(int chan, int period) throws DriverException 
    {
        maq.writeRegisterPair(getRegAddr(chan, FUNC_PWM_GEN, PERIOD_ADDR), period);
    }


    /**
     *  Gets the period.
     *
     *  @param  chan  The function channel (0 or 1)
     *  @return  The period
     *  @throws  DriverException
     */
    public int getPeriod(int chan) throws DriverException 
    {
        return maq.readRegisterPair(getRegAddr(chan, FUNC_PWM_GEN, PERIOD_ADDR));
    }


    /**
     *  Sets the low time for output 1.
     *
     *  @param  chan  The function channel (0 or 1)
     *  @param  time  The low time
     *  @throws  DriverException
     */
    public void setLowTime1(int chan, int time) throws DriverException 
    {
        maq.writeRegisterPair(getRegAddr(chan, FUNC_PWM_GEN, OUTP1_LOW_ADDR), time);
    }


    /**
     *  Gets the low time for output 1.
     *
     *  @param  chan  The function channel (0 or 1)
     *  @return  The low time
     *  @throws  DriverException
     */
    public int getLowTime1(int chan) throws DriverException 
    {
        return maq.readRegisterPair(getRegAddr(chan, FUNC_PWM_GEN, OUTP1_LOW_ADDR));
    }


    /**
     *  Sets the low time for output 2.
     *
     *  @param  chan  The function channel (0 or 1)
     *  @param  time  The low time
     *  @throws  DriverException
     */
    public void setLowTime2(int chan, int time) throws DriverException 
    {
        maq.writeRegisterPair(getRegAddr(chan, FUNC_PWM_GEN, OUTP2_LOW_ADDR), time);
    }


    /**
     *  Gets the low time for output 2.
     *
     *  @param  chan  The function channel (0 or 1)
     *  @return  The low time
     *  @throws  DriverException
     */
    public int getLowTime2(int chan) throws DriverException 
    {
        return maq.readRegisterPair(getRegAddr(chan, FUNC_PWM_GEN, OUTP2_LOW_ADDR));
    }

}
