package org.lsst.ccs.drivers.dataforth;

import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Routines for controlling a Dataforth MAQ20 discrete I/O module functions.
 *
 *  @author  Owen Saxton
 */
public class Maq20DiscreteFunc extends Maq20Discrete {

    /**
     *  Public data.
     */
    public static final short
        TIMEBASE_SECS = 0,
        TIMEBASE_MSECS = 1,
        TIMEBASE_USECS = 2;

    /**
     *  Package data.
     */
    static final int
        FUNC_NONE = 0,
        FUNC_PULSE = 1,
        FUNC_PULSE_DEB = 2,
        FUNC_WAVEFORM = 3,
        FUNC_EVENT_TIME = 4,
        FUNC_FREQ_GEN = 5,
        FUNC_PWM_GEN = 6,
        FUNC_PULSE_GEN = 7,
        NUM_FUNCTIONS = 8,
        NUM_FUNC_CHANS = 2,

        FUNC_BASE_ADDR = 1100,
        FUNC_BASE_INCR = 100,
        FUNCTION_ADDR = 0,
        ARM_ADDR = 1,
        FUNC_STAT_ADDR = 2,
        ALARM_STAT_ADDR = 3,
        SAVE_CONFIG_ADDR = 90;


    /**
     *  Constructor.
     *
     *  @param  maq    The underlying Maq20 object
     *  @param  modId  The module registration ID
     *  @throws DriverException
     */
    Maq20DiscreteFunc(Maq20 maq, int modId) throws DriverException
    {
        super(maq, modId);
    }


    /**
     *  Checks whether correct function is active
     *
     *  @param  chan  The function channel
     *  @param  func  The function code
     *  @throws  DriverException
     */
    void checkFunction(int chan, int func) throws DriverException
    {
        getBaseAddr(chan, func);
    }


    /**
     *  Sets the active function
     *
     *  @param  chan  The function channel
     *  @param  func  The function code
     *  @throws  DriverException
     */
    void setFunction(int chan, int func) throws DriverException
    {
        if (func < 0 || func >= NUM_FUNCTIONS) {
            throw new DriverException("Invalid function code (" + func + ")");
        }
        maq.writeRegister(getRegAddr(chan, FUNCTION_ADDR), (short)func);
    }


    /**
     *  Gets the active function.
     *
     *  @param  chan  The function channel
     *  @return  The function code
     *  @throws  DriverException
     */
    int getFunction(int chan) throws DriverException
    {
        return maq.readRegister(getRegAddr(chan, FUNCTION_ADDR));
    }


    /**
     *  Arms a function.
     *
     *  @param  chan  The function channel
     *  @throws  DriverException
     */
    public void armFunction(int chan) throws DriverException
    {
        maq.writeRegister(getRegAddr(chan, ARM_ADDR), Maq20.ONE);
    }


    /**
     *  Disarms a function.
     *
     *  @param  chan  The function channel
     *  @throws  DriverException
     */
    public void disarmFunction(int chan) throws DriverException
    {
        maq.writeRegister(getRegAddr(chan, ARM_ADDR), Maq20.ZERO);
    }


    /**
     *  Gets the function armed state.
     *
     *  @param  chan  The function channel
     *  @return  Whether function is armed
     *  @throws  DriverException
     */
    public boolean isFunctionArmed(int chan) throws DriverException
    {
        return maq.readRegister(getRegAddr(chan, ARM_ADDR)) != 0;
    }


    /**
     *  Saves the function configuration.
     *
     *  @param  chan  The function channel
     *  @throws  DriverException
     */
    public void saveConfig(int chan) throws DriverException
    {
        maq.writeRegister(getRegAddr(chan, SAVE_CONFIG_ADDR), Maq20.ZERO);
    }


    /**
     *  Gets a function base address.
     *
     *  @param  chan  The function channel
     *  @return  The base address
     *  @throws  DriverException
     */
    int getBaseAddr(int chan) throws DriverException
    {
        if (chan < 0 || chan >= NUM_FUNC_CHANS) {
            throw new DriverException("Invalid function channel (" + chan + ")");
        }
        return module.baseAddr + FUNC_BASE_ADDR + chan * FUNC_BASE_INCR;
    }


    /**
     *  Gets a function base address and checks that function is active.
     *
     *  @param  chan  The function channel
     *  @param  func  The function code
     *  @return  The base address
     *  @throws  DriverException
     */
    int getBaseAddr(int chan, int func) throws DriverException
    {
        int baseAddr = getBaseAddr(chan);
        int actFunc = maq.readRegister((short)(baseAddr + FUNCTION_ADDR));
        if (func != actFunc) {
            throw new DriverException("Active function (" + actFunc + ") doesn't match request");
        }
        return baseAddr;
    }


    /**
     *  Gets a function register address.
     *
     *  @param  chan    The function channel
     *  @param  offset  The register address offset
     *  @return  The register address
     *  @throws  DriverException
     */
    short getRegAddr(int chan, int offset) throws DriverException
    {
        return (short)(getBaseAddr(chan) + offset);
    }


    /**
     *  Gets a function register address and checks that function is active.
     *
     *  @param  chan    The function channel
     *  @param  func    The function code
     *  @param  offset  The register address offset
     *  @return  The register address
     *  @throws  DriverException
     */
    short getRegAddr(int chan, int func, int offset) throws DriverException
    {
        return (short)(getBaseAddr(chan, func) + offset);
    }

}
