package org.lsst.ccs.drivers.dataforth;

import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Control a Dataforth MAQ20 DIO module frequency generator.
 *
 *  @author  Owen Saxton
 */
public class Maq20DiscreteFreq extends Maq20DiscreteFunc {

    /**
     *  Private data.
     */
    private static final short
        FREQUENCY_ADDR = 4;


    /**
     *  Constructor.
     *
     *  @param  maq    The underlying Maq20 object
     *  @param  modId  The module registration ID
     *  @throws DriverException
     */
    Maq20DiscreteFreq(Maq20 maq, int modId) throws DriverException
    {
        super(maq, modId);
    }


    /**
     *  Enables the frequency generator.
     *
     *  @param  chan  The function channel (0 or 1)
     *  @throws  DriverException
     */
    public void enable(int chan) throws DriverException 
    {
        setFunction(chan, FUNC_FREQ_GEN);
    }


    /**
     *  Gets whether the frequency generator is enabled.
     *
     *  @param  chan  The function channel (0 or 1)
     *  @return  Whether enabled
     *  @throws  DriverException
     */
    public boolean isEnabled(int chan) throws DriverException 
    {
        return getFunction(chan) == FUNC_FREQ_GEN;
    }


    /**
     *  Sets the frequency.
     *
     *  @param  chan  The function channel (0 or 1)
     *  @param  freq  The frequency
     *  @throws  DriverException
     */
    public void setFrequency(int chan, int freq) throws DriverException 
    {
        maq.writeRegisterPair(getRegAddr(chan, FUNC_FREQ_GEN, FREQUENCY_ADDR), freq);
    }


    /**
     *  Gets the frequency.
     *
     *  @param  chan  The function channel (0 or 1)
     *  @return  The frequency
     *  @throws  DriverException
     */
    public int getFrequency(int chan) throws DriverException 
    {
        return maq.readRegisterPair(getRegAddr(chan, FUNC_FREQ_GEN, FREQUENCY_ADDR));
    }

}
