package org.lsst.ccs.drivers.dataforth;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.modbus.TestModbus;
import java.util.HashMap;
import java.util.Map;

/**
 *  Program to test the Dataforth MAQ20 device driver
 * 
 *  @author Owen Saxton
 */
public class TestMaq20 extends TestModbus {

    /**
     *  Private data
     */
    public enum DiscFunc {

        PULSE(Maq20DiscreteFunc.FUNC_PULSE),
        PWMGEN(Maq20DiscreteFunc.FUNC_PWM_GEN),
        FREQGEN(Maq20DiscreteFunc.FUNC_FREQ_GEN);

        int value;

        DiscFunc(int func) {
            this.value = func;
        }

        int getValue() {
            return value;
        }

    }

    public enum Timebase {

        SECS(Maq20DiscretePWM.TIMEBASE_SECS),
        MSECS(Maq20DiscretePWM.TIMEBASE_MSECS),
        USECS(Maq20DiscretePWM.TIMEBASE_USECS);

        int value;

        Timebase(int func) {
            this.value = func;
        }

        int getValue() {
            return value;
        }

    }

    private static final Map<Integer, String> funcDescs = new HashMap<>();
    static {
        funcDescs.put(Maq20DiscreteFunc.FUNC_NONE, "None");
        funcDescs.put(Maq20DiscreteFunc.FUNC_PULSE, "Pulse counter");
        funcDescs.put(Maq20DiscreteFunc.FUNC_PULSE_DEB, "Debounced pulse counter");
        funcDescs.put(Maq20DiscreteFunc.FUNC_WAVEFORM, "Waveform measurement");
        funcDescs.put(Maq20DiscreteFunc.FUNC_EVENT_TIME, "Time between events");
        funcDescs.put(Maq20DiscreteFunc.FUNC_FREQ_GEN, "Frequency generator");
        funcDescs.put(Maq20DiscreteFunc.FUNC_PWM_GEN, "PWM generator");
        funcDescs.put(Maq20DiscreteFunc.FUNC_PULSE_GEN, "One-shot pulse generator");
    }
    private static final Map<Integer, String> timebaseDescs = new HashMap<>();
    static {
        timebaseDescs.put((int)Maq20DiscreteFunc.TIMEBASE_SECS, "seconds");
        timebaseDescs.put((int)Maq20DiscreteFunc.TIMEBASE_MSECS, "milliseconds");
        timebaseDescs.put((int)Maq20DiscreteFunc.TIMEBASE_USECS, "microseconds");
    }
    private final Maq20 maq;


    /**
     *  Constructor
     */
    public TestMaq20()
    {
        super(new Maq20());
        maq = (Maq20)mod;
    }

    @Override
    @Command(description="Open connection to general device")
    public void open(@Argument(name="type", description="Connection type") Maq20.ConnType type,
                     @Argument(name="ident", description="Device identifier") String ident) throws DriverException
    {
        super.open(type, ident);
    }

    @Override
    @Command(description="Open connection to general device")
    public void open(@Argument(name="type", description="Connection type") Maq20.ConnType type,
                     @Argument(name="ident", description="Device identifier") String ident,
                     @Argument(name="param", description="Device parameter") int param) throws DriverException
    {
        super.open(type, ident, param);
    }

    @Override
    @Command(description="Open connection to serial device")
    public void open(@Argument(name="type", description="Connection type: net, serial or ftdi") Maq20.ConnType type,
                     @Argument(name="ident", description="Device identifier") String ident,
                     @Argument(name="baud", description="Baud rate") int baud,
                     @Argument(name="dbits", description="The number of data bits") Maq20.DataBits dbits,
                     @Argument(name="sbits", description="The number of stop bits") Maq20.StopBits sbits,
                     @Argument(name="parity", description="The parity") Maq20.Parity parity,
                     @Argument(name="flow", description="The flow control") Maq20.FlowCtrl flow) throws DriverException
    {
        super.open(type, ident, baud, dbits, sbits, parity, flow);
    }

    @Command(description="Open connection to USB device")
    public void openUsb(@Argument(name="serial", description="Serial number") String serial) throws DriverException
    {
        maq.openUsb(serial);
    }

    @Command(description="Register modules")
    public void register(@Argument(name="serial", description="Serial numbers") String... serial) throws DriverException
    {
        maq.register(serial);
    }

    @Command(description="Show the IP address")
    public String showIPAddress() throws DriverException
    {
        return "IP address: " + maq.getIPAddress();
    }

    @Command(description="Set the IP address")
    public void setIPAddress(@Argument(name="ipaddr", description="IP address") String ipAddr) throws DriverException
    {
        maq.setIPAddress(ipAddr);
    }

    @Command(description="Show the subnet mask")
    public String showSubnetMask() throws DriverException
    {
        return "Subnet mask: " + maq.getSubnetMask();
    }

    @Command(description="Set the subnet mask")
    public void setSubnetMask(@Argument(name="mask", description="subnet mask") String mask) throws DriverException
    {
        maq.setSubnetMask(mask);
    }

    @Command(description="Show the temperature")
    public String showTemperature() throws DriverException
    {
        return "Temperature: " + maq.readTemperature();
    }

    @Command(description="Save the parameters")
    public void saveParams() throws DriverException
    {
        maq.saveParameters();
    }

    @Command(description="Show all module IDs")
    public String showIds() throws DriverException
    {
        int[] ids = maq.getModuleIds();
        if (ids.length == 0) {
            return "No modules present";
        }
        StringBuilder text = new StringBuilder("Valid module IDs:");
        for (int id : ids) {
            text.append(" ").append(id);
        }
        return text.toString();
    }

    @Command(description="Show all modules")
    public String showModules() throws DriverException
    {
        int[] ids = maq.getModuleIds();
        if (ids.length == 0) {
            return "No modules present";
        }
        StringBuilder text = new StringBuilder("ID  Type  Serial No.");
        for (int id : ids) {
            text.append(String.format("\n%2s  %-4s  %s", id, maq.getModuleType(id), maq.getSerialNumber(id)));
        }
        return text.toString();
    }

    @Command(description="Show module's ID")
    public String showId(@Argument(name="serial", description="Serial number") String serial) throws DriverException
    {
        return "Module ID: " + maq.getModuleId(serial);
    }

    @Command(description="Blink a module's status light")
    public void identify(@Argument(name="modid", description="Module ID") int modId) throws DriverException
    {
        maq.identifyModule(modId);
    }

    @Command(description="Reset a module")
    public void reset(@Argument(name="modid", description="Module ID") int modId,
                      @Argument(name="deflt", description="Whether to apply defaults") boolean deflt) throws DriverException
    {
        maq.resetModule(modId, deflt);
    }

    @Command(description="Show module's name")
    public String showName(@Argument(name="modid", description="Module ID") int modId) throws DriverException
    {
        return "Device name: " + maq.getModuleName(modId);
    }

    @Command(description="Show module's type")
    public String showType(@Argument(name="modid", description="Module ID") int modId) throws DriverException
    {
        return "Module type: " + maq.getModuleType(modId);
    }

    @Command(description="Show module's serial number")
    public String showSerial(@Argument(name="modid", description="Module ID") int modId) throws DriverException
    {
        return "Serial number: " + maq.getSerialNumber(modId);
    }

    @Command(description="Show module's FW revision")
    public String showFwRevision(@Argument(name="modid", description="Module ID") int modId) throws DriverException
    {
        return "Firmware revision: " + maq.getFwRevision(modId);
    }

    @Command(description="Show module's date code")
    public String showDateCode(@Argument(name="modid", description="Module ID") int modId) throws DriverException
    {
        return "Date code: " + maq.getDateCode(modId);
    }

    @Command(description="Show module's input channel count")
    public String showNumInputs(@Argument(name="modid", description="Module ID") int modId) throws DriverException
    {
        return "Input channel count: " + maq.getNumInputs(modId);
    }

    @Command(description="Show module's output channel count")
    public String showNumOutputs(@Argument(name="modid", description="Module ID") int modId) throws DriverException
    {
        return "Output channel count: " + maq.getNumOutputs(modId);
    }

    @Command(description="Show module's range count")
    public String showNumRanges(@Argument(name="modid", description="Module ID") int modId) throws DriverException
    {
        return "Range count: " + maq.getAnalog(modId).getNumRanges();
    }

    @Command(description="Show channel's range")
    public String showRange(@Argument(name="modid", description="Module ID") int modId,
                            @Argument(name="chan", description="Channel number") int chan) throws DriverException
    {
        return "Range: " + maq.getAnalog(modId).getRange(chan);
    }

    @Command(description="Set channel's range")
    public void setRange(@Argument(name="modid", description="Module ID") int modId,
                         @Argument(name="chan", description="Channel number") int chan,
                         @Argument(name="range", description="Range number") int range) throws DriverException
    {
        maq.getAnalog(modId).setRange(chan, range);
    }

    @Command(description="Show channel's enabled state")
    public String showEnabled(@Argument(name="modid", description="Module ID") int modId,
                              @Argument(name="chan", description="Channel number") int chan) throws DriverException
    {
        return "Enabled state: " + (maq.getAnalIn(modId).isEnabled(chan) ? "on" : "off");
    }

    @Command(description="Set channel's enabled state")
    public void setEnabled(@Argument(name="modid", description="Module ID") int modId,
                           @Argument(name="chan", description="Channel number") int chan,
                           @Argument(name="state", description="Enabled state") OnOff state) throws DriverException
    {
        maq.getAnalIn(modId).enable(chan, state == OnOff.ON);
    }

    @Command(description="Read channels' data values")
    public String readValue(@Argument(name="modid", description="Module ID") int modId,
                            @Argument(name="chan", description="Channel number") int chan,
                            @Argument(name="count", description="Number of channels") int count) throws DriverException
    {
        return formatDoubles(maq.getAnalog(modId).readValue(chan, count));
    }

    @Command(description="Read a channel's data value")
    public String readValue(@Argument(name="modid", description="Module ID") int modId,
                            @Argument(name="chan", description="Channel number") int chan) throws DriverException
    {
        return formatDoubles(new double[]{maq.getAnalog(modId).readValue(chan)});
    }

    @Command(description="Read all channels' data values")
    public String readValue(@Argument(name="modid", description="Module ID") int modId) throws DriverException
    {
        return formatDoubles(maq.getAnalog(modId).readValue());
    }

    @Command(description="Write data to channels")
    public void writeValue(@Argument(name="modid", description="Module ID") int modId,
                           @Argument(name="chan", description="First channel number") int chan,
                           @Argument(name="values", description="Values to write") double... values) throws DriverException
    {
        maq.getAnalOut(modId).writeValue(chan, values.length, values);
    }

    @Command(description="Set default data for channels")
    public void setDefaultValue(@Argument(name="modid", description="Module ID") int modId,
                                @Argument(name="chan", description="First channel number") int chan,
                                @Argument(name="values", description="Values to write") double... values) throws DriverException
    {
        maq.getAnalOut(modId).setDefaultValue(chan, values.length, values);
    }

    @Command(description="Show channels' default data values")
    public String showDefaultValue(@Argument(name="modid", description="Module ID") int modId,
                                   @Argument(name="chan", description="Channel number") int chan,
                                   @Argument(name="count", description="Number of channels") int count) throws DriverException
    {
        return formatDoubles(maq.getAnalOut(modId).getDefaultValue(chan, count));
    }

    @Command(description="Show a channel's default data value")
    public String showDefaultValue(@Argument(name="modid", description="Module ID") int modId,
                                   @Argument(name="chan", description="Channel number") int chan) throws DriverException
    {
        return formatDoubles(new double[]{maq.getAnalOut(modId).getDefaultValue(chan)});
    }

    @Command(description="Show all channels' default data values")
    public String showDefaultValue(@Argument(name="modid", description="Module ID") int modId) throws DriverException
    {
        return formatDoubles(maq.getAnalOut(modId).getDefaultValue());
    }

    @Command(description="Save the default data values")
    public void saveDefaultValues(@Argument(name="modid", description="Module ID") int modId) throws DriverException
    {
        maq.getAnalOut(modId).saveDefaultValues();
    }

    @Command(description="Save the output ranges")
    public void saveRanges(@Argument(name="modid", description="Module ID") int modId) throws DriverException
    {
        maq.getAnalOut(modId).saveRanges();
    }

    @Command(description="Read discrete input channels' values")
    public String readDiscIn(@Argument(name="modid", description="Module ID") int modId,
                             @Argument(name="chan", description="Channel number") int chan,
                             @Argument(name="count", description="Number of channels") int count) throws DriverException
    {
        return formatIntegers(maq.getDiscrete(modId).readDiscIn(chan, count));
    }

    @Command(description="Read a discrete input channel's value")
    public String readDiscIn(@Argument(name="modid", description="Module ID") int modId,
                             @Argument(name="chan", description="Channel number") int chan) throws DriverException
    {
        return formatIntegers(new int[]{maq.getDiscrete(modId).readDiscIn(chan)});
    }

    @Command(description="Read all discrete input channels' values")
    public String readDiscIn(@Argument(name="modid", description="Module ID") int modId) throws DriverException
    {
        return formatIntegers(maq.getDiscrete(modId).readDiscIn());
    }

    @Command(description="Read discrete output channels' values")
    public String readDiscOut(@Argument(name="modid", description="Module ID") int modId,
                              @Argument(name="chan", description="Channel number") int chan,
                              @Argument(name="count", description="Number of channels") int count) throws DriverException
    {
        return formatIntegers(maq.getDiscrete(modId).readDiscOut(chan, count));
    }

    @Command(description="Read a discrete output channel's value")
    public String readDiscOut(@Argument(name="modid", description="Module ID") int modId,
                              @Argument(name="chan", description="Channel number") int chan) throws DriverException
    {
        return formatIntegers(new int[]{maq.getDiscrete(modId).readDiscOut(chan)});
    }

    @Command(description="Read all discrete output channels' values")
    public String readDiscOut(@Argument(name="modid", description="Module ID") int modId) throws DriverException
    {
        return formatIntegers(maq.getDiscrete(modId).readDiscOut());
    }

    @Command(description="Read all discrete input channels' values")
    public String readDiscAll(@Argument(name="modid", description="Module ID") int modId) throws DriverException
    {
        return String.format("Values mask: %08x", maq.getDiscIn(modId).readDiscAll());
    }

    @Command(description="Write data to discrete channels")
    public void writeDiscOut(@Argument(name="modid", description="Module ID") int modId,
                             @Argument(name="chan", description="First channel number") int chan,
                             @Argument(name="values", description="Values to write") int... values) throws DriverException
    {
        maq.getDiscrete(modId).writeDisc(chan, values.length, values);
    }

    @Command(description="Write data to discrete channels")
    public void writeDiscAll(@Argument(name="modid", description="Module ID") int modId,
                             @Argument(name="value", description="Value to write") int value) throws DriverException
    {
        maq.getDiscOut(modId).writeDiscAll(value);
    }

    @Command(description="Set default discrete output values")
    public void setDiscDefault(@Argument(name="modid", description="Module ID") int modId,
                               @Argument(name="chan", description="First channel number") int chan,
                               @Argument(name="values", description="Values to write") int... values) throws DriverException
    {
        maq.getDiscrete(modId).setDiscDefault(chan, values.length, values);
    }

    @Command(description="Save the default discrete output values")
    public void saveDiscDefaults(@Argument(name="modid", description="Module ID") int modId) throws DriverException
    {
        maq.getDiscrete(modId).saveDiscDefaults();
    }

    @Command(description="Set all default discrete output values")
    public void setDiscDefaultAll(@Argument(name="modid", description="Module ID") int modId,
                                  @Argument(name="value", description="Value to write") int value) throws DriverException
    {
        maq.getDiscOut(modId).setDiscDefaultAll(value);
    }

    @Command(description="Show all default discrete output values")
    public String showDiscDefault(@Argument(name="modid", description="Module ID") int modId) throws DriverException
    {
        return formatIntegers(maq.getDiscrete(modId).getDiscDefault());
    }

    @Command(description="Show all default discrete output values")
    public String showDiscDefaultAll(@Argument(name="modid", description="Module ID") int modId) throws DriverException
    {
        return String.format("Values mask: %08x", maq.getDiscOut(modId).getDiscDefaultAll());
    }

    @Command(description="Set power-off discrete output values")
    public void setDiscOff(@Argument(name="modid", description="Module ID") int modId,
                           @Argument(name="chan", description="First channel number") int chan,
                           @Argument(name="values", description="Values to write") int... values) throws DriverException
    {
        maq.getDiscOut(modId).setDiscOff(chan, values.length, values);
    }

    @Command(description="Set all default discrete output values")
    public void setDiscOffAll(@Argument(name="modid", description="Module ID") int modId,
                              @Argument(name="value", description="Value to write") int value) throws DriverException
    {
        maq.getDiscOut(modId).setDiscOffAll(value);
    }

    @Command(description="Show all power-off discrete output values")
    public String showDiscOff(@Argument(name="modid", description="Module ID") int modId) throws DriverException
    {
        return formatIntegers(maq.getDiscOut(modId).getDiscOff());
    }

    @Command(description="Show all default discrete output values")
    public String showDiscOffAll(@Argument(name="modid", description="Module ID") int modId) throws DriverException
    {
        return String.format("Values mask: %08x", maq.getDiscOut(modId).getDiscOffAll());
    }

    @Command(description="Set discrete function code")
    public void setFunction(@Argument(name="modid", description="Module ID") int modId,
                            @Argument(name="chan", description="Function channel number (0 or 1)") int chan,
                            @Argument(name="func", description="Function name") DiscFunc func) throws DriverException
    {
        maq.getDiscPulse(modId).setFunction(chan, func.getValue());
    }

    @Command(description="Show discrete function code")
    public String showFunction(@Argument(name="modid", description="Module ID") int modId,
                               @Argument(name="chan", description="Function channel number (0 or 1)") int chan) throws DriverException
    {
        int func = maq.getDiscPulse(modId).getFunction(chan);
        return "Function: " + func + " (" + funcDescs.get(func) + ")";
    }

    @Command(description="Arm discrete function")
    public void armFunction(@Argument(name="modid", description="Module ID") int modId,
                            @Argument(name="chan", description="Function channel number (0 or 1)") int chan) throws DriverException
    {
        maq.getDiscPulse(modId).armFunction(chan);
    }

    @Command(description="Disarm discrete function")
    public void disarmFunction(@Argument(name="modid", description="Module ID") int modId,
                               @Argument(name="chan", description="Function channel number (0 or 1)") int chan) throws DriverException
    {
        maq.getDiscPulse(modId).disarmFunction(chan);
    }

    @Command(description="Read pulse count")
    public String readPulseCount(@Argument(name="modid", description="Module ID") int modId,
                                 @Argument(name="chan", description="Function channel number (0 or 1)") int chan) throws DriverException
    {
        return "Pulse count: " + maq.getDiscPulse(modId).readPulseCount(chan);
    }

    @Command(description="Read frequency")
    public String readFrequency(@Argument(name="modid", description="Module ID") int modId,
                                @Argument(name="chan", description="Function channel number (0 or 1)") int chan) throws DriverException
    {
        return "Frequency: " + maq.getDiscPulse(modId).readFrequency(chan);
    }

    @Command(description="Read RPM")
    public String readRPM(@Argument(name="modid", description="Module ID") int modId,
                          @Argument(name="chan", description="Function channel number (0 or 1)") int chan) throws DriverException
    {
        return "RPM: " + maq.getDiscPulse(modId).readRPM(chan);
    }

    @Command(description="Set pulses per revolution")
    public void setPulsesPerRevn(@Argument(name="modid", description="Module ID") int modId,
                                 @Argument(name="chan", description="Function channel number (0 or 1)") int chan,
                                 @Argument(name="ppr", description="Pulses/revolution") int ppr) throws DriverException
    {
        Maq20DiscretePulse maqDisc = maq.getDiscPulse(modId);
        maqDisc.setPulsesPerRevn(chan, ppr);
        maqDisc.armFunction(chan);
    }

    @Command(description="Show pulses per revolution")
    public String showPulsesPerRevn(@Argument(name="modid", description="Module ID") int modId,
                                    @Argument(name="chan", description="Function channel number (0 or 1)") int chan) throws DriverException
    {
        return "Pulses/revolution: " + maq.getDiscPulse(modId).getPulsesPerRevn(chan);
    }

    @Command(description="Set PWM timebase")
    public void setTimebase(@Argument(name="modid", description="Module ID") int modId,
                            @Argument(name="chan", description="Function channel number (0 or 1)") int chan,
                            @Argument(name="time", description="Timebase") Timebase timebase) throws DriverException
    {
        Maq20DiscretePWM maqDisc = maq.getDiscPWM(modId);
        maqDisc.setTimebase(chan, timebase.getValue());
        maqDisc.armFunction(chan);
    }

    @Command(description="Show PWM timebase")
    public String showTimebase(@Argument(name="modid", description="Module ID") int modId,
                               @Argument(name="chan", description="Function channel number (0 or 1)") int chan) throws DriverException
    {
        int timebase = maq.getDiscPWM(modId).getTimebase(chan);
        return "PWM timebase: " + timebase + " (" + timebaseDescs.get(timebase) +")";
    }

    @Command(description="Set PWM period")
    public void setPeriod(@Argument(name="modid", description="Module ID") int modId,
                          @Argument(name="chan", description="Function channel number (0 or 1)") int chan,
                          @Argument(name="period", description="PWM period") int period) throws DriverException
    {
        Maq20DiscretePWM maqDisc = maq.getDiscPWM(modId);
        maqDisc.setPeriod(chan, period);
        maqDisc.armFunction(chan);
    }

    @Command(description="Show PWM period")
    public String showPeriod(@Argument(name="modid", description="Module ID") int modId,
                             @Argument(name="chan", description="Function channel number (0 or 1)") int chan) throws DriverException
    {
        return "PWM period: " + maq.getDiscPWM(modId).getPeriod(chan);
    }

    @Command(description="Set output 1 low time")
    public void setLowTime1(@Argument(name="modid", description="Module ID") int modId,
                            @Argument(name="chan", description="Function channel number (0 or 1)") int chan,
                            @Argument(name="time", description="Time") int time) throws DriverException
    {
        Maq20DiscretePWM maqDisc = maq.getDiscPWM(modId);
        maqDisc.setLowTime1(chan, time);
        maqDisc.armFunction(chan);
    }

    @Command(description="Show PWM output 1 low time")
    public String showLowTime1(@Argument(name="modid", description="Module ID") int modId,
                               @Argument(name="chan", description="Function channel number (0 or 1)") int chan) throws DriverException
    {
        return "PWM low time 1: " + maq.getDiscPWM(modId).getLowTime1(chan);
    }

    @Command(description="Set output 2 low time")
    public void setLowTime2(@Argument(name="modid", description="Module ID") int modId,
                            @Argument(name="chan", description="Function channel number (0 or 1)") int chan,
                            @Argument(name="time", description="Time") int time) throws DriverException
    {
        Maq20DiscretePWM maqDisc = maq.getDiscPWM(modId);
        maqDisc.setLowTime2(chan, time);
        maqDisc.armFunction(chan);
    }

    @Command(description="Show PWM output 2 low time")
    public String showLowTime2(@Argument(name="modid", description="Module ID") int modId,
                               @Argument(name="chan", description="Function channel number (0 or 1)") int chan) throws DriverException
    {
        return "PWM low time 2: " + maq.getDiscPWM(modId).getLowTime2(chan);
    }

    @Command(description="Enable/disable output 2")
    public void enableOut2(@Argument(name="modid", description="Module ID") int modId,
                           @Argument(name="chan", description="Function channel number (0 or 1)") int chan,
                           @Argument(name="time", description="Time") boolean enable) throws DriverException
    {
        Maq20DiscretePWM maqDisc = maq.getDiscPWM(modId);
        maqDisc.enableOutput2(chan, enable);
        maqDisc.armFunction(chan);
    }

    @Command(description="Show whether PWM output 2 is enabled")
    public String isOut2Enabled(@Argument(name="modid", description="Module ID") int modId,
                                @Argument(name="chan", description="Function channel number (0 or 1)") int chan) throws DriverException
    {
        return "PWM output 2 is " + (maq.getDiscPWM(modId).isOutput2Enabled(chan) ? "" : "not ") + "enabled";
    }

    private String formatDoubles(double[] values)
    {
        StringBuilder text = new StringBuilder(values.length > 1 ? "Values:" : "Value:");
        for (double value : values) {
            text.append(String.format(" %.6g", value));
        }
        return text.toString();
    }

    private String formatIntegers(int[] values)
    {
        StringBuilder text = new StringBuilder(values.length > 1 ? "Values:" : "Value:");
        for (int value : values) {
            text.append(" ").append(value);
        }
        return text.toString();
    }

}
