package org.lsst.ccs.drivers.dataforth;

import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Control a Dataforth MAQ20 discrete I/O module pulse counter.
 *
 *  @author  Owen Saxton
 */
public class Maq20DiscretePulse extends Maq20DiscreteFunc {

    /**
     *  Private data.
     */
    private static final short
        PULSE_COUNT_ADDR = 4,
        FREQUENCY_ADDR = 6,
        RPM_ADDR = 8,
        PULSE_REVN_ADDR = 9,
        INT_TRIG_ADDR = 10,
        EXT_ENAB_ADDR = 11,
        EXT_ENAB_STAT_ADDR = 12;


    /**
     *  Constructor.
     *
     *  @param  maq    The underlying Maq20 object
     *  @param  modId  The module registration ID
     *  @throws DriverException
     */
    public Maq20DiscretePulse(Maq20 maq, int modId) throws DriverException
    {
        super(maq, modId);
    }


    /**
     *  Enables the pulse counter.
     *
     *  @param  chan  The function channel (0 or 1)
     *  @throws  DriverException
     */
    public void enable(int chan) throws DriverException 
    {
        setFunction(chan, FUNC_PULSE);
    }


    /**
     *  Gets whether the pulse counter is enabled.
     *
     *  @param  chan  The function channel (0 or 1)
     *  @return  Whether anabled
     *  @throws  DriverException
     */
    public boolean isEnabled(int chan) throws DriverException 
    {
        return getFunction(chan) == FUNC_PULSE;
    }


    /**
     *  Reads the pulse count
     *
     *  @param  chan  The function channel (0 or 1)
     *  @return  The pulse count
     *  @throws  DriverException
     */
    public int readPulseCount(int chan) throws DriverException 
    {
        return maq.readRegisterPair(getRegAddr(chan, FUNC_PULSE, PULSE_COUNT_ADDR));
    }


    /**
     *  Resets the pulse count
     *
     *  @param  chan  The function channel (0 or 1)
     *  @throws  DriverException
     */
    public void resetPulseCount(int chan) throws DriverException 
    {
        maq.writeRegister(getRegAddr(chan, FUNC_PULSE, PULSE_COUNT_ADDR), Maq20.ZERO);
    }


    /**
     *  Reads the frequency.
     *
     *  @param  chan  The function channel (0 or 1)
     *  @return  The pulse count
     *  @throws  DriverException
     */
    public int readFrequency(int chan) throws DriverException 
    {
        return maq.readRegisterPair(getRegAddr(chan, FUNC_PULSE, FREQUENCY_ADDR));
    }


    /**
     *  Reads the RPM.
     *
     *  @param  chan  The function channel (0 or 1)
     *  @return  The pulse count
     *  @throws  DriverException
     */
    public int readRPM(int chan) throws DriverException 
    {
        return maq.readRegister(getRegAddr(chan, FUNC_PULSE, RPM_ADDR)) & 0xffff;
    }


    /**
     *  Sets the pulses per revolution.
     *
     *  @param  chan   The function channel (0 or 1)
     *  @param  count  The number of pulses per revolution
     *  @throws  DriverException
     */
    public void setPulsesPerRevn(int chan, int count) throws DriverException 
    {
        maq.writeRegister(getRegAddr(chan, FUNC_PULSE, PULSE_REVN_ADDR), (short)count);
    }


    /**
     *  Gets the pulses per revolution.
     *
     *  @param  chan   The function channel (0 or 1)
     *  @return  The number of pulses per revolution
     *  @throws  DriverException
     */
    public int getPulsesPerRevn(int chan) throws DriverException 
    {
        return maq.readRegister(getRegAddr(chan, FUNC_PULSE, PULSE_REVN_ADDR)) & 0xffff;
    }

}
