package org.lsst.ccs.drivers.dataforth;

import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Routines for controlling a Dataforth MAQ20 discrete output module.
 *
 *  @author  Owen Saxton
 */
public class Maq20DiscreteOut extends Maq20DiscreteIn {

    static final int
        DFLT_DATA_ALL_ADDR = 108,
        OFF_DATA_ALL_ADDR = 148,
        OFF_DATA_ADDR = 150,
        WRITE_ALL_ADDR = 1038;


    /**
     *  Constructor.
     *
     *  @param  maq    The underlying Maq20 object
     *  @param  modId  The module registration ID
     *  @throws DriverException
     */
    public Maq20DiscreteOut(Maq20 maq, int modId) throws DriverException
    {
        super(maq, modId);
    }


    /**
     *  Reads a set of output channels.
     *
     *  Overrides superclass to set inverted logic
     *
     *  @param  chan   The first channel number
     *  @param  count  The number of channels
     *  @return  The array of data values
     *  @throws  DriverException
     */
    @Override
    public int[] readDiscOut(int chan, int count) throws DriverException
    {
        setStraightLogic();
        return readDiscData(CHAN_DATA_ADDR, chan, count, false);
    }


    /**
     *  Writes a set of output channels.
     *
     *  Overrides superclass to set inverted logic
     *
     *  @param  chan   The first channel number
     *  @param  count  The number of channels
     *  @param  data   The array of data values (at least count elements long)
     *  @throws  DriverException
     */
    @Override
    public void writeDisc(int chan, int count, int[] data) throws DriverException
    {
        checkOutputChannels(chan, count);
        setStraightLogic();
        writeDiscData(CHAN_DATA_ADDR, chan, count, data, false);
    }


    /**
     *  Writes all channels.
     *
     *  @param  value  A bit mask of channels to activate
     *  @throws  DriverException
     */
    public void writeDiscAll(int value) throws DriverException
    {
        setStraightLogic();
        maq.writeRegisterPair((short)(module.baseAddr + WRITE_ALL_ADDR), value);
    }


    /**
     *  Sets several discrete output channel default values.
     *
     *  @param  chan   The first channel number
     *  @param  count  The number of channels
     *  @param  data   The array of data values (at least count elements long)
     *  @throws  DriverException
     */
    @Override
    public void setDiscDefault(int chan, int count, int[] data) throws DriverException
    {
        checkOutputChannels(chan, count);
        setStraightLogic();
        writeDiscData(DFLT_DATA_ADDR, chan, count, data, false);
        saveDiscDefaults();
    }


    /**
     *  Gets several discrete output channel default values.
     *
     *  @param  chan   The first channel number
     *  @param  count  The number of channels
     *  @return  The array of data values
     *  @throws  DriverException
     */
    @Override
    public int[] getDiscDefault(int chan, int count) throws DriverException
    {
        checkOutputChannels(chan, count);
        setStraightLogic();
        return readDiscData(DFLT_DATA_ADDR, chan, count, false);
    }


    /**
     *  Sets several discrete output channel off values.
     *
     *  @param  chan   The first channel number
     *  @param  count  The number of channels
     *  @param  data   The array of data values (at least count elements long)
     *  @throws  DriverException
     */
    public void setDiscOff(int chan, int count, int[] data) throws DriverException
    {
        checkOutputChannels(chan, count);
        setStraightLogic();
        writeDiscData(OFF_DATA_ADDR, chan, count, data, false);
        saveDiscDefaults();
    }


    /**
     *  Gets several discrete output channel off values.
     *
     *  @param  chan   The first channel number
     *  @param  count  The number of channels
     *  @return  The array of data values
     *  @throws  DriverException
     */
    public int[] getDiscOff(int chan, int count) throws DriverException
    {
        checkOutputChannels(chan, count);
        setStraightLogic();
        return readDiscData(OFF_DATA_ADDR, chan, count, false);
    }


    /**
     *  Gets a discrete output channel off value.
     *
     *  @param  chan   The channel number
     *  @return  The array of data values
     *  @throws  DriverException
     */
    public int getDiscOff(int chan) throws DriverException
    {
        return getDiscOff(chan, 1)[0];
    }


    /**
     *  Gets all discrete output channel off values.
     *
     *  @return  The array of data values
     *  @throws  DriverException
     */
    public int[] getDiscOff() throws DriverException
    {
        return getDiscOff(0, module.numOutChan);
    }


    /**
     *  Sets all channel defaults.
     *
     *  @param  value  A bit mask of channels to activate
     *  @throws  DriverException
     */
    public void setDiscDefaultAll(int value) throws DriverException
    {
        setStraightLogic();
        maq.writeRegisterPair((short)(module.baseAddr + DFLT_DATA_ALL_ADDR), value);
        saveDiscDefaults();
    }


    /**
     *  Gets all channel defaults.
     *
     *  @return  A bit mask of active channels
     *  @throws  DriverException
     */
    public int getDiscDefaultAll() throws DriverException
    {
        return maq.readRegisterPair((short)(module.baseAddr + DFLT_DATA_ALL_ADDR));
    }


    /**
     *  Sets all channel power off values.
     *
     *  @param  value  A bit mask of channels to activate
     *  @throws  DriverException
     */
    public void setDiscOffAll(int value) throws DriverException
    {
        setStraightLogic();
        maq.writeRegisterPair((short)(module.baseAddr + OFF_DATA_ALL_ADDR), value);
        saveDiscDefaults();
    }


    /**
     *  Gets all channel power off values.
     *
     *  @return  A bit mask of active channels
     *  @throws  DriverException
     */
    public int getDiscOffAll() throws DriverException
    {
        return maq.readRegisterPair((short)(module.baseAddr + OFF_DATA_ALL_ADDR));
    }

}
