package org.lsst.ccs.drivers.dataforth;

import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Routines for controlling a Dataforth MAQ20 analog output module.
 *
 *  @author  Owen Saxton
 */
public class Maq20AnalogOut extends Maq20Analog {

    /**
     *  Private data.
     */
    private static final short
        DFLT_DATA_ADDR = 110,
        SAVE_PARAMS_ADDR = 119;


    /**
     *  Constructor.
     *
     *  @param  maq    The underlying Maq20 object
     *  @param  modId  The module registration ID
     *  @throws DriverException
     */
    public Maq20AnalogOut(Maq20 maq, int modId) throws DriverException
    {
        super(maq, modId);
        if (module.opType != Maq20.OPER_ANALOUT) {
            throw new DriverException("Invalid module type (" + module.type + ") for analog output operations");
        }
    }


    /**
     *  Writes data values to several channels.
     *
     *  @param  chan   The first channel number
     *  @param  count  The number of channels
     *  @param  data   The array of data values (at least count elements long)
     *  @throws  DriverException
     */
    public void writeValue(int chan, int count, double[] data) throws DriverException
    {
        writeAnalogData(CHAN_DATA_ADDR, chan, count, data);
    }


    /**
     *  Writes a data value to a channel.
     *
     *  @param  chan   The channel number
     *  @param  data   The data value
     *  @throws  DriverException
     */
    public void writeValue(int chan, double data) throws DriverException
    {
        writeValue(chan, 1, new double[]{data});
    }


    /**
     *  Writes data values to all channels.
     *
     *  @param  data   The array of data values
     *  @throws  DriverException
     */
    public void writeValue(double[] data) throws DriverException
    {
        writeValue(0, module.numOutChan, data);
    }


    /**
     *  Sets default data values for several channels.
     *
     *  @param  chan   The first channel number
     *  @param  count  The number of channels
     *  @param  data   The array of data values (at least count elements long)
     *  @throws  DriverException
     */
    public void setDefaultValue(int chan, int count, double[] data) throws DriverException
    {
        writeAnalogData(DFLT_DATA_ADDR, chan, count, data);
        saveDefaultValues();
    }


    /**
     *  Sets default data value for a channel.
     *
     *  @param  chan   The channel number
     *  @param  data   The data value
     *  @throws  DriverException
     */
    public void setDefaultValue(int chan, double data) throws DriverException
    {
        setDefaultValue(chan, 1, new double[]{data});
    }


    /**
     *  Sets default data values for all channels.
     *
     *  @param  data   The array of data values
     *  @throws  DriverException
     */
    public void setDefaultValue(double[] data) throws DriverException
    {
        setDefaultValue(0, module.numOutChan, data);
    }


    /**
     *  Gets the default data values for several channels.
     *
     *  @param  chan   The first channel number
     *  @param  count  The number of channels
     *  @return  The array of data values
     *  @throws  DriverException
     */
    public double[] getDefaultValue(int chan, int count) throws DriverException
    {
        return readAnalogData(DFLT_DATA_ADDR, chan, count);
    }


    /**
     *  Gets the default data value for a channel.
     *
     *  @param  chan   The channel number
     *  @return  The data value
     *  @throws  DriverException
     */
    public double getDefaultValue(int chan) throws DriverException
    {
        return getDefaultValue(chan, 1)[0];
    }


    /**
     *  Gets the default data values for all channels.
     *
     *  @return  The array of data values
     *  @throws  DriverException
     */
    public double[] getDefaultValue() throws DriverException
    {
        return getDefaultValue(0, -1);
    }


    /**
     *  Saves the default output values to EEPROM
     * 
     *  @throws DriverException 
     */
    public void saveDefaultValues() throws DriverException
    {
        maq.writeRegister((short)(module.baseAddr + SAVE_PARAMS_ADDR), Maq20.ONE);
    }


    /**
     *  Saves the ranges to EEPROM
     * 
     *  @throws DriverException 
     */
    public void saveRanges() throws DriverException
    {
        maq.writeRegister((short)(module.baseAddr + SAVE_PARAMS_ADDR), Maq20.ZERO);
    }


    /**
     *  Writes data values to several locations.
     *
     *  @param  base   The base register address
     *  @param  chan   The first channel number
     *  @param  count  The number of channels
     *  @param  data   The array of data values (at least count elements long)
     *  @throws  DriverException
     */
    void writeAnalogData(int base, int chan, int count, double[] data) throws DriverException
    {
        checkChannel(chan, count);
        if (count == 0) return;
        short[]raw = new short[count];
        for (int j = 0; j < count; j++) {
            int range = module.range[chan + j];
            raw[j] = (short)((data[j] + module.offset[range]) / module.scale[range]);
        }
        maq.writeRegisters((short)(module.baseAddr + base + chan), raw);
    }

}
