package org.lsst.ccs.drivers.dataforth;

import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Routines for controlling a Dataforth MAQ20 discrete I/O module.
 *
 *  @author  Owen Saxton
 */
public class Maq20Discrete {

    /**
     *  Package data.
     */
    final Maq20 maq;
    final Maq20.ModuleData module;

    /**
     *  Private data.
     */
    static final short
        DISCR_CONFIG_ADDR = 100,
        DFLT_DATA_ADDR = 110,
        SAVE_PARAMS_ADDR = 190,
        CHAN_DATA_ADDR = 1000;


    /**
     *  Constructor.
     *
     *  @param  maq    The underlying Maq20 object
     *  @param  modId  The module registration ID
     *  @throws DriverException
     */
    public Maq20Discrete(Maq20 maq, int modId) throws DriverException
    {
        this.maq = maq;
        module = maq.getModuleData(modId, false);
        if (module.opType != Maq20.OPER_DISCRETE && module.opType != Maq20.OPER_DISCIN && module.opType != Maq20.OPER_DISCOUT) {
            throw new DriverException("Invalid module type (" + module.type + ") for discrete operations");
        }
    }


    /**
     *  Reads several discrete input channels.
     *
     *  @param  chan   The first channel number
     *  @param  count  The number of channels
     *  @return  The array of data values
     *  @throws  DriverException
     */
    public int[] readDiscIn(int chan, int count) throws DriverException
    {
        checkInputChannels(chan, count);
        return readDiscData(CHAN_DATA_ADDR + module.numOutChan, chan, count, true);
    }


    /**
     *  Reads a discrete input channel.
     *
     *  @param  chan   The channel number
     *  @return  The data value
     *  @throws  DriverException
     */
    public int readDiscIn(int chan) throws DriverException
    {
        return readDiscIn(chan, 1)[0];
    }


    /**
     *  Reads all discrete input channels.
     *
     *  @return  The array of data values
     *  @throws  DriverException
     */
    public int[] readDiscIn() throws DriverException
    {
        return readDiscIn(0, module.numInChan);
    }


    /**
     *  Reads several discrete output channels.
     *
     *  @param  chan   The first channel number
     *  @param  count  The number of channels
     *  @return  The array of data values
     *  @throws  DriverException
     */
    public int[] readDiscOut(int chan, int count) throws DriverException
    {
        checkOutputChannels(chan, count);
        return readDiscData(CHAN_DATA_ADDR, chan, count, true);
    }


    /**
     *  Reads a discrete output channel.
     *
     *  @param  chan   The channel number
     *  @return  The data value
     *  @throws  DriverException
     */
    public int readDiscOut(int chan) throws DriverException
    {
        return readDiscOut(chan, 1)[0];
    }


    /**
     *  Reads all discrete output channels.
     *
     *  @return  The array of data values
     *  @throws  DriverException
     */
    public int[] readDiscOut() throws DriverException
    {
        return readDiscOut(0, module.numOutChan);
    }


    /**
     *  Writes to several discrete output channels.
     *
     *  @param  chan   The first channel number
     *  @param  count  The number of channels
     *  @param  data   The array of data values (at least count elements long)
     *  @throws  DriverException
     */
    public void writeDisc(int chan, int count, int[] data) throws DriverException
    {
        checkOutputChannels(chan, count);
        writeDiscData(CHAN_DATA_ADDR, chan, count, data, true);
    }


    /**
     *  Writes to a discrete output channel.
     *
     *  @param  chan   The channel number
     *  @param  data   The data value
     *  @throws  DriverException
     */
    public void writeDisc(int chan, int data) throws DriverException
    {
        writeDisc(chan, 1, new int[]{data});
    }


    /**
     *  Writes to all discrete output channels.
     *
     *  @param  data   The array of data values
     *  @throws  DriverException
     */
    public void writeDisc(int[] data) throws DriverException
    {
        writeDisc(0, module.numOutChan, data);
    }


    /**
     *  Sets several discrete output channel default values.
     *
     *  @param  chan   The first channel number
     *  @param  count  The number of channels
     *  @param  data   The array of data values (at least count elements long)
     *  @throws  DriverException
     */
    public void setDiscDefault(int chan, int count, int[] data) throws DriverException
    {
        checkOutputChannels(chan, count);
        writeDiscData(DFLT_DATA_ADDR, chan, count, data, true);
        saveParameters();
    }


    /**
     *  Sets a discrete output channel default value.
     *
     *  @param  chan   The channel number
     *  @param  data   The data value
     *  @throws  DriverException
     */
    public void setDiscDefault(int chan, int data) throws DriverException
    {
        setDiscDefault(chan, 1, new int[]{data});
    }


    /**
     *  Sets all discrete output channel default values.
     *
     *  @param  data   The array of data values
     *  @throws  DriverException
     */
    public void setDiscDefault(int[] data) throws DriverException
    {
        setDiscDefault(0, module.numOutChan, data);
    }


    /**
     *  Gets several discrete output channel default values.
     *
     *  @param  chan   The first channel number
     *  @param  count  The number of channels
     *  @return  The array of data values
     *  @throws  DriverException
     */
    public int[] getDiscDefault(int chan, int count) throws DriverException
    {
        checkOutputChannels(chan, count);
        return readDiscData(DFLT_DATA_ADDR, chan, count, true);
    }


    /**
     *  Gets a discrete output channel default value.
     *
     *  @param  chan   The channel number
     *  @return  The data value
     *  @throws  DriverException
     */
    public int getDiscDefault(int chan) throws DriverException
    {
        return getDiscDefault(chan, 1)[0];
    }


    /**
     *  Gets all discrete output channel default values.
     *
     *  @return  The array of data values
     *  @throws  DriverException
     */
    public int[] getDiscDefault() throws DriverException
    {
        return getDiscDefault(0, module.numOutChan);
    }


    /**
     *  Saves the parameters.
     *
     *  @throws  DriverException
     */
    public void saveParameters() throws DriverException
    {
        short addr = (short)(module.baseAddr + SAVE_PARAMS_ADDR);
        maq.writeRegister(addr, Maq20.ZERO);
    }


    /**
     *  Reads several discrete registers.
     *
     *  Used to read both actual and default values
     *
     *  @param  base   The base register to read from
     *  @param  chan   The first channel number
     *  @param  count  The number of channels
     *  @param  invert  Whether to invert the values
     *  @return  The array of data values
     *  @throws  DriverException
     */
    int[] readDiscData(int base, int chan, int count, boolean invert) throws DriverException
    {
        if (count == 0) return new int[0];
        short addr = (short)(module.baseAddr + base + chan);
        short[] raw = maq.readRegisters(addr, (short)count);
        int[] data = new int[count];
        int invMask = invert ? 1 : 0;
        for (int j = 0; j < data.length; j++) {
            data[j] = raw[j] ^ invMask;
        }
        return data;
    }


    /**
     *  Writes to several discrete registers.
     *
     *  Used to write both actual and default values
     *
     *  @param  base   The base register to write to
     *  @param  chan   The first channel number
     *  @param  count  The number of channels
     *  @param  data   The array of data values (at least count elements long)
     *  @param  invert  Whether to invert the values
     *  @throws  DriverException
     */
    void writeDiscData(int base, int chan, int count, int[] data, boolean invert) throws DriverException
    {
        if (count == 0) return;
        short[]raw = new short[count];
        int invMask = invert ? 1 : 0;
        for (int j = 0; j < data.length; j++) {
            raw[j] = (short)((data[j] == 0 ? 0 : 1) ^ invMask);
        }
        maq.writeRegisters((short)(module.baseAddr + base + chan), raw);
    }


    /**
     *  Checks a range of input channel numbers for validity.
     *
     *  @param  chan    The first channel number
     *  @param  count   The number of channels
     *  @throws  DriverException
     */
    void checkInputChannels(int chan, int count) throws DriverException
    {
        if (chan < 0 || chan >= module.numInChan) {
            throw new DriverException("Invalid input channel number");
        }
        if (count < 0 || chan + count > module.numInChan) {
            throw new DriverException("Invalid input channel count");
        }
    }


    /**
     *  Checks a range of output channel numbers for validity.
     *
     *  @param  chan    The first channel number
     *  @param  count   The number of channels
     *  @throws  DriverException
     */
    void checkOutputChannels(int chan, int count) throws DriverException
    {
        if (chan < 0 || chan >= module.numOutChan) {
            throw new DriverException("Invalid output channel number");
        }
        if (count < 0 || chan + count > module.numOutChan) {
            throw new DriverException("Invalid output channel count");
        }
    }

}
