package org.lsst.ccs.drivers.dataforth;

import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Routines for controlling a Dataforth MAQ20 analog input or output module.
 *
 *  @author  Owen Saxton
 */
public class Maq20Analog {

    public static final int
        RANGE_TC_T_400   = 0,
        RANGE_TC_T_220   = 1,
        RANGE_TC_J_760   = 0,
        RANGE_TC_J_393   = 1,
        RANGE_TC_J_199   = 2,
        RANGE_TC_K_1350  = 0,
        RANGE_TC_K_651   = 1,
        RANGE_TC_K_332   = 2,
        RANGE_TC_R_1750  = 0,
        RANGE_TC_R_990   = 1,
        RANGE_TC_S_1750  = 2,
        RANGE_TC_S_970   = 3,
        RANGE_RTD_850    = 0,
        RANGE_RTD_200    = 1,
        RANGE_RTD_100    = 2,
        RANGE_VOLT_60    = 0,
        RANGE_VOLT_40    = 1,
        RANGE_VOLT_20    = 2,
        RANGE_VOLT_10    = 3,
        RANGE_VOLT_5     = 4,
        RANGE_MVOLT_2000 = 0,
        RANGE_MVOLT_1000 = 1,
        RANGE_MVOLT_250  = 2,
        RANGE_MVOLT_100  = 3,
        RANGE_MVOLT_50   = 4,
        RANGE_MAMP_0_20  = 0,
        RANGE_MAMP_4_20  = 1,
        RANGE_VOUT_PM10  = 0,
        RANGE_VOUT_PM5   = 1,
        RANGE_VOUT_PM2   = 2,
        RANGE_VOUT_P10   = 3,
        RANGE_VOUT_P5    = 4,
        RANGE_VOUT_P2    = 5;

    /**
     *  Package data.
     */
    static final short
        CHAN_WEIGHT_ADDR = 120,
        CHAN_ENABLE_ADDR = 140,
        CHAN_DATA_ADDR = 1000;

    final Maq20 maq;
    final Maq20.ModuleData module;

    /**
     *  Private data.
     */
    private static final short
        INVALID_LIMIT = 6000;


    /**
     *  Constructor.
     *
     *  @param  maq    The underlying Maq20 object
     *  @param  modId  The module registration ID
     *  @throws DriverException
     */
    public Maq20Analog(Maq20 maq, int modId) throws DriverException
    {
        this.maq = maq;
        module = maq.getModuleData(modId, false);
        if (module.opType != Maq20.OPER_ANALOG && module.opType != Maq20.OPER_ANALOUT) {
            throw new DriverException("Invalid module type (" + module.type + ") for analog operations");
        }
    }


    /**
     *  Gets the number of ranges.
     *
     *  @return  The number of ranges
     */
    public int getNumRanges()
    {
        return module.offset.length;
    }


    /**
     *  Gets the range for a channel.
     *
     *  @param  chan   The channel number
     *  @return  The range number
     *  @throws  DriverException
     */
    public int getRange(int chan) throws DriverException
    {
        checkChannel(chan);
        return module.range[chan];
    }


    /**
     *  Sets the range for a channel.
     *
     *  @param  chan   The channel number
     *  @param  range  The range number
     *  @throws  DriverException
     */
    public void setRange(int chan, int range) throws DriverException
    {
        checkChannel(chan);
        if (range < 0 || range >= module.offset.length) {
            throw new DriverException("Invalid range value");
        }
        maq.writeRegister((short)(module.baseAddr + Maq20.RANGE_ADDR + chan), (short)range);
        module.range[chan] = range;
    }


    /**
     *  Reads the data values from several channels.
     *
     *  @param  chan   The first channel number
     *  @param  count  The number of channels
     *  @return  The array of data values
     *  @throws  DriverException
     */
    public double[] readValue(int chan, int count) throws DriverException
    {
        int nChan = count >= 0 ? count : module.numOutChan + module.numInChan - chan;
        checkChannel(chan, nChan);
        if (nChan == 0) return new double[0];
        short[] raw = maq.readRegisters((short)(module.baseAddr + CHAN_DATA_ADDR + chan), (short)nChan);
        double[] data = new double[nChan];
        for (int j = 0; j < nChan; j++, chan++) {
            if (raw[j] <= -INVALID_LIMIT || raw[j] >= INVALID_LIMIT) {
                data[j] = Double.NaN;
            }
            else {
                int range = module.range[chan];
                data[j] = module.scale[range] * raw[j] - module.offset[range];
            }
        }
        return data;
    }


    /**
     *  Reads the data value from a channel.
     *
     *  @param  chan   The channel number
     *  @return  The data value
     *  @throws  DriverException
     */
    public double readValue(int chan) throws DriverException
    {
        return readValue(chan, 1)[0];
    }


    /**
     *  Reads the data values from all channels.
     *
     *  @return  The array of data values
     *  @throws  DriverException
     */
    public double[] readValue() throws DriverException
    {
        return readValue(0, -1);
    }


    /**
     *  Checks a channel number for validity.
     *
     *  @param  chan    The channel number
     *  @throws  DriverException
     */
    void checkChannel(int chan) throws DriverException
    {
        checkChannel(chan, 1);
    }


    /**
     *  Checks a range of channel numbers for validity.
     *
     *  @param  chan    The first channel number
     *  @param  count   The number of channels
     *  @throws  DriverException
     */
    void checkChannel(int chan, int count) throws DriverException
    {
        int numChan = module.numOutChan + module.numInChan;
        if (chan < 0 || chan >= numChan) {
            throw new DriverException("Invalid channel number");
        }
        if (count < 0 || chan + count > numChan) {
            throw new DriverException("Invalid channel count");
        }
    }

}
