package org.lsst.ccs.drivers.dataforth;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.modbus.TestModbus;
import java.util.HashMap;
import java.util.Map;

/**
 *  Program to test the Dataforth MAQ20 device driver
 * 
 *  @author Owen Saxton
 */
public class TestMaq20 extends TestModbus {

    /**
     *  Inner class for holding access objects
     */
    static class Access {

        Maq20Analog maqAna;
        Maq20AnalogOut maqAnaOut;
        Maq20Discrete maqDisc;
        Maq20DiscreteFreq maqDiscFreq;
        Maq20DiscretePulse maqDiscPulse;
        Maq20DiscretePWM maqDiscPWM;

    }

    /**
     *  Private data
     */
    public enum DiscFunc {

        PULSE(Maq20DiscreteFunc.FUNC_PULSE),
        PWMGEN(Maq20DiscreteFunc.FUNC_PWM_GEN),
        FREQGEN(Maq20DiscreteFunc.FUNC_FREQ_GEN);

        int value;

        DiscFunc(int func) {
            this.value = func;
        }

        int getValue() {
            return value;
        }

    }

    public enum Timebase {

        SECS(Maq20DiscretePWM.TIMEBASE_SECS),
        MSECS(Maq20DiscretePWM.TIMEBASE_MSECS),
        USECS(Maq20DiscretePWM.TIMEBASE_USECS);

        int value;

        Timebase(int func) {
            this.value = func;
        }

        int getValue() {
            return value;
        }

    }

    private static final Map<Integer, String> funcDescs = new HashMap<>();
    static {
        funcDescs.put(Maq20DiscreteFunc.FUNC_NONE, "None");
        funcDescs.put(Maq20DiscreteFunc.FUNC_PULSE, "Pulse counter");
        funcDescs.put(Maq20DiscreteFunc.FUNC_PULSE_DEB, "Debounced pulse counter");
        funcDescs.put(Maq20DiscreteFunc.FUNC_WAVEFORM, "Waveform measurement");
        funcDescs.put(Maq20DiscreteFunc.FUNC_EVENT_TIME, "Time between events");
        funcDescs.put(Maq20DiscreteFunc.FUNC_FREQ_GEN, "Frequency generator");
        funcDescs.put(Maq20DiscreteFunc.FUNC_PWM_GEN, "PWM generator");
        funcDescs.put(Maq20DiscreteFunc.FUNC_PULSE_GEN, "One-shot pulse generator");
    }
    private static final Map<Integer, String> timebaseDescs = new HashMap<>();
    static {
        timebaseDescs.put((int)Maq20DiscreteFunc.TIMEBASE_SECS, "seconds");
        timebaseDescs.put((int)Maq20DiscreteFunc.TIMEBASE_MSECS, "milliseconds");
        timebaseDescs.put((int)Maq20DiscreteFunc.TIMEBASE_USECS, "microseconds");
    }
    private final Maq20 maq;
    private final Access[] access = new Access[Maq20.NUM_MODULES];


    /**
     *  Constructor
     */
    public TestMaq20()
    {
        super(new Maq20());
        maq = (Maq20)mod;
    }


    /**
     *  Opens connection to a device.
     *
     *  @param  type   The connection type
     *  @param  ident  The device identification
     *  @throws  DriverException
     */
    @Override
    @Command(name="open", description="Open connection to general device")
    public void open(@Argument(name="type", description="Connection type")
                     Maq20.ConnType type,
                     @Argument(name="ident", description="Device identifier")
                     String ident) throws DriverException
    {
        super.open(type, ident);
        setModuleAccess();
    }


    /**
     *  Opens connection to a device.
     *
     *  @param  type   The connection type
     *  @param  ident  The device identification
     *  @param  param  The device parameter
     *  @throws  DriverException
     */
    @Override
    @Command(name="open", description="Open connection to general device")
    public void open(@Argument(name="type", description="Connection type")
                     Maq20.ConnType type,
                     @Argument(name="ident", description="Device identifier")
                     String ident,
                     @Argument(name="param", description="Device parameter")
                     int param) throws DriverException
    {
        super.open(type, ident, param);
        setModuleAccess();
    }


    /**
     *  Opens connection to a serially-connected device.
     * 
     *  @param  type    The connection type
     *  @param  ident   The device identification
     *  @param  baud    The baud rate
     *  @param  dbits   The number of data bits (eight or seven)
     *  @param  sbits   The number of stop bits (one or two)
     *  @param  parity  The parity (none, odd, even, mark or space)
     *  @param  flow    The flow control (none, xon, rts or dtr)
     *  @throws  DriverException
     */
    @Override
    @Command(name="open", description="Open connection to serial device")
    public void open(@Argument(name="type", description="Connection type: net, serial or ftdi")
                     Maq20.ConnType type,
                     @Argument(name="ident", description="Device identifier")
                     String ident,
                     @Argument(name="baud", description="Baud rate")
                     int baud,
                     @Argument(name="dbits", description="The number of data bits")
                     Maq20.DataBits dbits,
                     @Argument(name="sbits", description="The number of stop bits")
                     Maq20.StopBits sbits,
                     @Argument(name="parity", description="The parity")
                     Maq20.Parity parity,
                     @Argument(name="flow", description="The flow control")
                     Maq20.FlowCtrl flow) throws DriverException
    {
        super.open(type, ident, baud, dbits, sbits, parity, flow);
        setModuleAccess();
    }


    /**
     *  Set up suitable objects for module access
     */
    private void setModuleAccess() throws DriverException
    {
        for (int modId = 1; modId < Maq20.NUM_MODULES; modId++) {
            access[modId] = null;
        }
        int[] modIds = maq.getModuleIds();
        for (int modId : modIds) {
            Access acc = access[modId] = new Access();
            switch (maq.getModuleOpType(modId)) {

            case Maq20.OPER_ANALOG:
                acc.maqAna = new Maq20Analog(maq, modId);
                break;

            case Maq20.OPER_ANALOUT:
                acc.maqAna = acc.maqAnaOut = new Maq20AnalogOut(maq, modId);
                break;

            case Maq20.OPER_DISCRETE:
                acc.maqDisc = new Maq20Discrete(maq, modId);
                acc.maqDiscPulse = new Maq20DiscretePulse(maq, modId);
                acc.maqDiscPWM = new Maq20DiscretePWM(maq, modId);
                acc.maqDiscFreq = new Maq20DiscreteFreq(maq, modId);
                break;
            }
        }
    }


    /**
     *  Registers a set of modules.
     * 
     *  @param  serial  The serial numbers of the modules to be registered
     *  @throws  DriverException
     */
    @Command(name="register", description="Register modules")
    public void register(@Argument(name="serial", description="Serial numbers")
                         String... serial) throws DriverException
    {
        maq.register(serial);
    }


    /**
     *  Shows the IP address.
     *
     *  @return The result string
     *  @throws  DriverException
     */
    @Command(name="showipaddress", description="Show the IP address")
    public String showIPAddress() throws DriverException
    {
        return "IP address: " + maq.getIPAddress();
    }


    /**
     *  Sets the IP address.
     *
     *  @param  ipAddr  The IP address string
     *  @throws  DriverException
     */
    @Command(name="setipaddress", description="Set the IP address")
    public void setIPAddress(@Argument(name="ipaddr", description="IP address")
                             String ipAddr) throws DriverException
    {
        maq.setIPAddress(ipAddr);
    }


    /**
     *  Shows the subnet mask.
     * 
     *  @return The result string
     *  @throws  DriverException
     */
    @Command(name="showsubnet", description="Show the subnet mask")
    public String showSubnetMask() throws DriverException
    {
        return "Subnet mask: " + maq.getSubnetMask();
    }


    /**
     *  Sets the subnet mask.
     *
     *  @param  mask  The subnet mask
     *  @throws  DriverException
     */
    @Command(name="setsubnet", description="Set the subnet mask")
    public void setSubnetMask(@Argument(name="mask", description="subnet mask")
                              String mask) throws DriverException
    {
        maq.setSubnetMask(mask);
    }


    /**
     *  Shows the temperature.
     * 
     *  @return The result string
     *  @throws  DriverException
     */
    @Command(name="showtemperature", description="Show the temperature")
    public String showTemperature() throws DriverException
    {
        return "Temperature: " + maq.readTemperature();
    }


    /**
     *  Shows all valid module IDs.
     * 
     *  @return The result string
     *  @throws  DriverException
     */
    @Command(name="showids", description="Show all module IDs")
    public String showIds() throws DriverException
    {
        int[] ids = maq.getModuleIds();
        StringBuilder text = new StringBuilder("Valid module IDs:");
        for (int id : ids) {
            text.append(" ").append(id);
        }
        return text.toString();
    }


    /**
     *  Shows a module's ID given it's serial number.
     * 
     *  @param  serial  The module's serial number
     *  @return The result string
     *  @throws  DriverException
     */
    @Command(name="showid", description="Show module's ID")
    public String showId(@Argument(name="serial", description="Serial number")
                         String serial) throws DriverException
    {
        return "Module ID: " + maq.getModuleId(serial);
    }


    /**
     *  Shows a module's name.
     * 
     *  @param  modId  The module ID
     *  @return The result string
     *  @throws  DriverException
     */
    @Command(name="showname", description="Show module's name")
    public String showName(@Argument(name="modid", description="Module ID")
                           int modId) throws DriverException
    {
        return "Device name: " + maq.getModuleName(modId);
    }


    /**
     *  Shows a module's type.
     * 
     *  @param  modId  The module ID
     *  @return The result string
     *  @throws  DriverException
     */
    @Command(name="showtype", description="Show module's type")
    public String showType(@Argument(name="modid", description="Module ID")
                           int modId) throws DriverException
    {
        return "Module type: " + maq.getModuleType(modId);
    }


    /**
     *  Shows a module's serial number.
     * 
     *  @param  modId  The module ID
     *  @return The result string
     *  @throws  DriverException
     */
    @Command(name="showserial", description="Show module's serial number")
    public String showSerial(@Argument(name="modid", description="Module ID")
                           int modId) throws DriverException
    {
        return "Serial number: " + maq.getSerialNumber(modId);
    }


    /**
     *  Shows a module's firmware revision number.
     * 
     *  @param  modId  The module ID
     *  @return The result string
     *  @throws  DriverException
     */
    @Command(name="showrevision", description="Show module's FW revision")
    public String showFwRevision(@Argument(name="modid", description="Module ID")
                                 int modId) throws DriverException
    {
        return "Firmware revision: " + maq.getFwRevision(modId);
    }


    /**
     *  Shows a module's date code.
     * 
     *  @param  modId  The module ID
     *  @return The result string
     *  @throws  DriverException
     */
    @Command(name="showdate", description="Show module's date code")
    public String showDate(@Argument(name="modid", description="Module ID")
                           int modId) throws DriverException
    {
        return "Date code: " + maq.getDateCode(modId);
    }


    /**
     *  Shows a module's input channel count.
     * 
     *  @param  modId  The module ID
     *  @return The result string
     *  @throws  DriverException
     */
    @Command(name="showninput", description="Show module's input channel count")
    public String showNumInput(@Argument(name="modid", description="Module ID")
                               int modId) throws DriverException
    {
        return "Input channel count: " + maq.getNumInputs(modId);
    }


    /**
     *  Shows a module's output channel count.
     * 
     *  @param  modId  The module ID
     *  @return The result string
     *  @throws  DriverException
     */
    @Command(name="shownoutput", description="Show module's output channel count")
    public String showNumOutput(@Argument(name="modid", description="Module ID")
                                int modId) throws DriverException
    {
        return "Output channel count: " + maq.getNumOutputs(modId);
    }


    /**
     *  Shows a module's range count.
     * 
     *  @param  modId  The module ID
     *  @return The result string
     *  @throws  DriverException
     */
    @Command(name="shownrange", description="Show module's range count")
    public String showNumRange(@Argument(name="modid", description="Module ID")
                               int modId) throws DriverException
    {
        return "Range count: " + maq.getNumRanges(modId);
    }


    /**
     *  Shows a channel's range number.
     * 
     *  @param  modId  The module ID
     *  @param  chan   The channel number
     *  @return The result string
     *  @throws  DriverException
     */
    @Command(name="showrange", description="Show channel's range")
    public String showRange(@Argument(name="modid", description="Module ID")
                            int modId,
                            @Argument(name="chan", description="Channel number")
                            int chan) throws DriverException
    {
        Maq20Analog maqAna = getAnalogAccess(modId);
        return "Range: " + maqAna.getRange(chan);
    }


    /**
     *  Sets a channel's range number.
     * 
     *  @param  modId  The module ID
     *  @param  chan   The channel number
     *  @param  range  The range number
     *  @throws  DriverException
     */
    @Command(name="setrange", description="Set channel's range")
    public void setRange(@Argument(name="modid", description="Module ID")
                         int modId,
                         @Argument(name="chan", description="Channel number")
                         int chan,
                         @Argument(name="range", description="Range number")
                         int range) throws DriverException
    {
        Maq20Analog maqAna = getAnalogAccess(modId);
        maqAna.setRange(chan, range);
    }


    /**
     *  Shows a channel's enabled state.
     * 
     *  @param  modId  The module ID
     *  @param  chan   The channel number
     *  @return The result string
     *  @throws  DriverException
     */
    @Command(name="showenabled", description="Show channel's enabled state")
    public String showEnabled(@Argument(name="modid", description="Module ID")
                              int modId,
                              @Argument(name="chan", description="Channel number")
                              int chan) throws DriverException
    {
        Maq20Analog maqAna = getAnalogAccess(modId);
        return "Enabled state: " + (maqAna.isEnabled(chan) ? "on" : "off");
    }


    /**
     *  Sets a channel's enabled state.
     * 
     *  @param  modId  The module ID
     *  @param  chan   The channel number
     *  @param  state  The enabled state
     *  @throws  DriverException
     */
    @Command(name="setenabled", description="Set channel's enabled state")
    public void setEnabled(@Argument(name="modid", description="Module ID")
                           int modId,
                           @Argument(name="chan", description="Channel number")
                           int chan,
                           @Argument(name="state", description="Enabled state")
                           OnOff state) throws DriverException
    {
        Maq20Analog maqAna = getAnalogAccess(modId);
        maqAna.enable(chan, state == OnOff.ON);
    }


    /**
     *  Shows selected analog channels' data values.
     * 
     *  @param  modId  The module ID
     *  @param  chan   The first channel number
     *  @param  count  The number of channels
     *  @return The result string
     *  @throws  DriverException
     */
    @Command(name="readvalue", description="Show channels' data values")
    public String readValue(@Argument(name="modid", description="Module ID")
                            int modId,
                            @Argument(name="chan", description="Channel number")
                            int chan,
                            @Argument(name="count", description="Number of channels")
                            int count) throws DriverException
    {
        Maq20Analog maqAna = getAnalogAccess(modId);
        return formatDoubles(maqAna.readValue(chan, count));
    }


    /**
     *  Shows one analog channel's data value.
     * 
     *  @param  modId  The module ID
     *  @param  chan   The channel number
     *  @return The result string
     *  @throws  DriverException
     */
    @Command(name="readvalue", description="Show channel's data value")
    public String readValue(@Argument(name="modid", description="Module ID")
                            int modId,
                            @Argument(name="chan", description="Channel number")
                            int chan) throws DriverException
    {
        Maq20Analog maqAna = getAnalogAccess(modId);
        return formatDoubles(new double[]{maqAna.readValue(chan)});
    }


    /**
     *  Shows all analog channels' data values.
     * 
     *  @param  modId  The module ID
     *  @return The result string
     *  @throws  DriverException
     */
    @Command(name="readvalue", description="Show all channels' data values")
    public String readValue(@Argument(name="modid", description="Module ID")
                            int modId) throws DriverException
    {
        Maq20Analog maqAna = getAnalogAccess(modId);
        return formatDoubles(maqAna.readValue());
    }


    /**
     *  Shows selected discrete input channels' data values.
     * 
     *  @param  modId  The module ID
     *  @param  chan   The first channel number
     *  @param  count  The number of channels
     *  @return The result string
     *  @throws  DriverException
     */
    @Command(name="readdiscin", description="Show discrete input channels' values")
    public String readDiscIn(@Argument(name="modid", description="Module ID")
                             int modId,
                             @Argument(name="chan", description="Channel number")
                             int chan,
                             @Argument(name="count", description="Number of channels")
                             int count) throws DriverException
    {
        Maq20Discrete maqDisc = getDiscreteAccess(modId);
        return formatIntegers(maqDisc.readDiscIn(chan, count));
    }


    /**
     *  Shows one discrete input channel's data value.
     * 
     *  @param  modId  The module ID
     *  @param  chan   The channel number
     *  @return The result string
     *  @throws  DriverException
     */
    @Command(name="readdiscin", description="Show discrete input channel's value")
    public String readDiscIn(@Argument(name="modid", description="Module ID")
                             int modId,
                             @Argument(name="chan", description="Channel number")
                             int chan) throws DriverException
    {
        Maq20Discrete maqDisc = getDiscreteAccess(modId);
        return formatIntegers(new int[]{maqDisc.readDiscIn(chan)});
    }


    /**
     *  Shows all discrete input channels' data values.
     * 
     *  @param  modId  The module ID
     *  @return The result string
     *  @throws  DriverException
     */
    @Command(name="readdiscin", description="Show all discrete input channels' values")
    public String readDiscIn(@Argument(name="modid", description="Module ID")
                             int modId) throws DriverException
    {
        Maq20Discrete maqDisc = getDiscreteAccess(modId);
        return formatIntegers(maqDisc.readDiscIn());
    }


    /**
     *  Shows selected discrete output channels' data values.
     * 
     *  @param  modId  The module ID
     *  @param  chan   The first channel number
     *  @param  count  The number of channels
     *  @return The result string
     *  @throws  DriverException
     */
    @Command(name="readdiscout", description="Show discrete output channels' values")
    public String readDiscOut(@Argument(name="modid", description="Module ID")
                              int modId,
                              @Argument(name="chan", description="Channel number")
                              int chan,
                              @Argument(name="count", description="Number of channels")
                              int count) throws DriverException
    {
        Maq20Discrete maqDisc = getDiscreteAccess(modId);
        return formatIntegers(maqDisc.readDiscOut(chan, count));
    }


    /**
     *  Shows one discrete output channel's data value.
     * 
     *  @param  modId  The module ID
     *  @param  chan   The channel number
     *  @return The result string
     *  @throws  DriverException
     */
    @Command(name="readdiscout", description="Show discrete output channel's value")
    public String readDiscOut(@Argument(name="modid", description="Module ID")
                              int modId,
                              @Argument(name="chan", description="Channel number")
                              int chan) throws DriverException
    {
        Maq20Discrete maqDisc = getDiscreteAccess(modId);
        return formatIntegers(new int[]{maqDisc.readDiscOut(chan)});
    }


    /**
     *  Shows all discrete output channels' data values.
     * 
     *  @param  modId  The module ID
     *  @return The result string
     *  @throws  DriverException
     */
    @Command(name="readdiscout", description="Show all discrete output channels' values")
    public String readDiscOut(@Argument(name="modid", description="Module ID")
                              int modId) throws DriverException
    {
        Maq20Discrete maqDisc = getDiscreteAccess(modId);
        return formatIntegers(maqDisc.readDiscOut());
    }


    /**
     *  Sets a discrete function.
     * 
     *  @param  modId  The module ID
     *  @param  chan   The function channel (0 or 1)
     *  @param  func   The function name
     *  @throws  DriverException
     */
    @Command(name="setfunction", description="Set discrete function code")
    public void setFunction(@Argument(name="modid", description="Module ID")
                            int modId,
                            @Argument(name="chan", description="Function channel number")
                            int chan,
                            @Argument(name="func", description="Function name")
                            DiscFunc func) throws DriverException
    {
        Maq20DiscretePulse maqDisc = getDiscretePulseAccess(modId);
        maqDisc.setFunction(chan, func.getValue());
    }


    /**
     *  Shows a discrete function.
     * 
     *  @param  modId  The module ID
     *  @param  chan   The function channel (0 or 1)
     *  @return The result string
     *  @throws  DriverException
     */
    @Command(name="showfunction", description="Show discrete function code")
    public String showFunction(@Argument(name="modid", description="Module ID")
                               int modId,
                               @Argument(name="chan", description="Function channel number")
                               int chan) throws DriverException
    {
        Maq20DiscretePulse maqDisc = getDiscretePulseAccess(modId);
        int func = maqDisc.getFunction(chan);
        return "Function: " + func + " (" + funcDescs.get(func) + ")";
    }


    /**
     *  Arms a discrete function.
     * 
     *  @param  modId  The module ID
     *  @param  chan   The function channel (0 or 1)
     *  @throws  DriverException
     */
    @Command(name="armfunction", description="Arm discrete function")
    public void armFunction(@Argument(name="modid", description="Module ID")
                            int modId,
                            @Argument(name="chan", description="Function channel number")
                            int chan) throws DriverException
    {
        Maq20DiscreteFunc maqDisc = getDiscretePulseAccess(modId);
        maqDisc.armFunction(chan);
    }


    /**
     *  Disarms a discrete function.
     * 
     *  @param  modId  The module ID
     *  @param  chan   The function channel (0 or 1)
     *  @throws  DriverException
     */
    @Command(name="disarmfunction", description="Disarm discrete function")
    public void disarmFunction(@Argument(name="modid", description="Module ID")
                               int modId,
                               @Argument(name="chan", description="Function channel number")
                               int chan) throws DriverException
    {
        Maq20DiscreteFunc maqDisc = getDiscretePulseAccess(modId);
        maqDisc.disarmFunction(chan);
    }


    /**
     *  Shows a discrete pulse count.
     * 
     *  @param  modId  The module ID
     *  @param  chan   The function channel (0 or 1)
     *  @return The result string
     *  @throws  DriverException
     */
    @Command(name="readpulse", description="Show pulse count")
    public String readPulseCount(@Argument(name="modid", description="Module ID")
                                 int modId,
                                 @Argument(name="chan", description="Function channel number")
                                 int chan) throws DriverException
    {
        Maq20DiscretePulse maqDisc = getDiscretePulseAccess(modId);
        return "Pulse count: " + maqDisc.readPulseCount(chan);
    }


    /**
     *  Shows a discrete pulse frequency value.
     * 
     *  @param  modId  The module ID
     *  @param  chan   The function channel (0 or 1)
     *  @return The result string
     *  @throws  DriverException
     */
    @Command(name="readfreq", description="Show frequency")
    public String readFrequency(@Argument(name="modid", description="Module ID")
                                int modId,
                                @Argument(name="chan", description="Function channel number")
                                int chan) throws DriverException
    {
        Maq20DiscretePulse maqDisc = getDiscretePulseAccess(modId);
        return "Frequency: " + maqDisc.readFrequency(chan);
    }


    /**
     *  Shows a discrete pulse RPM.
     * 
     *  @param  modId  The module ID
     *  @param  chan   The function channel (0 or 1)
     *  @return The result string
     *  @throws  DriverException
     */
    @Command(name="readrpm", description="Show RPM")
    public String readRPM(@Argument(name="modid", description="Module ID")
                          int modId,
                          @Argument(name="chan", description="Function channel number")
                          int chan) throws DriverException
    {
        Maq20DiscretePulse maqDisc = getDiscretePulseAccess(modId);
        return "RPM: " + maqDisc.readRPM(chan);
    }


    /**
     *  Sets a discrete pulse counter pulses per revolution.
     * 
     *  @param  modId  The module ID
     *  @param  chan   The function channel (0 or 1)
     *  @param  ppr    The number of pulses per revolution
     *  @throws  DriverException
     */
    @Command(name="setpulserev", description="Set pulses per revolution")
    public void setPulsesPerRevn(@Argument(name="modid", description="Module ID")
                                 int modId,
                                 @Argument(name="chan", description="Function channel number")
                                 int chan,
                                 @Argument(name="ppr", description="Pulses/revolution")
                                 int ppr) throws DriverException
    {
        Maq20DiscretePulse maqDisc = getDiscretePulseAccess(modId);
        maqDisc.setPulsesPerRevn(chan, ppr);
        maqDisc.armFunction(chan);
    }


    /**
     *  Shows a discrete pulse counter pulses per revolution.
     * 
     *  @param  modId  The module ID
     *  @param  chan   The function channel (0 or 1)
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="showpulserev", description="Set pulses per revolution")
    public String showPulsesPerRevn(@Argument(name="modid", description="Module ID")
                                    int modId,
                                    @Argument(name="chan", description="Function channel number")
                                    int chan) throws DriverException
    {
        Maq20DiscretePulse maqDisc = getDiscretePulseAccess(modId);
        return "Pulses/revolution: " + maqDisc.getPulsesPerRevn(chan);
    }


    /**
     *  Sets a discrete PWM timebase.
     * 
     *  @param  modId     The module ID
     *  @param  chan      The function channel (0 or 1)
     *  @param  timebase  The timebase name
     *  @throws  DriverException
     */
    @Command(name="settimebase", description="Set PWM timebase")
    public void setTimebase(@Argument(name="modid", description="Module ID")
                            int modId,
                            @Argument(name="chan", description="Function channel number")
                            int chan,
                            @Argument(name="time", description="Timebase")
                            Timebase timebase) throws DriverException
    {
        Maq20DiscretePWM maqDisc = getDiscretePWMAccess(modId);
        maqDisc.setTimebase(chan, timebase.getValue());
        maqDisc.armFunction(chan);
    }


    /**
     *  Shows a discrete PWM timebase setting.
     * 
     *  @param  modId  The module ID
     *  @param  chan   The function channel (0 or 1)
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="showtimebase", description="Show PWM timebase")
    public String showTimebase(@Argument(name="modid", description="Module ID")
                               int modId,
                               @Argument(name="chan", description="Function channel number")
                               int chan) throws DriverException
    {
        Maq20DiscretePWM maqDisc = getDiscretePWMAccess(modId);
        int timebase = maqDisc.getTimebase(chan);
        return "PWM timebase: " + timebase + " (" + timebaseDescs.get(timebase) +")";
    }


    /**
     *  Sets a discrete PWM period.
     * 
     *  @param  modId   The module ID
     *  @param  chan    The function channel (0 or 1)
     *  @param  period  The period, in timebase units
     *  @throws  DriverException
     */
    @Command(name="setperiod", description="Set PWM period")
    public void setPeriod(@Argument(name="modid", description="Module ID")
                          int modId,
                          @Argument(name="chan", description="Function channel number")
                          int chan,
                          @Argument(name="period", description="PWM period")
                          int period) throws DriverException
    {
        Maq20DiscretePWM maqDisc = getDiscretePWMAccess(modId);
        maqDisc.setPeriod(chan, period);
        maqDisc.armFunction(chan);
    }


    /**
     *  Shows a discrete PWM period.
     * 
     *  @param  modId  The module ID
     *  @param  chan   The function channel (0 or 1)
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="showperiod", description="Show PWM period")
    public String showPeriod(@Argument(name="modid", description="Module ID")
                             int modId,
                             @Argument(name="chan", description="Function channel number")
                             int chan) throws DriverException
    {
        Maq20DiscretePWM maqDisc = getDiscretePWMAccess(modId);
        return "PWM period: " + maqDisc.getPeriod(chan);
    }


    /**
     *  Sets a discrete PWM output 1 low time.
     * 
     *  @param  modId  The module ID
     *  @param  chan   The function channel (0 or 1)
     *  @param  time   The time value
     *  @throws  DriverException
     */
    @Command(name="setlowtime1", description="Set output 1 low time")
    public void setLowTime1(@Argument(name="modid", description="Module ID")
                            int modId,
                            @Argument(name="chan", description="Function channel number")
                            int chan,
                            @Argument(name="time", description="Time")
                            int time) throws DriverException
    {
        Maq20DiscretePWM maqDisc = getDiscretePWMAccess(modId);
        maqDisc.setLowTime1(chan, time);
        maqDisc.armFunction(chan);
    }


    /**
     *  Shows a discrete PWM output 1 low time.
     * 
     *  @param  modId  The module ID
     *  @param  chan   The function channel (0 or 1)
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="showlowtime1", description="Show PWM output 1 low time")
    public String showLowTime1(@Argument(name="modid", description="Module ID")
                               int modId,
                               @Argument(name="chan", description="Function channel number")
                               int chan) throws DriverException
    {
        Maq20DiscretePWM maqDisc = getDiscretePWMAccess(modId);
        return "PWM low time 1: " + maqDisc.getLowTime1(chan);
    }


    /**
     *  Sets a discrete PWM output 2 low time.
     * 
     *  @param  modId  The module ID
     *  @param  chan   The function channel (0 or 1)
     *  @param  time   The time value
     *  @throws  DriverException
     */
    @Command(name="setlowtime2", description="Set output 2 low time")
    public void setLowTime2(@Argument(name="modid", description="Module ID")
                            int modId,
                            @Argument(name="chan", description="Function channel number")
                            int chan,
                            @Argument(name="time", description="Time")
                            int time) throws DriverException
    {
        Maq20DiscretePWM maqDisc = getDiscretePWMAccess(modId);
        maqDisc.setLowTime2(chan, time);
        maqDisc.armFunction(chan);
    }


    /**
     *  Shows a discrete PWM output 2 low time.
     * 
     *  @param  modId  The module ID
     *  @param  chan   The function channel (0 or 1)
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="showlowtime2", description="Show PWM output 2 low time")
    public String showLowTime2(@Argument(name="modid", description="Module ID")
                               int modId,
                               @Argument(name="chan", description="Function channel number")
                               int chan) throws DriverException
    {
        Maq20DiscretePWM maqDisc = getDiscretePWMAccess(modId);
        return "PWM low time 2: " + maqDisc.getLowTime2(chan);
    }


    /**
     *  Enables/disables a discrete PWM output 2.
     * 
     *  @param  modId   The module ID
     *  @param  chan    The function channel (0 or 1)
     *  @param  enable  The time value
     *  @throws  DriverException
     */
    @Command(name="enableout2", description="Enable/disable output 2")
    public void enableOut2(@Argument(name="modid", description="Module ID")
                           int modId,
                           @Argument(name="chan", description="Function channel number")
                           int chan,
                           @Argument(name="time", description="Time")
                           boolean enable) throws DriverException
    {
        Maq20DiscretePWM maqDisc = getDiscretePWMAccess(modId);
        maqDisc.enableOutput2(chan, enable);
        maqDisc.armFunction(chan);
    }


    /**
     *  Shows whether a discrete PWM output 2 is enabled.
     * 
     *  @param  modId  The module ID
     *  @param  chan   The function channel (0 or 1)
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(name="is2enabled", description="Show whether PWM output 2 is enabled")
    public String is2Enabled(@Argument(name="modid", description="Module ID")
                             int modId,
                             @Argument(name="chan", description="Function channel number")
                             int chan) throws DriverException
    {
        Maq20DiscretePWM maqDisc = getDiscretePWMAccess(modId);
        return "PWM output 2 is " + (maqDisc.isOutput2Enabled(chan) ? "" : "not ") + "enabled";
    }


    /**
     *  Gets analog module access.
     *
     *  @param  modId  The module ID
     *  @return  The analog module access object
     *  @throws  DriverException
     */
    private Maq20Analog getAnalogAccess(int modId) throws DriverException
    {
        Access acc = getAccess(modId);
        checkPresent(acc.maqAna);
        return acc.maqAna;
    }


    /**
     *  Gets analog output module access.
     *
     *  @param  modId  The module ID
     *  @return  The analog module output access object
     *  @throws  DriverException
     */
    private Maq20AnalogOut getAnalogOutAccess(int modId) throws DriverException
    {
        Access acc = getAccess(modId);
        checkPresent(acc.maqAnaOut);
        return acc.maqAnaOut;
    }


    /**
     *  Gets discrete module access.
     *
     *  @param  modId  The module ID
     *  @return  The discrete module access object
     *  @throws  DriverException
     */
    private Maq20Discrete getDiscreteAccess(int modId) throws DriverException
    {
        Access acc = getAccess(modId);
        checkPresent(acc.maqDisc);
        return acc.maqDisc;
    }


    /**
     *  Gets discrete module pulse function access.
     *
     *  @param  modId  The module ID
     *  @return  The discrete module access object
     *  @throws  DriverException
     */
    private Maq20DiscretePulse getDiscretePulseAccess(int modId) throws DriverException
    {
        Access acc = getAccess(modId);
        checkPresent(acc.maqDiscPulse);
        return acc.maqDiscPulse;
    }


    /**
     *  Gets discrete module PWM function access.
     *
     *  @param  modId  The module ID
     *  @return  The discrete module access object
     *  @throws  DriverException
     */
    private Maq20DiscretePWM getDiscretePWMAccess(int modId) throws DriverException
    {
        Access acc = getAccess(modId);
        checkPresent(acc.maqDiscPWM);
        return acc.maqDiscPWM;
    }


    /**
     *  Gets discrete module frequency function access.
     *
     *  @param  modId  The module ID
     *  @return  The discrete module access object
     *  @throws  DriverException
     */
    private Maq20DiscreteFreq getDiscreteFreqAccess(int modId) throws DriverException
    {
        Access acc = getAccess(modId);
        checkPresent(acc.maqDiscFreq);
        return acc.maqDiscFreq;
    }


    /**
     *  Gets analog access.
     *
     *  @param  modId  The module ID
     *  @return  The overall access object
     *  @throws  DriverException
     */
    private Access getAccess(int modId) throws DriverException
    {
        Access acc = (modId > 0 && modId < Maq20.NUM_MODULES) ? access[modId] : null;
        if (acc == null) {
            throw new DriverException("Invalid module ID");
        }
        return acc;
    }


    /**
     *  Checks for access object's presence.
     *
     *  @param  obj  The access object
     *  @throws  DriverException
     */
    private void checkPresent(Object obj) throws DriverException
    {
        if (obj == null) {
            throw new DriverException("Invalid module type for command");
        }
    }


    /**
     *  Formats an array of data values.
     * 
     *  @param  values  The array of values
     */
    private String formatDoubles(double[] values)
    {
        StringBuilder text = new StringBuilder(values.length > 1 ? "Values:" : "Value:");
        for (double value : values) {
            text.append(String.format(" %.6g", value));
        }
        return text.toString();
    }


    /**
     *  Formats an array of integer values.
     * 
     *  @param  values  The array of values
     */
    private String formatIntegers(int[] values)
    {
        StringBuilder text = new StringBuilder(values.length > 1 ? "Values:" : "Value:");
        for (int value : values) {
            text.append(" ").append(value);
        }
        return text.toString();
    }

}
