package org.lsst.ccs.drivers.dataforth;

import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Routines for controlling a Dataforth MAQ20 discrete I/O module.
 *
 *  @author  Owen Saxton
 */
public class Maq20Discrete {

    /**
     *  Package data.
     */
    final Maq20 maq;
    final Maq20.ModuleData module;

    /**
     *  Private data.
     */
    private static final short
        DISCR_CONFIG_ADDR = 100,
        CHAN_DATA_ADDR = 1000;


    /**
     *  Constructor.
     *
     *  @param  maq    The underlying Maq20 object
     *  @param  modId  The module registration ID
     *  @throws DriverException
     */
    public Maq20Discrete(Maq20 maq, int modId) throws DriverException
    {
        this.maq = maq;
        module = maq.getModuleData(modId, false);
        if (module.opType != Maq20.OPER_DISCRETE) {
            throw new DriverException("Invalid module type (" + module.type + ") for discrete operations");
        }
    }


    /**
     *  Reads several discrete input channels.
     *
     *  @param  chan   The first channel number
     *  @param  count  The number of channels
     *  @return  The array of data values
     *  @throws  DriverException
     */
    public int[] readDiscIn(int chan, int count) throws DriverException
    {
        int nChan = count >= 0 ? count : module.numInChan - chan;
        checkInputChannel(chan, nChan);
        if (nChan == 0) return new int[0];
        short addr = (short)(module.baseAddr + CHAN_DATA_ADDR + module.numOutChan + chan);
        short[] raw = maq.readRegisters(addr, (short)nChan);
        int[] data = new int[nChan];
        for (int j = 0; j < data.length; j++) {
            data[j] = raw[j];
        }
        return data;
    }


    /**
     *  Reads a discrete input channel.
     *
     *  @param  chan   The channel number
     *  @return  The data value
     *  @throws  DriverException
     */
    public int readDiscIn(int chan) throws DriverException
    {
        return readDiscIn(chan, 1)[0];
    }


    /**
     *  Reads all discrete input channels.
     *
     *  @return  The array of data values
     *  @throws  DriverException
     */
    public int[] readDiscIn() throws DriverException
    {
        return readDiscIn(0, -1);
    }


    /**
     *  Reads several discrete output channels.
     *
     *  @param  chan   The first channel number
     *  @param  count  The number of channels
     *  @return  The array of data values
     *  @throws  DriverException
     */
    public int[] readDiscOut(int chan, int count) throws DriverException
    {
        int nChan = count >= 0 ? count : module.numOutChan - chan;
        checkOutputChannel(chan, nChan);
        if (nChan == 0) return new int[0];
        short addr = (short)(module.baseAddr + CHAN_DATA_ADDR + chan);
        short[] raw = maq.readRegisters(addr, (short)nChan);
        int[] data = new int[nChan];
        for (int j = 0; j < data.length; j++) {
            data[j] = raw[j];
        }
        return data;
    }


    /**
     *  Reads a discrete output channel.
     *
     *  @param  chan   The channel number
     *  @return  The data value
     *  @throws  DriverException
     */
    public int readDiscOut(int chan) throws DriverException
    {
        return readDiscOut(chan, 1)[0];
    }


    /**
     *  Reads all discrete output channels.
     *
     *  @return  The array of data values
     *  @throws  DriverException
     */
    public int[] readDiscOut() throws DriverException
    {
        return readDiscOut(0, -1);
    }


    /**
     *  Writes to several discrete output channels.
     *
     *  @param  chan   The first channel number
     *  @param  count  The number of channels
     *  @param  data   The array of data values (at least count elements long)
     *  @throws  DriverException
     */
    public void writeDisc(int chan, int count, int[] data) throws DriverException
    {
        int nChan = count >= 0 ? count : module.numOutChan - chan;
        checkOutputChannel(chan, nChan);
        if (nChan == 0) return;
        short[]raw = new short[nChan];
        for (int j = 0; j < data.length; j++) {
            raw[j] = (short)data[j];
        }
        maq.writeRegisters((short)(module.baseAddr + CHAN_DATA_ADDR + chan), raw);
    }


    /**
     *  Writes to a discrete output channel.
     *
     *  @param  chan   The channel number
     *  @param  data   The data value
     *  @throws  DriverException
     */
    public void writeDisc(int chan, int data) throws DriverException
    {
        writeDisc(chan, 1, new int[]{data});
    }


    /**
     *  Writes to all discrete output channels.
     *
     *  @param  data   The array of data values
     *  @throws  DriverException
     */
    public void writeDisc(int[] data) throws DriverException
    {
        writeDisc(0, -1, data);
    }


    /**
     *  Checks a range of input channel numbers for validity.
     *
     *  @param  chan    The first channel number
     *  @param  count   The number of channels
     *  @throws  DriverException
     */
    private void checkInputChannel(int chan, int count) throws DriverException
    {
        if (chan < 0 || chan >= module.numInChan) {
            throw new DriverException("Invalid input channel number");
        }
        if (count < 0 || chan + count > module.numInChan) {
            throw new DriverException("Invalid input channel count");
        }
    }


    /**
     *  Checks a range of output channel numbers for validity.
     *
     *  @param  chan    The first channel number
     *  @param  count   The number of channels
     *  @throws  DriverException
     */
    private void checkOutputChannel(int chan, int count) throws DriverException
    {
        if (chan < 0 || chan >= module.numOutChan) {
            throw new DriverException("Invalid output channel number");
        }
        if (count < 0 || chan + count > module.numOutChan) {
            throw new DriverException("Invalid output channel count");
        }
    }

}
