package org.lsst.ccs.drivers.dataforth;

import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Routines for controlling a Dataforth MAQ20 analog output module.
 *
 *  @author  Owen Saxton
 */
public class Maq20AnalogOut extends Maq20Analog {

    /**
     *  Constructor.
     *
     *  @param  maq    The underlying Maq20 object
     *  @param  modId  The module registration ID
     *  @throws DriverException
     */
    public Maq20AnalogOut(Maq20 maq, int modId) throws DriverException
    {
        super(maq, modId);
        if (module.opType != Maq20.OPER_ANALOUT) {
            throw new DriverException("Invalid module type (" + module.type + ") for analog output operations");
        }
    }


    /**
     *  Writes data values to several channels.
     *
     *  @param  chan   The first channel number
     *  @param  count  The number of channels
     *  @param  data   The array of data values (at least count elements long)
     *  @throws  DriverException
     */
    public void writeValue(int chan, int count, double[] data) throws DriverException
    {
        int nChan = count >= 0 ? count : module.numOutChan - chan;
        checkChannel(chan, nChan);
        if (nChan == 0) return;
        short[]raw = new short[nChan];
        for (int j = 0; j < nChan; j++, chan++) {
            int range = module.range[chan];
            raw[j] = (short)((data[j] + module.offset[range]) / module.scale[range]);
        }
        maq.writeRegisters((short)(module.baseAddr + CHAN_DATA_ADDR + chan), raw);
    }


    /**
     *  Writes a data value to a channel.
     *
     *  @param  chan   The channel number
     *  @param  data   The data value
     *  @throws  DriverException
     */
    public void writeValue(int chan, double data) throws DriverException
    {
        writeValue(chan, 1, new double[]{data});
    }


    /**
     *  Writes data values to all channels.
     *
     *  @param  data   The array of data values
     *  @throws  DriverException
     */
    public void writeValue(double[] data) throws DriverException
    {
        writeValue(0, -1, data);
    }


    /**
     *  Checks a range of channel numbers for validity.
     *
     *  @param  chan    The first channel number
     *  @param  count   The number of channels
     *  @throws  DriverException
     */
    private void checkChannel(int chan, int count) throws DriverException
    {
        int numChan = module.numOutChan;
        if (chan < 0 || chan >= numChan) {
            throw new DriverException("Invalid channel number");
        }
        if (count < 0 || chan + count > numChan) {
            throw new DriverException("Invalid channel count");
        }
    }

}
