package org.lsst.ccs.drivers.dataforth;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.drivers.modbus.Modbus;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Routines for controlling a Dataforth MAQ20 DAQ system.
 *
 *  @author  Owen Saxton
 */
public class Maq20 extends Modbus {

   /**
    *  Inner class for storing module data.
    */
    static class ModuleData {

        ModuleType type;
        int opType;
        int baseAddr;
        int numInChan;
        int numOutChan;
        int[] range = new int[NUM_CHANNELS];
        double[] offset;
        double[] scale;

    }

   /**
    *  Public constants.
    */
    public enum ModuleType {UNKNOWN,
                            COMM,  // Communications
                            JTC,   // Type J thermocouple
                            KTC,   // Type K thermocouple
                            TTC,   // Type T thermocouple
                            RSTC,  // Type R and S thermocouple
                            RTD,   // RTD sensor
                            MVD,   // Millivolt input (differential)
                            VD,    // Voltage input (differential)
                            VS,    // Voltage input (single-ended)
                            ID,    // Current input (differential)
                            IS,    // Current input (single-ended)
                            VO,    // Voltage output
                            IO,    // Current output
                            DIOH,  // Discrete I/O (AC/DC)
                            DIOL}  // Discrete I/O (DC only)

    public static final int
        NUM_MODULES      = 24,   // The maximum number of modules
        OPER_ANALOG      = 0,    // Module can do analog operations
        OPER_ANALOUT     = 1,    // Module can do analog output operations
        OPER_DISCRETE    = 2;    // Module can do discrete operations

   /**
    *  Package data.
    */
    static final short
        ZERO = 0,
        ONE = 1,
        TWO = 2;

   /**
    *  Private data.
    */
    private static final short
        SLAVE_ADDR = 0,
        MODU_ADDR_INCR = 2000,
        DEV_NAME_ADDR = 0,
        DEV_NAME_LENG = 15,
        SER_NUM_ADDR = 20,
        SER_NUM_LENG = 10,
        DATE_CODE_ADDR = 31,
        DATE_CODE_LENG = 4,
        FW_REVN_ADDR = 36,
        FW_REVN_LENG = 4,

        IP_ADDRESS_ADDR = 50,
        IP_ADDRESS_LENG = 4,
        SUBNET_MASK_ADDR = 55,
        MODU_DETECT_ADDR = 98,
        RESET_ADDR = 99,
        MODU_STATUS_ADDR = 100,
        REGN_SN_ID_ADDR = 1000,
        AUTO_REGN_ADDR = 1020,
        SAVE_REGN_ADDR = 1021,
        DEL_REGN_ADDR = 1022,
        TEMPERATURE_ADDR = 1210,

        NUM_INP_CHANS_ADDR = 40,
        NUM_OUT_CHANS_ADDR = 41,
        INPUT_RANGE_ADDR = 100,
        RANGE_COUNT_ADDR = 1700,
        RANGE_DATA_ADDR = 1710,
        RANGE_DATA_INCR = 20,
        ENG_NFS_OFFS = 0,
        ENG_PFS_OFFS = 2,
        ENG_FS_PWR_OFFS = 4,
        CNT_NFS_OFFS = 8,
        CNT_PFS_OFFS = 10,
        NUM_CHANNELS = 16;

    private static final Map<String, ModuleType> typeMap = new HashMap<>();
    static {
        typeMap.put("JTC", ModuleType.JTC);
        typeMap.put("KTC", ModuleType.KTC);
        typeMap.put("TTC", ModuleType.TTC);
        typeMap.put("RSTC", ModuleType.RSTC);
        typeMap.put("RTD31", ModuleType.RTD);
        typeMap.put("MVDN", ModuleType.MVD);
        typeMap.put("VDN", ModuleType.VD);
        typeMap.put("VSN", ModuleType.VS);
        typeMap.put("IDN", ModuleType.ID);
        typeMap.put("ISN", ModuleType.IS);
        typeMap.put("VO", ModuleType.VO);
        typeMap.put("IO", ModuleType.IO);
        typeMap.put("DIOH", ModuleType.DIOH);
        typeMap.put("DIOL", ModuleType.DIOL);
    }
    private static final Map<ModuleType, Integer> operMap = new HashMap<>();
    static {
        operMap.put(ModuleType.JTC, OPER_ANALOG);
        operMap.put(ModuleType.KTC, OPER_ANALOG);
        operMap.put(ModuleType.TTC, OPER_ANALOG);
        operMap.put(ModuleType.RSTC, OPER_ANALOG);
        operMap.put(ModuleType.RTD, OPER_ANALOG);
        operMap.put(ModuleType.MVD, OPER_ANALOG);
        operMap.put(ModuleType.VD, OPER_ANALOG);
        operMap.put(ModuleType.VS, OPER_ANALOG);
        operMap.put(ModuleType.ID, OPER_ANALOG);
        operMap.put(ModuleType.IS, OPER_ANALOG);
        operMap.put(ModuleType.VO, OPER_ANALOUT);
        operMap.put(ModuleType.IO, OPER_ANALOUT);
        operMap.put(ModuleType.DIOH, OPER_DISCRETE);
        operMap.put(ModuleType.DIOL, OPER_DISCRETE);
    }
    private final ModuleData[] modules = new ModuleData[NUM_MODULES];


   /**
    *  Constructor.
    */
    public Maq20()
    {
        modules[0] = new ModuleData();
        modules[0].type = ModuleType.COMM;
        modules[0].baseAddr = 0;
    }


   /**
    *  Opens a connection to the device.
    *
    *  @param  type   The type of connection to make
    *  @param  ident  The device identifier:
    *                   host name or IP address for network;
    *                   serial number for FTDI device;
    *                   port name for serial
    *  @param  parm1  The first device parameter:
    *                   port number for network;
    *                   baud rate for FTDI or serial
    *  @param  parm2  The second device parameter:
    *                   unused for network;
    *                   data characteristics for FTDI or serial
    *  @throws  DriverException
    */
    @Override
    public void open(ConnType type, String ident, int parm1, int parm2)
        throws DriverException
    {
        super.open(type, ident, parm1, parm2);
        super.setAddressMode(true);
        try {
            fillModuleData();
        }
        catch (DriverException e) {
            closeSilent();
            throw e;
        }
    }


   /**
    *  Sets whether entity numbers are direct addresses.
    *
    *  This routine is a no-op since address mode is always set and should
    *  not be changed.
    *
    *  @param  mode  Whether or not address mode is to be set
    */
    @Override
    public void setAddressMode(boolean mode)
    {
    }


   /**
    *  Reads a set of registers.
    *
    *  @param  addr   The first register address
    *  @param  count  The number of registers to read
    *  @return  An array of register values
    *  @throws DriverException
    */
    public short[] readRegisters(short addr, short count) throws DriverException
    {
        return readRegisters(SLAVE_ADDR, addr, count);
    }


   /**
    *  Reads a register.
    *
    *  @param  addr   The register address
    *  @return  The read value as a signed integer
    *  @throws DriverException
    */
    public int readRegister(short addr) throws DriverException
    {
        return readRegisters(SLAVE_ADDR, addr, ONE)[0];
    }


   /**
    *  Reads an integer from a pair of registers in big-endian order.
    *
    *  @param  addr   The first register address
    *  @return  The combined integer
    *  @throws DriverException
    */
    public int readRegisterPair(short addr) throws DriverException
    {
        short[] value = readRegisters(SLAVE_ADDR, addr, TWO);
        return (value[0] << 16) | (value[1] & 0xffff);
    }


   /**
    *  Writes a single register.
    *
    *  @param  addr   The register address
    *  @param  value  The value to write
    *  @throws DriverException
    */
    public void writeRegister(short addr, short value) throws DriverException
    {
        writeRegister(SLAVE_ADDR, addr, value);
    }


   /**
    *  Writes a set of registers.
    *
    *  @param  addr   The first register address
    *  @param  value  The array of values to write
    *  @throws DriverException
    */
    public void writeRegisters(short addr, short[] value) throws DriverException
    {
        writeRegisters(SLAVE_ADDR, addr, value);
    }


   /**
    *  Writes an integer to a pair of registers in big-endian order.
    *
    *  @param  addr   The first register address
    *  @param  value  The integer to write
    *  @throws DriverException
    */
    public void writeRegisterPair(short addr, int value) throws DriverException
    {
        writeRegisters(SLAVE_ADDR, addr, new short[]{(short)(value >> 16), (short)value});
    }


   /**
    *  Tests whether a module exists.
    *
    *  @param  modId  The module registration ID
    *  @return  Whether the module exists
    */
    public boolean moduleExists(int modId)
    {
        return modId > 0 && modId < NUM_MODULES && modules[modId] != null;
    }


   /**
    *  Gets the module name.
    *
    *  @param  modId  The module registration ID
    *  @return  The module name
    *  @throws  DriverException
    */
    public String getModuleName(int modId) throws DriverException
    {
        short addr = (short)(getModuleData(modId, true).baseAddr + DEV_NAME_ADDR);
        short[] text = readRegisters(addr, DEV_NAME_LENG);
        return makeString(text);
    }


   /**
    *  Gets the module type.
    *
    *  @param  modId  The module registration ID
    *  @return  The module type
    *  @throws  DriverException
    */
    public ModuleType getModuleType(int modId) throws DriverException
    {
        return getModuleData(modId, true).type;
    }


   /**
    *  Gets the module operation type.
    *
    *  @param  modId  The module registration ID
    *  @return  The module operation type
    *  @throws  DriverException
    */
    public int getModuleOpType(int modId) throws DriverException
    {
        return getModuleData(modId, false).opType;
    }


   /**
    *  Gets the module operation type.
    *
    *  @param  modType  The module type
    *  @return  The module operation type
    */
    public static int getModuleOpType(ModuleType modType)
    {
        return operMap.get(modType);
    }


   /**
    *  Gets the serial number.
    *
    *  @param  modId  The module registration ID
    *  @return  The serial number
    *  @throws  DriverException
    */
    public String getSerialNumber(int modId) throws DriverException
    {
        getModuleData(modId, true);
        return getSerialNum(modId);
    }


   /**
    *  Gets the date code.
    *
    *  @param  modId  The module registration ID
    *  @return  The date code
    *  @throws  DriverException
    */
    public String getDateCode(int modId) throws DriverException
    {
        short addr = (short)(getModuleData(modId, true).baseAddr + DATE_CODE_ADDR);
        short[] text = readRegisters(addr, DATE_CODE_LENG);
        return makeString(text);
    }


   /**
    *  Gets the firmware revision number.
    *
    *  @param  modId  The module registration ID
    *  @return  The firmware revision number
    *  @throws  DriverException
    */
    public String getFwRevision(int modId) throws DriverException
    {
        short addr = (short)(getModuleData(modId, true).baseAddr + FW_REVN_ADDR);
        short[] text = readRegisters(addr, FW_REVN_LENG);
        return makeString(text);
    }


   /**
    *  Gets the IP address.
    *
    *  @return  The IP address, as a string
    *  @throws  DriverException
    */
    public String getIPAddress() throws DriverException
    {
        short[] addr = readRegisters(IP_ADDRESS_ADDR, IP_ADDRESS_LENG);
        return makeIPString(addr);
    }


   /**
    *  Sets the IP address.
    *
    *  @param  ipAddr  The IP address, as an IP string
    *  @throws  DriverException
    */
    public void setIPAddress(String ipAddr) throws DriverException
    {
        writeRegisters(IP_ADDRESS_ADDR, makeIPAddress(ipAddr));
    }


   /**
    *  Gets the subnet mask.
    *
    *  @return  The subnet mask, as a string
    *  @throws  DriverException
    */
    public String getSubnetMask() throws DriverException
    {
        short[] addr = readRegisters(SUBNET_MASK_ADDR, IP_ADDRESS_LENG);
        return makeIPString(addr);
    }


   /**
    *  Sets the subnet mask.
    *
    *  @param  mask  The subnet mask, as an IP string
    *  @throws  DriverException
    */
    public void setSubnetMask(String mask) throws DriverException
    {
        writeRegisters(SUBNET_MASK_ADDR, makeIPAddress(mask));
    }


   /**
    *  Reads the board temperature.
    *
    *  @return  The board temperature
    *  @throws  DriverException
    */
    public double readTemperature() throws DriverException
    {
        return readRegister(TEMPERATURE_ADDR);
    }


   /**
    *  Register the modules.
    *
    *  @param  serial  The serial numbers of all the modules, in the desired order
    *  @throws  DriverException
    */
    public void register(String... serial) throws DriverException
    {
        List<Short> ids = new ArrayList<>();
        for (short j = 1; j < NUM_MODULES; j++) {
            if (modules[j] != null) {
                ids.add(j);
            }
        }
        for (String sn : serial) {
            if (sn.length() > SER_NUM_LENG) {
                throw new DriverException("Serial number (" + sn + ") too long");
            }
        }
        writeRegister(AUTO_REGN_ADDR, ZERO);
        for (short id : ids) {
            writeRegister(DEL_REGN_ADDR, id);
        }
        short[] snId = new short[SER_NUM_LENG + 1];
        for (String sn : serial) {
            for (int j = 0; j < SER_NUM_LENG; j++) {
                if (j < sn.length()) {
                    snId[j] = (short)sn.charAt(j);
                }
                else {
                    snId[j] = (short)' ';
                }
            }
            snId[SER_NUM_LENG]++;
            writeRegisters(REGN_SN_ID_ADDR, snId);
        }
        writeRegister(SAVE_REGN_ADDR, ONE);
        writeRegister(AUTO_REGN_ADDR, ONE);
        fillModuleData();
    }


   /**
    *  Gets a module's ID, given its serial number.
    *
    *  @param  serial  The module's serial number
    *  @return  The module's registration ID, or -1 if not found
    *  @throws  DriverException
    */
    public int getModuleId(String serial) throws DriverException
    {
        short[] status = readRegisters(MODU_STATUS_ADDR, (short)NUM_MODULES);
        for (int id = 1; id < NUM_MODULES; id++) {
            if (status[id] != 0 && serial.equals(getSerialNum(id))) return id;
        }
        return -1;
    }


   /**
    *  Gets all valid module IDs.
    *
    *  @return  An array of the existing module IDs
    *  @throws  DriverException
    */
    public int[] getModuleIds() throws DriverException
    {
        short[] status = readRegisters(MODU_STATUS_ADDR, (short)NUM_MODULES);
        int count = 0;
        for (int id = 1; id < NUM_MODULES; id++) {
            count += status[id] == 0 ? 0 : 1;
        }
        int[] ids = new int[count];
        int index = 0;
        for (int id = 1; id < NUM_MODULES; id++) {
            if (status[id] != 0) {
                ids[index++] = id;
            }
        }
        return ids;
    }


   /**
    *  Gets the number of input channels for a module.
    *
    *  @param  modId  The module registration ID
    *  @return  The number of input channels
    *  @throws  DriverException
    */
    public int getNumInputs(int modId) throws DriverException
    {
        return getModuleData(modId, false).numInChan;
    }


   /**
    *  Gets the number of output channels for a module.
    *
    *  @param  modId  The module registration ID
    *  @return  The number of output channels
    *  @throws  DriverException
    */
    public int getNumOutputs(int modId) throws DriverException
    {
        return getModuleData(modId, false).numOutChan;
    }


   /**
    *  Gets the number of ranges for a module.
    *
    *  @param  modId  The module registration ID
    *  @return  The number of ranges
    *  @throws  DriverException
    */
    public int getNumRanges(int modId) throws DriverException
    {
        return getModuleData(modId, false).offset.length;
    }


   /**
    *  Gets the serial number without checking the module ID.
    *
    *  @param  modId  The module registration ID
    *  @return  The serial number
    *  @throws  DriverException
    */
    private String getSerialNum(int modId) throws DriverException
    {
        short addr = (short)(MODU_ADDR_INCR * modId + SER_NUM_ADDR);
        short[] text = readRegisters(addr, SER_NUM_LENG);
        return makeString(text);
    }


   /**
    *  Gets saved module data.
    *
    *  @throws  DriverException
    */
    private void fillModuleData() throws DriverException
    {
        short[] status = readRegisters(MODU_STATUS_ADDR, (short)NUM_MODULES);
        for (int id = 1; id < NUM_MODULES; id++) {
            if (status[id] != 0) {
                if (modules[id] == null) {
                    modules[id] = new ModuleData();
                    modules[id].baseAddr = MODU_ADDR_INCR * id;
                }
                ModuleData module = modules[id];
                module.type = ModuleType.UNKNOWN;
                module.opType = 0;
                String name = getModuleName(id);
                if (name.substring(0, 6).equals("MAQ20-")) {
                    ModuleType type = typeMap.get(name.substring(6));
                    if (type != null) {
                        module.type = type;
                        module.opType = operMap.get(type);
                    }
                }
                short addr = (short)(module.baseAddr + NUM_INP_CHANS_ADDR);
                module.numInChan = readRegister(addr);
                addr = (short)(module.baseAddr + NUM_OUT_CHANS_ADDR);
                module.numOutChan = readRegister(addr);
                for (int chan = 0; chan < module.numInChan; chan++) {
                    addr = (short)(module.baseAddr + INPUT_RANGE_ADDR + chan);
                    module.range[chan] = readRegister(addr);
                }
                setConversion(id);
            }
            else {
                modules[id] = null;
            }
        }
    }


   /**
    *  Sets conversion constants for a module.
    *
    *  @param  modId  The module ID
    *  @throws  DriverException
    */
    private void setConversion(int modId) throws DriverException
    {
        ModuleData module = modules[modId];
        short addr = (short)(module.baseAddr + RANGE_COUNT_ADDR);
        int nRange = readRegister(addr);
        if (nRange < 0) return;
        module.offset = new double[nRange];
        module.scale = new double[nRange];
        for (int j = 0; j < nRange; j++) {
            addr = (short)(module.baseAddr + RANGE_DATA_ADDR + j * RANGE_DATA_INCR);
            short[] rData = readRegisters(addr, RANGE_DATA_INCR);
            double cntRange = rData[CNT_PFS_OFFS] - rData[CNT_NFS_OFFS];
            double mult = Math.pow(10, rData[ENG_FS_PWR_OFFS]);
            module.scale[j] = (rData[ENG_PFS_OFFS] - rData[ENG_NFS_OFFS]) * mult / cntRange;
            module.offset[j] = (rData[ENG_PFS_OFFS] * rData[CNT_NFS_OFFS] - rData[ENG_NFS_OFFS] * rData[CNT_PFS_OFFS])
                                 * mult / cntRange;
        }
    }


   /**
    *  Gets module data.
    *
    *  @param  modId   The module ID
    *  @param  zeroOk  True if 0 is a valid module ID
    *  @throws  DriverException if module ID is invalid
    */
    ModuleData getModuleData(int modId, boolean zeroOk) throws DriverException
    {
        ModuleData module = null;
        if (modId >= (zeroOk ? 0 : 1) && modId < NUM_MODULES) {
            module = modules[modId];
        }
        if (module == null) {
            throw new DriverException("Invalid module ID");
        }
        return module;
    }


   /**
    *  Converts an array of shorts to a string.
    *
    *  @param  array  The array of shorts
    *  @return  The created string
    */
    private static String makeString(short[] array)
    {
        int leng;
        for (leng = array.length; leng > 0; leng--) {
            char elem = (char)array[leng - 1];
            if (elem != '\0' && elem != ' ' && elem != '\uffff') break;
        }
        char[] conv = new char[leng];
        for (int j = 0; j < leng; j++) {
            conv[j] = (char)array[j];
        }
        return new String(conv);
    }


   /**
    *  Converts an array of shorts to an IP address string.
    *
    *  @param  array  The array of shorts
    *  @return  The created string
    */
    private static String makeIPString(short[] array)
    {
        return String.format("%s.%s.%s.%s", array[0], array[1], array[2], array[3]);
    }


   /**
    *  Converts an IP address string to an array of shorts.
    *
    *  @param  ipAddr  The IP address string
    *  @return  The created 4-element array
    */
    private static short[] makeIPAddress(String ipAddr) throws DriverException
    {
        String[] elems = ipAddr.split("\\.");
        if (elems.length == IP_ADDRESS_LENG) {
            short[] array = new short[IP_ADDRESS_LENG];
            try {
                for (int j = 0; j < IP_ADDRESS_LENG; j++) {
                    array[j] = Short.valueOf(elems[j]);
                }
                return array;
            }
            catch (NumberFormatException e) {
            }
        }
        throw new DriverException("Invalid IP address string");
    }

}
