package org.lsst.ccs.drivers.dataforth;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.modbus.TestModbus;

/**
 *************************************************************************** 
 ** 
 **  Program to test the Dataforth MAQ20 device driver
 ** 
 **  @author Owen Saxton
 ** 
 *************************************************************************** 
 */
public class TestMaq20 extends TestModbus {

    private final Maq20 maq;

    //public enum OnOff {
    //    OFF, ON;
    //}


   /**
    *************************************************************************** 
    ** 
    **  Constructor
    ** 
    *************************************************************************** 
    */
    public TestMaq20()
    {
        super(new Maq20());
        maq = (Maq20)mod;
    }


   /**
    *************************************************************************** 
    ** 
    **  Registers a set of modules.
    ** 
    **  @param  serial  The serial numbers of the modules to be registered
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="register", description="Register modules")
    public void register(@Argument(name="serial", description="Serial numbers")
                         String... serial) throws DriverException
    {
        maq.register(serial);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the IP address.
    ** 
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showipaddress", description="Show the IP address")
    public void showIPAddress() throws DriverException
    {
        out.println("IP address = " + maq.getIPAddress());
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the IP address.
    **
    **  @param  ipAddr  The IP address string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="setipaddress", description="Set the IP address")
    public void setIPAddress(@Argument(name="ipaddr", description="IP address")
                             String ipAddr) throws DriverException
    {
        maq.setIPAddress(ipAddr);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the subnet mask.
    ** 
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showsubnet", description="Show the subnet mask")
    public void showSubnetMask() throws DriverException
    {
        out.println("Subnet mask = " + maq.getSubnetMask());
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the subnet mask.
    **
    **  @param  mask  The subnet mask
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="setsubnet", description="Set the subnet mask")
    public void setSubnetMask(@Argument(name="mask", description="subnet mask")
                              String mask) throws DriverException
    {
        maq.setSubnetMask(mask);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the temperature.
    ** 
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showtemperature", description="Show the temperature")
    public void showTemperature() throws DriverException
    {
        out.println("Temperature = " + maq.readTemperature());
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows a module's ID given it's serial number.
    ** 
    **  @param  serial  The module's serial number
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showid", description="Show module's ID")
    public void showId(@Argument(name="serial", description="Serial number")
                       String serial) throws DriverException
    {
        out.println("Module ID = " + maq.getModuleId(serial));
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows a module's name.
    ** 
    **  @param  modId  The module ID
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showname", description="Show module's name")
    public void showName(@Argument(name="modid", description="Module ID")
                         int modId) throws DriverException
    {
        out.println("Device name = " + maq.getModuleName(modId));
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows a module's serial number.
    ** 
    **  @param  modId  The module ID
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showserial", description="Show module's serial number")
    public void showSerial(@Argument(name="modid", description="Module ID")
                           int modId) throws DriverException
    {
        out.println("Serial number = " + maq.getSerialNumber(modId));
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows a module's firmware revision number.
    ** 
    **  @param  modId  The module ID
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showrevision", description="Show module's FW revision")
    public void showFwRevision(@Argument(name="modid", description="Module ID")
                               int modId) throws DriverException
    {
        out.println("Firmware revision = " + maq.getFwRevision(modId));
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows a module's date code.
    ** 
    **  @param  modId  The module ID
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showdate", description="Show module's date code")
    public void showDate(@Argument(name="modid", description="Module ID")
                         int modId) throws DriverException
    {
        out.println("Date code = " + maq.getDateCode(modId));
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows a channel's range number.
    ** 
    **  @param  modId  The module ID
    **
    **  @param  chan   The channel number
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showrange", description="Show channel's range")
    public void showRange(@Argument(name="modid", description="Module ID")
                          int modId,
                          @Argument(name="chan", description="Channel number")
                          int chan) throws DriverException
    {
        out.println("Range = " + maq.getRange(modId, chan));
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets a channel's range number.
    ** 
    **  @param  modId  The module ID
    **
    **  @param  chan   The channel number
    **
    **  @param  range  The range number
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="setrange", description="Set channel's range")
    public void setRange(@Argument(name="modid", description="Module ID")
                         int modId,
                         @Argument(name="chan", description="Channel number")
                         int chan,
                         @Argument(name="range", description="Range number")
                         int range) throws DriverException
    {
        maq.setRange(modId, chan, range);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows a channel's enabled state.
    ** 
    **  @param  modId  The module ID
    **
    **  @param  chan   The channel number
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showenabled", description="Show channel's enabled state")
    public void showEnabled(@Argument(name="modid", description="Module ID")
                            int modId,
                            @Argument(name="chan", description="Channel number")
                            int chan) throws DriverException
    {
        out.println("Enabled state = " + (maq.isEnabled(modId, chan) ? "on" : "off"));
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets a channel's enabled state.
    ** 
    **  @param  modId  The module ID
    **
    **  @param  chan   The channel number
    **
    **  @param  state  The enabled state
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="setenabled", description="Set channel's enabled state")
    public void setEnabled(@Argument(name="modid", description="Module ID")
                           int modId,
                           @Argument(name="chan", description="Channel number")
                           int chan,
                           @Argument(name="state", description="Enabled state")
                           OnOff state) throws DriverException
    {
        maq.enable(modId, chan, state == OnOff.ON);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows selected channels' data values.
    ** 
    **  @param  modId  The module ID
    **
    **  @param  chan   The first channel number
    **
    **  @param  count  The number of channels
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="readvalue", description="Show channels' data values")
    public void readValue(@Argument(name="modid", description="Module ID")
                          int modId,
                          @Argument(name="chan", description="Channel number")
                          int chan,
                          @Argument(name="count", description="Number of channels")
                          int count) throws DriverException
    {
        showValues(maq.readValue(modId, chan, count));
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows one channel's data value.
    ** 
    **  @param  modId  The module ID
    **
    **  @param  chan   The channel number
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="readvalue", description="Show channel's data value")
    public void readValue(@Argument(name="modid", description="Module ID")
                          int modId,
                          @Argument(name="chan", description="Channel number")
                          int chan) throws DriverException
    {
        showValues(new double[]{maq.readValue(modId, chan)});
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows all channels' data values.
    ** 
    **  @param  modId  The module ID
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="readvalue", description="Show all channels' data values")
    public void readValue(@Argument(name="modid", description="Module ID")
                          int modId) throws DriverException
    {
        showValues(maq.readValue(modId));
    }


   /**
    *************************************************************************** 
    ** 
    **  Displays an array of data values.
    ** 
    **  @param  values  The array of values
    **
    *************************************************************************** 
    */
    private void showValues(double[] values)
    {
        out.print(values.length > 1 ? "Values =" : "Value =");
        for (double value : values) {
            out.format(" %.6g", value);
        }
        out.println();
    }

}
