package org.lsst.ccs.drivers.dataforth;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.drivers.modbus.Modbus;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;

/**
 ******************************************************************************
 **
 **  Routines for controlling a Dataforth MAQ20 DAQ system.
 **
 **  @author  Owen Saxton
 **
 ******************************************************************************
 */
public class Maq20 extends Modbus {

   /**
    ***************************************************************************
    **
    **  Inner class for storing module data.
    **
    ***************************************************************************
    */
    static class ModuleData {

        int type;
        int numChan;
        int[] range = new int[NUM_CHANNELS];
        double[] offset;
        double[] scale;

    }

   /**
    **************************************************************************
    **
    **  Public constants.
    **
    **************************************************************************
    */
    public final static int
        NUM_MODULES      = 24,
        MOD_TYPE_UNKNOWN = -1,
        MOD_TYPE_JTC     = 0,
        MOD_TYPE_KTC     = 1,
        MOD_TYPE_TTC     = 2,
        MOD_TYPE_RSTC    = 3,
        MOD_TYPE_RTD     = 4,
        MOD_TYPE_MVD     = 5,
        MOD_TYPE_VD      = 6,
        MOD_TYPE_VS      = 7,
        MOD_TYPE_ID      = 8,
        MOD_TYPE_IS      = 9,
        RANGE_TC_T_400   = 0,
        RANGE_TC_T_220   = 1,
        RANGE_TC_J_760   = 0,
        RANGE_TC_J_393   = 1,
        RANGE_TC_J_199   = 2,
        RANGE_TC_K_1350  = 0,
        RANGE_TC_K_651   = 1,
        RANGE_TC_K_332   = 2,
        RANGE_TC_R_1750  = 0,
        RANGE_TC_R_990   = 1,
        RANGE_TC_S_1750  = 2,
        RANGE_TC_S_970   = 3,
        RANGE_RTD_850    = 0,
        RANGE_RTD_200    = 1,
        RANGE_RTD_100    = 2,
        RANGE_VOLT_60    = 0,
        RANGE_VOLT_40    = 1,
        RANGE_VOLT_20    = 2,
        RANGE_VOLT_10    = 3,
        RANGE_VOLT_5     = 4,
        RANGE_MVOLT_2000 = 0,
        RANGE_MVOLT_1000 = 1,
        RANGE_MVOLT_250  = 2,
        RANGE_MVOLT_100  = 3,
        RANGE_MVOLT_50   = 4,
        RANGE_MAMP_0_20  = 0,
        RANGE_MAMP_4_20  = 1;

   /**
    **************************************************************************
    **
    **  Private data.
    **
    **************************************************************************
    */
    private final static short
        ZERO = 0,
        ONE = 1,
        SLAVE_ADDR = 0,
        MODU_ADDR_INCR = 2000,

        DEV_NAME_ADDR = 0,
        DEV_NAME_LENG = 10,
        SER_NUM_ADDR = 20,
        SER_NUM_LENG = 10,
        DATE_CODE_ADDR = 31,
        DATE_CODE_LENG = 4,
        FW_REVN_ADDR = 36,
        FW_REVN_LENG = 4,

        IP_ADDRESS_ADDR = 50,
        IP_ADDRESS_LENG = 4,
        SUBNET_MASK_ADDR = 55,
        MODU_DETECT_ADDR = 98,
        RESET_ADDR = 99,
        MODU_STATUS_ADDR = 100,
        REGN_SN_ID_ADDR = 1000,
        AUTO_REGN_ADDR = 1020,
        SAVE_REGN_ADDR = 1021,
        DEL_REGN_ADDR = 1022,
        TEMPERATURE_ADDR = 1210,

        NUM_INP_CHANS_ADDR = 40,
        NUM_OUT_CHANS_ADDR = 41,
        INPUT_RANGE_ADDR = 100,
        CHAN_WEIGHT_ADDR = 120,
        CHAN_ENABLE_ADDR = 140,
        CHAN_DATA_ADDR = 1000,
        RANGE_COUNT_ADDR = 1700,
        RANGE_DATA_ADDR = 1710,
        RANGE_DATA_INCR = 20,
        ENG_NFS_OFFS = 0,
        ENG_PFS_OFFS = 2,
        ENG_FS_PWR_OFFS = 4,
        CNT_NFS_OFFS = 8,
        CNT_PFS_OFFS = 10,
        NUM_CHANNELS = 16;

    private static final Map<String, Integer> typeMap = new HashMap<>();
    static {
        typeMap.put("JTC", MOD_TYPE_JTC);
        typeMap.put("KTC", MOD_TYPE_KTC);
        typeMap.put("TTC", MOD_TYPE_TTC);
        typeMap.put("RSTC", MOD_TYPE_RSTC);
        typeMap.put("RTD31", MOD_TYPE_RTD);
        typeMap.put("MVDN", MOD_TYPE_MVD);
        typeMap.put("VDN", MOD_TYPE_VD);
        typeMap.put("VDS", MOD_TYPE_VS);
        typeMap.put("IDN", MOD_TYPE_ID);
        typeMap.put("ISN", MOD_TYPE_IS);
    }
    private final ModuleData[] modules = new ModuleData[NUM_MODULES];


   /**
    **************************************************************************
    **
    **  Opens a connection to the device.
    **
    **  @param  type   The type of connection to make
    **
    **  @param  ident  The device identifier:
    **                   host name or IP address for network;
    **                   serial number for FTDI device;
    **
    **  @param  parm   The device parameter:
    **                   port number for network;
    **                   baud rate for FTDI
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    @Override
    public void open(int type, String ident, int parm) throws DriverException
    {
        super.open(type, ident, parm);
        setAddressMode(true);
        try {
            fillModuleData();
        }
        catch (DriverException e) {
            try {
                close();
            }
            catch (DriverException ex) {
            }
            throw e;
        }
    }


   /**
    ***************************************************************************
    **
    **  Tests whether a module exists.
    **
    **  @param  modId  The module registration ID
    **
    **  @return  Whether the module exists
    **
    ***************************************************************************
    */
    public boolean moduleExists(int modId)
    {
        return modId > 0 && modId < NUM_MODULES && modules[modId] != null;
    }


   /**
    ***************************************************************************
    **
    **  Gets the module name.
    **
    **  @param  modId  The module registration ID
    **
    **  @return  The module name
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public String getModuleName(int modId) throws DriverException
    {
        checkModuleId(modId, true);
        short addr = (short)(MODU_ADDR_INCR * modId + DEV_NAME_ADDR);
        short[] text = readRegisters(SLAVE_ADDR, addr, DEV_NAME_LENG);
        return makeString(text);
    }


   /**
    ***************************************************************************
    **
    **  Gets the module type.
    **
    **  @param  modId  The module registration ID
    **
    **  @return  The module type
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public int getModuleType(int modId) throws DriverException
    {
        checkModuleId(modId, true);
        return modules[modId].type;
    }


   /**
    ***************************************************************************
    **
    **  Gets the serial number.
    **
    **  @param  modId  The module registration ID
    **
    **  @return  The serial number
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public String getSerialNumber(int modId) throws DriverException
    {
        checkModuleId(modId, true);
        return getSerialNum(modId);
    }


   /**
    ***************************************************************************
    **
    **  Gets the date code.
    **
    **  @param  modId  The module registration ID
    **
    **  @return  The date code
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public String getDateCode(int modId) throws DriverException
    {
        checkModuleId(modId, true);
        short addr = (short)(MODU_ADDR_INCR * modId + DATE_CODE_ADDR);
        short[] text = readRegisters(SLAVE_ADDR, addr, DATE_CODE_LENG);
        return makeString(text);
    }


   /**
    ***************************************************************************
    **
    **  Gets the firmware revision number.
    **
    **  @param  modId  The module registration ID
    **
    **  @return  The firmware revision number
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public String getFwRevision(int modId) throws DriverException
    {
        checkModuleId(modId, true);
        short addr = (short)(MODU_ADDR_INCR * modId + FW_REVN_ADDR);
        short[] text = readRegisters(SLAVE_ADDR, addr, FW_REVN_LENG);
        return makeString(text);
    }


   /**
    ***************************************************************************
    **
    **  Gets the IP address.
    **
    **  @return  The IP address, as a string
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public String getIPAddress() throws DriverException
    {
        short[] addr = readRegisters(SLAVE_ADDR, IP_ADDRESS_ADDR,
                                     IP_ADDRESS_LENG);
        return makeIPString(addr);
    }


   /**
    ***************************************************************************
    **
    **  Sets the IP address.
    **
    **  @param  ipAddr  The IP address, as an IP string
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public void setIPAddress(String ipAddr) throws DriverException
    {
        writeRegisters(SLAVE_ADDR, IP_ADDRESS_ADDR, makeIPAddress(ipAddr));
    }


   /**
    ***************************************************************************
    **
    **  Gets the subnet mask.
    **
    **  @return  The subnet mask, as a string
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public String getSubnetMask() throws DriverException
    {
        short[] addr = readRegisters(SLAVE_ADDR, SUBNET_MASK_ADDR,
                                     IP_ADDRESS_LENG);
        return makeIPString(addr);
    }


   /**
    ***************************************************************************
    **
    **  Sets the subnet mask.
    **
    **  @param  mask  The subnet mask, as an IP string
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public void setSubnetMask(String mask) throws DriverException
    {
        writeRegisters(SLAVE_ADDR, SUBNET_MASK_ADDR, makeIPAddress(mask));
    }


   /**
    ***************************************************************************
    **
    **  Reads the board temperature.
    **
    **  @return  The board temperature
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public double readTemperature() throws DriverException
    {
        return readRegisters(SLAVE_ADDR, TEMPERATURE_ADDR, ONE)[0];
    }


   /**
    ***************************************************************************
    **
    **  Register the modules.
    **
    **  @param  serial  The serial numbers of all the modules, in the desired
    **                  order
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public void register(String... serial) throws DriverException
    {
        List<Short> ids = new ArrayList<>();
        for (short j = 1; j < NUM_MODULES; j++) {
            if (modules[j] != null) {
                ids.add(j);
            }
        }
        if (serial.length != ids.size()) {
            throw new DriverException("Incorrect number of modules");
        }
        for (String sn : serial) {
            if (sn.length() > SER_NUM_LENG) {
                throw new DriverException("Serial number (" + sn
                                            + ") too long");
            }
            if (getModuleId(sn) < 0) {
                throw new DriverException("Serial number (" + sn
                                            + ") not found");
            }
        }
        writeRegister(SLAVE_ADDR, AUTO_REGN_ADDR, ZERO);
        for (short id : ids) {
            writeRegister(SLAVE_ADDR, DEL_REGN_ADDR, id);
        }
        short[] snId = new short[SER_NUM_LENG + 1];
        for (String sn : serial) {
            for (int j = 0; j < SER_NUM_LENG; j++) {
                if (j < sn.length()) {
                    snId[j] = (short)sn.charAt(j);
                }
                else {
                    snId[j] = (short)' ';
                }
            }
            snId[SER_NUM_LENG]++;
            int oldTimeout = timeout;   // **** Kludge
            timeout = 100;              // ****
            try {                       // ****
                writeRegisters(SLAVE_ADDR, REGN_SN_ID_ADDR, snId);
            }                           // ****
            catch (DriverTimeoutException e) {  // ****
            }                                   // ****
            finally {                   // ****
                timeout = oldTimeout;   // ****
            }                           // ****
        }
        writeRegister(SLAVE_ADDR, SAVE_REGN_ADDR, ONE);
        writeRegister(SLAVE_ADDR, AUTO_REGN_ADDR, ONE);
        fillModuleData();
    }


   /**
    ***************************************************************************
    **
    **  Gets a module's ID.
    **
    **  @param  serial  The module's serial number
    **
    **  @return  The module's registration ID, or -1 if not found
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public int getModuleId(String serial) throws DriverException
    {
        short[] status = readRegisters(SLAVE_ADDR, MODU_STATUS_ADDR,
                                       (short)NUM_MODULES);
        for (int id = 1; id < NUM_MODULES; id++) {
            if (status[id] != 0 && serial.equals(getSerialNum(id))) return id;
        }
        return -1;
    }


   /**
    ***************************************************************************
    **
    **  Gets the number of input channels for a module.
    **
    **  @param  modId  The module registration ID
    **
    **  @return  The number of input channels
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public int getNumInputs(int modId) throws DriverException
    {
        checkModuleId(modId, true);
        return modules[modId].numChan;
    }


   /**
    ***************************************************************************
    **
    **  Gets the number of ranges for a module.
    **
    **  @param  modId  The module registration ID
    **
    **  @return  The number of ranges
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public int getNumRanges(int modId) throws DriverException
    {
        checkModuleId(modId, true);
        return modules[modId].offset.length;
    }


   /**
    ***************************************************************************
    **
    **  Gets the range for a channel.
    **
    **  @param  modId  The module registration ID
    **
    **  @param  chan   The channel number
    **
    **  @return  The range number
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public int getRange(int modId, int chan) throws DriverException
    {
        checkModuleId(modId, false);
        checkChannel(modId, chan);
        checkOpen();
        return modules[modId].range[chan];
    }


   /**
    ***************************************************************************
    **
    **  Sets the range for a channel.
    **
    **  @param  modId  The module registration ID
    **
    **  @param  chan   The channel number
    **
    **  @param  range  The range number
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public void setRange(int modId, int chan, int range) throws DriverException
    {
        checkModuleId(modId, false);
        checkChannel(modId, chan);
        ModuleData module = modules[modId];
        if (range < 0 || range >= module.offset.length) {
            throw new DriverException("Invalid range value");
        }
        short addr = (short)(MODU_ADDR_INCR * modId + INPUT_RANGE_ADDR + chan);
        writeRegister(SLAVE_ADDR, addr, (short)range);
        module.range[chan] = range;
    }


   /**
    ***************************************************************************
    **
    **  Gets the enabled state of a channel.
    **
    **  @param  modId  The module registration ID
    **
    **  @param  chan   The channel number
    **
    **  @return  Whether the channel is enabled
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public boolean isEnabled(int modId, int chan) throws DriverException
    {
        checkModuleId(modId, false);
        checkChannel(modId, chan);
        short addr = (short)(MODU_ADDR_INCR * modId + CHAN_ENABLE_ADDR + chan);
        return readRegisters(SLAVE_ADDR, addr, ONE)[0] != 0;
    }


   /**
    ***************************************************************************
    **
    **  Enables or disables a channel.
    **
    **  @param  modId  The module registration ID
    **
    **  @param  chan   The channel number
    **
    **  @param  state  The enabled state to set: true or false
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public void enable(int modId, int chan, boolean state) throws DriverException
    {
        checkModuleId(modId, false);
        checkChannel(modId, chan);
        short addr = (short)(MODU_ADDR_INCR * modId + CHAN_ENABLE_ADDR + chan);
        writeRegister(SLAVE_ADDR, addr, state ? ONE : ZERO);
    }


   /**
    ***************************************************************************
    **
    **  Reads the data values for several channels.
    **
    **  @param  modId  The module registration ID
    **
    **  @param  chan   The first channel number
    **
    **  @param  count  The number of channels
    **
    **  @return  The array of data values
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public double[] readValue(int modId, int chan, int count)
        throws DriverException
    {
        checkModuleId(modId, false);
        checkChannel(modId, chan);
        ModuleData module = modules[modId];
        if (count <= 0 || chan + count > module.numChan) {
            throw new DriverException("Invalid channel count");
        }
        short addr = (short)(MODU_ADDR_INCR * modId + CHAN_DATA_ADDR + chan);
        short[] raw = readRegisters(SLAVE_ADDR, addr, (short)count);
        double[] data = new double[count];
        for (int j = 0; j < count; j++, chan++) {
            int range = module.range[chan];
            data[j] = module.scale[range] * raw[j] + module.offset[range];
        }
        return data;
    }


   /**
    ***************************************************************************
    **
    **  Reads the data value for a channel.
    **
    **  @param  modId  The module registration ID
    **
    **  @param  chan   The channel number
    **
    **  @return  The data value
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public double readValue(int modId, int chan) throws DriverException
    {
        return readValue(modId, chan, 1)[0];
    }


   /**
    ***************************************************************************
    **
    **  Reads the data values for all channels.
    **
    **  @param  modId  The module registration ID
    **
    **  @return  The array of data values
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    public double[] readValue(int modId) throws DriverException
    {
        checkModuleId(modId, false);
        return readValue(modId, 0, modules[modId].numChan);
    }


   /**
    ***************************************************************************
    **
    **  Gets the serial number without checking the module ID.
    **
    **  @param  modId  The module registration ID
    **
    **  @return  The serial number
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    private String getSerialNum(int modId) throws DriverException
    {
        short addr = (short)(MODU_ADDR_INCR * modId + SER_NUM_ADDR);
        short[] text = readRegisters(SLAVE_ADDR, addr, SER_NUM_LENG);
        return makeString(text);
    }


   /**
    ***************************************************************************
    **
    **  Gets saved module data.
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    private void fillModuleData() throws DriverException
    {
        short[] status = readRegisters(SLAVE_ADDR, MODU_STATUS_ADDR,
                                       (short)NUM_MODULES);
        for (int id = 1; id < NUM_MODULES; id++) {
            if (status[id] != 0) {
                if (modules[id] == null) {
                    modules[id] = new ModuleData();
                }
                ModuleData module = modules[id];
                module.type = MOD_TYPE_UNKNOWN;
                String name = getModuleName(id);
                if (name.substring(0, 6).equals("MAQ20-")) {
                    Integer type = typeMap.get(name.substring(6));
                    if (type != null) {
                        module.type = type;
                    }
                }
                short addr = (short)(MODU_ADDR_INCR * id + NUM_INP_CHANS_ADDR);
                module.numChan = readRegisters(SLAVE_ADDR, addr, ONE)[0];
                for (int chan = 0; chan < module.numChan; chan++) {
                    module.range[chan] = readRange(id, chan);
                }
                setConversion(id);
            }
            else {
                modules[id] = null;
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Sets conversion constants for a module.
    **
    **  @param  modId  The module ID
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    private void setConversion(int modId) throws DriverException
    {
        ModuleData module = modules[modId];
        short addr = (short)(MODU_ADDR_INCR * modId + RANGE_COUNT_ADDR);
        int nRange = readRegisters(SLAVE_ADDR, addr, ONE)[0];
        module.offset = new double[nRange];
        module.scale = new double[nRange];
        for (int j = 0; j < nRange; j++) {
            addr = (short)(MODU_ADDR_INCR * modId + RANGE_DATA_ADDR
                             + j * RANGE_DATA_INCR);
            short[] rData = readRegisters(SLAVE_ADDR, addr, RANGE_DATA_INCR);
            double cntRange = rData[CNT_PFS_OFFS] - rData[CNT_NFS_OFFS];
            double mult = Math.pow(10, rData[ENG_FS_PWR_OFFS]);
            module.scale[j] = (rData[ENG_PFS_OFFS] - rData[ENG_NFS_OFFS])
                                 * mult / cntRange;
            module.offset[j] = (rData[ENG_PFS_OFFS] * rData[CNT_NFS_OFFS]
                                 - rData[ENG_NFS_OFFS] * rData[CNT_PFS_OFFS])
                                 * mult / cntRange;
        }
    }


   /**
    ***************************************************************************
    **
    **  Reads the range for a channel.
    **
    **  @param  modId  The module registration ID
    **
    **  @param  chan   The channel number
    **
    **  @return  The range number
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    private int readRange(int modId, int chan) throws DriverException
    {
        short addr = (short)(MODU_ADDR_INCR * modId + INPUT_RANGE_ADDR + chan);
        return readRegisters(SLAVE_ADDR, addr, ONE)[0];
    }


   /**
    ***************************************************************************
    **
    **  Checks a module ID for validity.
    **
    **  @param  modId   The module ID
    **
    **  @param  zeroOk  True if 0 is a valid value
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    private void checkModuleId(int modId, boolean zeroOk) throws DriverException
    {
        if (zeroOk && modId == 0) return;
        if (!moduleExists(modId)) {
            throw new DriverException("Invalid module ID");
        }
    }


   /**
    ***************************************************************************
    **
    **  Checks a channel number for validity.
    **
    **  @param  modId  The module ID
    **
    **  @param  chan   The channel number
    **
    **  @throws  DriverException
    **
    ***************************************************************************
    */
    private void checkChannel(int modId, int chan) throws DriverException
    {
        if (chan < 0 || chan >= modules[modId].numChan) {
            throw new DriverException("Invalid channel number");
        }
    }


   /**
    ***************************************************************************
    **
    **  Converts an array of shorts to a string.
    **
    **  @param  array  The array of shorts
    **
    **  @return  The created string
    **
    ***************************************************************************
    */
    private static String makeString(short[] array)
    {
        int leng;
        for (leng = array.length; leng > 0; leng--) {
            char elem = (char)array[leng - 1];
            if (elem != '\0' && elem != ' ') break;
        }
        char[] conv = new char[leng];
        for (int j = 0; j < leng; j++) {
            conv[j] = (char)array[j];
        }
        return new String(conv);
    }


   /**
    ***************************************************************************
    **
    **  Converts an array of shorts to an IP address string.
    **
    **  @param  array  The array of shorts
    **
    **  @return  The created string
    **
    ***************************************************************************
    */
    private static String makeIPString(short[] array)
    {
        return String.format("%s.%s.%s.%s",
                             array[0], array[1], array[2], array[3]);
    }


   /**
    ***************************************************************************
    **
    **  Converts an IP address string to an array of shorts.
    **
    **  @param  ipAddr  The IP address string
    **
    **  @return  The created 4-element array
    **
    ***************************************************************************
    */
    private static short[] makeIPAddress(String ipAddr) throws DriverException
    {
        String[] elems = ipAddr.split("\\.");
        if (elems.length == IP_ADDRESS_LENG) {
            short[] array = new short[IP_ADDRESS_LENG];
            try {
                for (int j = 0; j < IP_ADDRESS_LENG; j++) {
                    array[j] = Short.valueOf(elems[j]);
                }
                return array;
            }
            catch (NumberFormatException e) {
            }
        }
        throw new DriverException("Invalid IP address string");
    }

}
