package org.lsst.ccs.drivers.cryocon;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.scpi.Scpi;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *
 * Enables access to a Cryogenic Temperature controller Model 24C
 *
 * @author Heather Kelly
 *
 */

/*
 * TODO define a common interface for temperature controllers. An issue is
 * temperature, which can be given in K, F, C, or sensor units depending on
 * controllers : how to unify?
 */
public class M24C extends Scpi {

    /**
     * Public constants
     */
    public static final int NUM_LOOPS = 4;

    /**
     * Private constants
     */
    private static final int DEFAULT_PORT = 5000, DEFAULT_BAUD = 9600;

    /**
     * Private fields
     */
    private static final Map<String, Integer> typeMap = new HashMap<>();
    static {
        typeMap.put("DIODE", 5);
        typeMap.put("PTC100", 20);
        typeMap.put("PTC1K", 21);
        typeMap.put("NONE", 0);
        typeMap.put("INTERNAL", 59);  // ???
        typeMap.put("DIODE1", 1);
        typeMap.put("DIODE2", 2);
        typeMap.put("DIODE3", 3);
        typeMap.put("DIODE4", 4);
        typeMap.put("TC70K", 45);
        typeMap.put("TC70E", 46);
        typeMap.put("TC70T", 47);
        typeMap.put("TC70AF", 48);
        typeMap.put("NTC10UA", 31);
        typeMap.put("ACR", 32);
    }
    private static final Map<Integer, String> invTypeMap = new HashMap<>();
    static {
        for (String type : typeMap.keySet()) {
            invTypeMap.put(typeMap.get(type), type);
        }
    }
    private final String[] rangeCheck = {"HI|MID|LOW", "HI|LOW", "10V|5V", "10V|5V"};
    private final Logger log = Logger.getLogger("lsst.ccs.drivers.cryocon");


    /**
     *  Opens a connection to the device.
     *
     *  This is the open method that overrides the superclass open when extra
     *  functionality is needed.
     *
     *  @param  type   The enumerated type of connection to make
     *  @param  ident  The device identifier:
     *                   host name or IP address for network;
     *                   device name for serial
     *  @param  parm1  The first device parameter:
     *                   port number for network;
     *                   baud rate for serial
     *  @param  parm2  The second device parameter:
     *                   unused for network;
     *                   encoded data characteristics for serial:
     *                     0 sets 8-bit, no parity, 1 stop bit, no flow ctrl
     *  @throws  DriverException
     */
    @Override
    public void open(ConnType type, String ident, int parm1, int parm2) throws DriverException {
        super.open(type, ident, parm1 == 0 ? type == ConnType.NET ? DEFAULT_PORT : DEFAULT_BAUD : parm1, parm2);
        try {
            checkIdentification("Cryo-con", CHECK_EQUALS, "24C", CHECK_CONTAINS);
        }
        catch (DriverException e) {
            closeSilent();
            throw e;
        }
        log.debug("Opened Cryo-con M24C temperature controller");
    }


    /**
     * Gets the maximum set point for a loop
     *
     * @param loop The loop number (1-4)
     * @return The maximum set point
     * @throws DriverException
     */
    public double getMaxSetPoint(int loop) throws DriverException {
        checkLoop(loop);
        return Double.parseDouble(readString("LOOP " + loop + ":MAXSET?").replaceAll("[a-zA-Z]", ""));
    }


    /**
     * Sets the maximum set point for a loop
     *
     * @param loop The loop number (1-4)
     * @param temp The temperature to set in the units specified via setUnit
     * @throws DriverException
     */
    public void setMaxSetPoint(int loop, double temp) throws DriverException {
        checkLoop(loop);
        writeCommand("LOOP " + loop + ":MAXSET " + temp);
    }


    /**
     * Gets the current set point for a loop
     *
     * @param loop The loop number (1-4)
     * @return The current set point
     * @throws DriverException
     */
    public double getSetPoint(int loop) throws DriverException {
        checkLoop(loop);
        return Double.parseDouble(readString("LOOP " + loop + ":SETPT?").replaceAll("[a-zA-Z]", ""));
    }


    /**
     * Sets the set point for a loop
     *
     * @param loop The loop number (1-4)
     * @param temp The temperature to set in the units specified via setUnit
     * @throws DriverException
     */
    public void setSetPoint(int loop, double temp) throws DriverException {
        checkLoop(loop);
        if (temp > getMaxSetPoint(loop)) {
            throw new DriverException("Temperature is above maximum set point: " + temp);
        }
        writeCommand("LOOP " + loop + ":SETPT " + temp);
    }


    /**
     * Gets the heater power level for a loop
     *
     * @param loop The loop number (1-4)
     * @return % of full power
     * @throws DriverException
     */
    public double getHtrRead(int loop) throws DriverException {
        checkLoop(loop);
        return(Double.parseDouble(readString("LOOP " + loop + ":HTRR?").replace("%", "")));
    }


    /**
     * Sets the PID P parameter for a loop.
     * 
     * @param loop The loop number (1-4)
     * @param pValue The P value to set
     * @throws DriverException 
     */
    public void setPidP(int loop, double pValue) throws DriverException {
        checkLoop(loop);
        writeCommand("LOOP " + loop + ":PGAIN " + pValue);
    }


    /**
     * Gets the PID P parameter for a loop
     *
     * @param loop The loop number (1-4)
     * @return The P parameter
     * @throws DriverException
     */
    public double getPidP(int loop) throws DriverException {
        checkLoop(loop);
        return readDouble("LOOP " + loop + ":PGAIN?");
    }

    @Deprecated
    public double getPID_P(int loop) throws DriverException {
        return getPidP(loop);
    }


    /**
     * Sets the PID I parameter for a loop.
     * 
     * @param loop The loop number (1-4)
     * @param iValue The I value to set
     * @throws DriverException 
     */
    public void setPidI(int loop, double iValue) throws DriverException {
        checkLoop(loop);
        writeCommand("LOOP " + loop + ":IGAIN " + iValue);
    }


    /**
     * Gets the PID I parameter for a loop
     *
     * @param loop The loop number (1-4)
     * @return The I parameter
     * @throws DriverException
     */
    public double getPidI(int loop) throws DriverException {
        checkLoop(loop);
        return readDouble("LOOP " + loop + ":IGAIN?");
    }

    @Deprecated
    public double getPID_I(int loop) throws DriverException {
        return getPidI(loop);
    }


    /**
     * Sets the PID D parameter for a loop.
     * 
     * @param loop The loop number (1-4)
     * @param dValue The D value to set
     * @throws DriverException 
     */
    public void setPidD(int loop, double dValue) throws DriverException {
        checkLoop(loop);
        writeCommand("LOOP " + loop + ":DGAIN " + dValue);
    }


    /**
     * Gets the PID D parameter for a loop
     *
     * @param loop The loop number (1-4)
     * @return The D parameter
     * @throws DriverException
     */
    public double getPidD(int loop) throws DriverException {
        checkLoop(loop);
        return readDouble("LOOP " + loop + ":DGAIN?");
    }

    @Deprecated
    public double getPID_D(int loop) throws DriverException {
        return getPidD(loop);
    }


    /**
     * Gets the channel source for a loop
     *
     * @param loop The loop number (1-4)
     * @return channel [A,D]
     * @throws DriverException
     */
    public char getSource(int loop) throws DriverException {
        checkLoop(loop);
        return readString("LOOP " + loop + ":SOURCE?").charAt(2);  // Reply is "CHA", etc
    }


    /**
     * Sets the channel source for a loop
     *
     * @param loop The loop number (1-4)
     * @param channel The channel ID (A-D)
     * @throws DriverException
     */
    public void setSource(int loop, char channel) throws DriverException {
        checkLoop(loop);
        checkChannel(channel);
        writeCommand("LOOP " + loop + ":SOURCE " + channel);
    }


    /**
     * Gets the control type for a loop
     * 
     * @param loop The loop number (1-4)
     * @return The type string: OFF, PID, MAN, TABLE. RAMPP or RAMPT
     * @throws DriverException
     */
    public String getControlType(int loop) throws DriverException {
        checkLoop(loop);
        return readString("LOOP " + loop + ":TYPE?");
    }

    
    /**
     * Sets the control type for a loop
     * 
     * @param loop The loop number (1-4)
     * @param type The type string: OFF, PID, MAN, TABLE. RAMPP or RAMPT
     * @throws DriverException
     */
    public void setControlType(int loop, String type) throws DriverException {
        checkLoop(loop);
        String check = "OFF|PID|MAN|TABLE|RAMPP|RAMPT";
        if (!type.toUpperCase().matches(check)) {
            throw new DriverException("Invalid control type (" + type + "): must match " + check);
        }
        writeCommand("LOOP " + loop + ":TYPE " + type);
    }

    
    /**
     * Gets the heater range for a loop
     * 
     * @param loop The loop number (1-4)
     * @return The range string: HI, MID, LOW, 10V or 5V
     * @throws DriverException
     */
    public String getHeaterRange(int loop) throws DriverException {
        checkLoop(loop);
        return readString("LOOP " + loop + ":RANGE?");
    }

    
    /**
     * Sets the heater range for a loop
     * 
     * @param loop The loop number (1-4)
     * @param range The range string: HI, MID, LOW, 10V or 5V
     * @throws DriverException
     */
    public void setHeaterRange(int loop, String range) throws DriverException {
        checkLoop(loop);
        String urange = range.toUpperCase();
        String check = rangeCheck[loop - 1];
        if (!urange.matches(check)) {
            throw new DriverException("Invalid range (" + range + ") for loop " + loop + ": must match " + check);
        }
        writeCommand("LOOP " + loop + ":RANGE " + range);
    }

    
    /**
     * Gets the maximum heater power for a loop
     * 
     * @param loop The loop number (1-4)
     * @return The heater power as a percentage of the maximum
     * @throws DriverException
     */
    public double getHeaterMax(int loop) throws DriverException {
        checkLoop(loop);
        return readDouble("LOOP " + loop + ":MAXP?");
    }

    
    /**
     * Sets the maximum heater power for a loop
     * 
     * @param loop The loop number (1-4)
     * @param power The heater power as a percentage of the maximum
     * @throws DriverException
     */
    public void setHeaterMax(int loop, double power) throws DriverException {
        checkLoop(loop);
        writeCommand("LOOP " + loop + ":MAXP " + power);
    }

    
    /**
     * Gets the heater power for a manually-controlled loop
     * 
     * @param loop The loop number (1-4)
     * @return The heater power as a percentage of the maximum
     * @throws DriverException
     */
    public double getHeaterPower(int loop) throws DriverException {
        checkLoop(loop);
        return readDouble("LOOP " + loop + ":OUTP?");
    }

    
    /**
     * Sets the heater power for a manually-controlled loop
     * 
     * @param loop The loop number (1-4)
     * @param power The heater power as a percentage of the maximum
     * @throws DriverException
     */
    public void setHeaterPower(int loop, double power) throws DriverException {
        checkLoop(loop);
        writeCommand("LOOP " + loop + ":PMAN " + power);
    }

    
    /**
     * Reads the temperature of a channel
     *
     * @param channel The channel ID (A-D)
     * @return The temperature in the units specified vis setUnit 
     * @throws DriverException
     */
    public double getTemp(char channel) throws DriverException {
        double temp = -999.;
//        double temp = Double.NaN;
        checkChannel(channel);
        String resp = readString("INPUT " + channel + ":TEMP?");
        if (resp.matches("\\.+|-+| +")) {
            log.debug("CryoCon channel " + channel + " appears not to be connected.");
        } else {
            temp = Double.parseDouble(resp.replaceAll("[a-zA-Z]", "")); // remove units before converting 
        }
        return temp;
    }


    /**
     * Gets the temperature units for a channel
     *
     * @param channel The channel ID (A-D)
     * @return Temperature unit: K, F, C, S (sensor units) or blank (not used)
     * @throws DriverException
     */
    public char getUnit(char channel) throws DriverException {
        checkChannel(channel);
        return readString("INPUT " + channel + ":UNIT?").charAt(0);
    }


    /**
     * Sets the temperature units for a channel
     *
     * @param channel The channel ID (A-D)
     * @param unit Temperature unit: K, F, C or S (sensor units)
     * @throws DriverException
     */
    public void setUnit(char channel, char unit) throws DriverException {
        checkChannel(channel);
        if (!String.valueOf(unit).toUpperCase().matches("[K|C|F|S]")) {
            throw new DriverException("Invalid unit value: " + unit);
        }
        writeCommand("INPUT " + channel + ":UNITS " + unit);
    }


    /**
     * Gets the sensor type for a channel
     *
     * @param channel The channel ID (A-D)
     * @return The sensor type string
     * @throws DriverException
     */
    public String getType(char channel) throws DriverException {
        checkChannel(channel);
        String type = invTypeMap.get(readInteger("INPUT " + channel + ":SENS?"));
        return type == null ? "UNKNOWN" : type;
    }


    /**
     * Sets the sensor type for a channel
     * 
     * @param channel The channel ID (A-D)
     * @param typ The type string: DIODE, ACR, PTC100, PTC1K, NTC10UA, TC70 or NONE
     * @throws DriverException
     */
    public void setType(char channel, String typ) throws DriverException {
        checkChannel(channel);
        Integer ityp = typeMap.get(typ.toUpperCase());
        if (ityp == null) {
            throw new DriverException("Invalid sensor type: " + typ);
        }
        writeCommand("INPUT " + channel + ":SENSOR " + ityp);
    }


    /**
     * Sets the device to control the temperature
     *
     * @throws DriverException
     */
    public void setToControl() throws DriverException {
        writeCommand("CONTROL");
    }


    /**
     * Sets the device to stop temperature control
     *
     * @throws DriverException
     */
    public void stopControl() throws DriverException {
        writeCommand("STOP");
    }


    /**
     * Gets whether the device is controlling the temperature
     *
     * @return Whether device is in control
     * @throws DriverException
     */
    public boolean isInControl() throws DriverException {
        return readString("CONTROL?").startsWith("ON");
    }


    /**
     * Gets the temperature control state
     *
     * @return The control state: ON, OFF or OTD
     * @throws DriverException
     */
    public String getControl() throws DriverException {
        return readString("CONTROL?").substring(0, 3).trim();
    }


    /**
     * Gets whether Over Temperature Disconnect (OTD) is enabled
     *
     * @return Whether OTD is enabled
     * @throws DriverException
     */
    public boolean isOtdEnabled() throws DriverException {
        return readString("OVER:ENAB?").startsWith("ON");
    }


    /**
     * Sets whether Over Temperature Disconnect (OTD) is enabled
     *
     * @param enab Whether OTD is to be enabled
     * @throws DriverException
     */
    public void enableOtd(boolean enab) throws DriverException {
        writeCommand("OVER:ENAB " + (enab ? "ON" : "OFF"));
    }


    /**
     * Gets the OTD source channel
     *
     * @return The channel ID
     * @throws DriverException
     */
    public char getOtdSource() throws DriverException {
        return readString("OVER:SOUR?").charAt(2);  // Reply is "CHA", etc
    }


    /**
     * Sets the OTD source channel
     *
     * @param channel The channel ID (A-D)
     * @throws DriverException
     */
    public void setOtdSource(char channel) throws DriverException {
        checkChannel(channel);
        writeCommand("OVER:SOUR " + channel);
    }


    /**
     * Gets the OTD temperature limit
     *
     * @return The temperature, in source channel units
     * @throws DriverException
     */
    public double getOtdTemp() throws DriverException {
        return readDouble("OVER:TEMP?");
    }


    /**
     * Sets the OTD temperature limit
     *
     * @param temp The temperature, in source channel units
     * @throws DriverException
     */
    public void setOtdTemp(double temp) throws DriverException {
        writeCommand("OVER:TEMP " + temp);
    }


    /**
     * Reads a string response after sending a command.
     * 
     * Overrides the Scpi method in order to do specific error checking
     *
     * @param cmnd The command to send
     * @return The response string
     * @throws DriverException
     */
    @Override
    public String readString(String cmnd) throws DriverException {
        String resp = read(cmnd);
        if (resp.equals("NAK")) {
            throw new DriverException("Can't execute command: \"" + cmnd + "\"");
        }
        return resp;
    }


    /**
     * Sends a command.
     * 
     * Overrides the Scpi method in order to do specific error checking
     *
     * @param cmnd The command to send
     * @throws DriverException
     */
    @Override
    public void writeCommand(String cmnd) throws DriverException {
        if (read(cmnd + ";:*OPC?").equals("NAK")) {
            throw new DriverException("Can't execute command: \"" + cmnd + "\"");
        }
    }


    /**
     * Checks loop value for validity
     *
     * @param loop
     * @throws DriverException If loop is invalid
     */
    private void checkLoop(int loop) throws DriverException {
        if (loop < 1 || loop > 4) {
            throw new DriverException("Invalid loop number: " + loop);
        }
    }


    /**
     * Checks channel value for validity
     *
     * @param chan
     * @throws DriverException If channel is invalid
     */
    private void checkChannel(char chan) throws DriverException {
        char uchan = Character.toUpperCase(chan);
        if (uchan < 'A' || uchan > 'D') {
            throw new DriverException("Invalid channel: " + chan);
        }
    }

}
