package org.lsst.ccs.drivers.cryocon;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.scpi.TestScpi;

/**
 * Program to test the Cryogenic Temperature Controller 24C device driver
 * 
 * @author Heather Kelly
 */
public class TestM24C extends TestScpi {

    private static final M24C.ConnType DEFAULT_TYPE = M24C.ConnType.NET;
    private static final String BNL_DEFAULT_IP = "192.168.2.198";
    private final M24C m24c;

    /**
     *  Constructor
     */
    public TestM24C()
    {
        super(new M24C());
        m24c = (M24C)scpi;
    }

    /*.
     * Opens connection to a network device with default port.
     */
    @Command(description = "Open network connection to Model 24C")
    public void open(@Argument(description = "Host name") String host) throws DriverException {
        m24c.open(DEFAULT_TYPE, host, 0);
    }

    /*.
     * Opens connection to a device with default port/baudrate.
     */
    @Command(description = "Open connection to Model 24C")
    public void open(@Argument(description = "Communications type") M24C.ConnType type,
                     @Argument(description = "Device identifier") String ident) throws DriverException {
        m24c.open(type, ident, 0);
    }

    /*.
     * Opens connection to a Default BNL Temp Control Model 24C
     */
    @Command(description = "Open connection to BNL default Model 24C")
    public void openBNLDefault() throws DriverException {
        m24c.open(DEFAULT_TYPE, BNL_DEFAULT_IP, 0);
    }

    /*.
     * Get MaxSetPoint for Loop [1-4]
     */
    @Command(description = "Retrieve MaxSetPoint for loop")
    public double getMaxSetPoint(@Argument(description = "loop number [1-4]") int loop) throws DriverException {
        return m24c.getMaxSetPoint(loop);
    }

    /*.
     * Set MaxSetPoint for Loop [1-4]
     */
    @Command(description = "setMaxSetPoint for loop")
    public void setMaxSetPoint(@Argument(description = "loop number [1-4]") int loop,
                               @Argument(description = "temperature") double temp) throws DriverException {
        m24c.setMaxSetPoint(loop, temp);
    }

    /*.
     * Get SetPoint for Loop [1-4]
     */
    @Command(description = "Retrieve setPoint for loop")
    public double getSetPoint(@Argument(description = "loop number [1-4]") int loop) throws DriverException {
        return m24c.getSetPoint(loop);
    }

    /*.
     * Set SetPoint for Loop [1-4]
     */
    @Command(description = "setSetPoint for loop")
    public void setSetPoint(@Argument(description = "loop number [1-4]") int loop,
                            @Argument(description = "temperature") double temp) throws DriverException {
        m24c.setSetPoint(loop, temp);
    }

    /*.
     * Get Source for Loop returns Channels A through D
     */
    @Command(description = "Retrieve source channel for loop")
    public char getLoopSource(@Argument(description = "loop number [1-4]") int loop) throws DriverException {
        return m24c.getSource(loop);
    }

    /*.
     * Set Source Channel for Loop
     */
    @Command(description = "Set source channel for loop")
    public void setLoopSource(@Argument(description = "loop number [1-4]") int loop,
                              @Argument(description = "channel [A-D]") String channel) throws DriverException {
        m24c.setSource(loop, stringToChar(channel));
    }

    /*.
     * Get heater range for loop
     */
    @Command(description = "Retrieve heater range for loop")
    public String getHeaterRange(@Argument(description = "loop number [1-4]") int loop) throws DriverException {
        return m24c.getHeaterRange(loop);
    }

    /*.
     * Set heater range for loop
     */
    @Command(description = "Set heater range for loop")
    public void setHeaterRange(@Argument(description = "loop number [1-4]") int loop,
                               @Argument(description = "range (lo, mid or hi)") String range) throws DriverException {
        m24c.setHeaterRange(loop, range);
    }

    /*.
     * Get maximum heater power for loop
     */
    @Command(description = "Retrieve maximum heater power for loop")
    public double getHeaterMax(@Argument(description = "loop number [1-4]") int loop) throws DriverException {
        return m24c.getHeaterMax(loop);
    }

    /*.
     * Set maximum heater power for loop
     */
    @Command(description = "Set maximum heater power for loop")
    public void setHeaterMax(@Argument(description = "loop number [1-4]") int loop,
                             @Argument(description = "maximum power (%)") double power) throws DriverException {
        m24c.setHeaterMax(loop, power);
    }

    /*.
     * Get manual heater power for loop
     */
    @Command(description = "Retrieve manual heater power for loop")
    public double getHeaterPower(@Argument(description = "loop number [1-4]") int loop) throws DriverException {
        return m24c.getHeaterPower(loop);
    }

    /*.
     * Set manual heater power for loop
     */
    @Command(description = "Set manual heater power for loop")
    public void setHeaterPower(@Argument(description = "loop number [1-4]") int loop,
                               @Argument(description = "manual power (%)") double power) throws DriverException {
        m24c.setHeaterPower(loop, power);
    }

    /*.
     * Get control type for loop
     */
    @Command(description = "Retrieve control type for loop")
    public String getControlType(@Argument(description = "loop number [1-4]") int loop) throws DriverException {
        return m24c.getControlType(loop);
    }

    /*.
     * Set control type for loop
     */
    @Command(description = "Set control type for loop")
    public void setControlType(@Argument(description = "loop number [1-4]") int loop,
                               @Argument(description = "control type string") String type) throws DriverException {
        m24c.setControlType(loop, type);
    }

    /*.
     * Get Temperature Reading For Channels A through D
     */
    @Command(description = "Retrieve temperature")
    public double getTemp(@Argument(description = "Temp Channel to read") String channel) throws DriverException {
        return m24c.getTemp(stringToChar(channel));
    }

    /*.
     * Get Temperature Units Channels A through D
     */
    @Command(description = "Retrieve temperature units")
    public char getUnit(@Argument(description = "Temp Channel to read") String channel) throws DriverException {
        return m24c.getUnit(stringToChar(channel));
    }

    /*.
     * Set Temperature Units for Channels A through D
     */
    @Command(description = "Set temperature units")
    public void setUnit(@Argument(description = "Temp Channel to set") String channel,
                        @Argument(description = "Units in [K|C|F|S]") String unit) throws DriverException {
        m24c.setUnit(stringToChar(channel), stringToChar(unit));
    }

    /*.
     * Gets the sensor type for a channel
     */
    @Command(description = "Get sensor type")
    public String getType(@Argument(description = "Channel ID") String channel) throws DriverException {
        return m24c.getType(stringToChar(channel));
    }

    /*.
     * Sets the sensor type for a channel
     */
    @Command(description = "Set sensor type")
    public void setType(@Argument(description = "Channel ID") String channel,
                        @Argument(description = "Sensor type") String type) throws DriverException {
        m24c.setType(stringToChar(channel), type);
    }

    /*.
     * Gets whether OTD is enabled
     */
    @Command(description = "Get whether OTD system enabled")
    public boolean isOtdEnabled() throws DriverException {
        return m24c.isOtdEnabled();
    }

    /*.
     * Sets OTD enabled state
     */
    @Command(description = "Set OTD system enabled state")
    public void enableOtd(@Argument(description = "Whether to enable OTD system") boolean enab) throws DriverException {
        m24c.enableOtd(enab);
    }

    /*.
     * Gets OTD source channel
     */
    @Command(description = "Get source channel for the OTD system")
    public char getOtdSource() throws DriverException {
        return m24c.getOtdSource();
    }

    /*.
     * Sets OTD source channel
     */
    @Command(description = "Set source channel for the OTD system")
    public void setOtdSource(@Argument(description = "channel [A-D]") String channel) throws DriverException {
        m24c.setOtdSource(stringToChar(channel));
    }

    /*.
     * Gets OTD temperature limit
     */
    @Command(description = "Get temperature limit for the OTD system")
    public double getOtdTemp() throws DriverException {
        return m24c.getOtdTemp();
    }

    /*.
     * Sets OTD temperature limit
     */
    @Command(description = "Set temperature limit for the OTD system")
    public void setOtdTemp(@Argument(description = "Temperature") double temp) throws DriverException {
        m24c.setOtdTemp(temp);
    }

    /*.
     * Gets whether temperature is being controlled
     */
    @Command(description = "Get whether control is on")
    public boolean isInControl() throws DriverException {
        return m24c.isInControl();
    }

    /*.
     * Gets temperature control state
     */
    @Command(description = "Get control state")
    public String getControl() throws DriverException {
        return m24c.getControl();
    }

    /*.
     * Sets temperature control on
     */
    @Command(description = "Set control on")
    public void setToControl() throws DriverException {
        m24c.setToControl();
    }

    /*.
     * Sets temperature control off
     */
    @Command(description = "Set control off")
    public void stopControl() throws DriverException {
        m24c.stopControl();
    }

    /*.
     * Convert string to char
     */
    private static char stringToChar(String str) throws DriverException {
        if (str.length() != 1) {
            throw new DriverException("String must be of length 1");
        }
        return str.charAt(0);
    }

}
