package org.lsst.ccs.drivers.cryocon;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.scpi.TestScpi;

/**
 * 
 * Program to test the Cryogenic Temperature Controller 24C device driver
 * 
 * @author Heather Kelly
 ** 
 */
public class TestM24C extends TestScpi {

    private static final M24C.ConnType DEFAULT_TYPE = M24C.ConnType.NET;
    private static final String BNL_DEFAULT_IP = "192.168.2.198";
    private final M24C m24c;

   /**
    *  Constructor
    */
    public TestM24C()
    {
        super(new M24C());
        m24c = (M24C)scpi;
    }

    /*.
     * Opens connection to a network device with default port.
     */
    @Command(name = "open", description = "Open network connection to Model 24C")
    public void open(@Argument(name = "host", description = "Host name") String host)
                     throws DriverException {
        m24c.open(DEFAULT_TYPE, host, 0);
    }

    /*.
     * Opens connection to a device with default port/baudrate.
     */
    @Command(name = "open", description = "Open connection to Model 24C")
    public void open(@Argument(name = "type", description = "Communications type") M24C.ConnType type,
                     @Argument(name = "identity", description = "Device identifier") String ident)
                     throws DriverException {
        m24c.open(type, ident, 0);
    }

    /*.
     * Opens connection to a Default BNL Temp Control Model 24C
     */
    @Command(name = "openDefault", description = "Open connection to BNL default Model 24C")
    public void openBNLDefault() throws DriverException {
        m24c.open(DEFAULT_TYPE, BNL_DEFAULT_IP, 0);
    }

    /*.
     * Get MaxSetPoint for Loop [1-4]
     */
    @Command(name = "getMaxSetPoint", description = "Retrieve MaxSetPoint for loop")
    public double getMaxSetPoint(@Argument(name = "loop", description = "loop number [1-4]") int loop)
                                 throws DriverException {
        return m24c.getMaxSetPoint(loop);
    }

    /*.
     * Set MaxSetPoint for Loop [1-4]
     */
    @Command(name = "setMaxSetPoint", description = "setMaxSetPoint for loop")
    public void setMaxSetPoint(@Argument(name = "loop", description = "loop number [1-4]") int loop,
                               @Argument(name = "temp", description = "temperature") double temp)
                               throws DriverException {
        m24c.setMaxSetPoint(loop, temp);
    }

    /*.
     * Get SetPoint for Loop [1-4]
     */
    @Command(name = "getSetPoint", description = "Retrieve setPoint for loop")
    public double getSetPoint(@Argument(name = "loop", description = "loop number [1-4]") int loop)
                              throws DriverException {
        return m24c.getSetPoint(loop);
    }

    /*.
     * Set SetPoint for Loop [1-4]
     */
    @Command(name = "setSetPoint", description = "setSetPoint for loop")
    public void setSetPoint(@Argument(name = "loop", description = "loop number [1-4]") int loop,
                            @Argument(name = "temp", description = "temperature") double temp)
                            throws DriverException {
        m24c.setSetPoint(loop, temp);
    }

    /*.
     * Get Source for Loop returns Channels A through D
     */
    @Command(name = "getLoopSource", description = "Retrieve source channel for loop")
    public char getLoopSource(@Argument(name = "loop", description = "loop number [1-4]") int loop)
                              throws DriverException {
        return m24c.getSource(loop);
    }

    /*.
     * Set Source Channel for Loop
     */
    @Command(name = "setLoopSource", description = "Set source channel for loop")
    public void setLoopSource(@Argument(name = "loop", description = "loop number [1-4]") int loop,
                              @Argument(name = "channel", description = "channel [A-D]") String channel)
                              throws DriverException {
        m24c.setSource(loop, stringToChar(channel));
    }

    /*.
     * Get heater range for loop
     */
    @Command(name = "getHeaterRange", description = "Retrieve heater range for loop")
    public String getHeaterRange(@Argument(name = "loop", description = "loop number [1-4]") int loop)
                                 throws DriverException {
        return m24c.getHeaterRange(loop);
    }

    /*.
     * Set heater range for loop
     */
    @Command(name = "setHeaterRange", description = "Set heater range for loop")
    public void setHeaterRange(@Argument(name = "loop", description = "loop number [1-4]") int loop,
                               @Argument(name = "range", description = "range (lo, mid or hi)") String range)
                               throws DriverException {
        m24c.setHeaterRange(loop, range);
    }

    /*.
     * Get maximum heater power for loop
     */
    @Command(name = "getHeaterMax", description = "Retrieve maximum heater power for loop")
    public double getHeaterMax(@Argument(name = "loop", description = "loop number [1-4]") int loop)
                               throws DriverException {
        return m24c.getHeaterMax(loop);
    }

    /*.
     * Set maximum heater power for loop
     */
    @Command(name = "setHeaterMax", description = "Set maximum heater power for loop")
    public void setHeaterMax(@Argument(name = "loop", description = "loop number [1-4]") int loop,
                             @Argument(name = "power", description = "maximum power (%)") double power)
                             throws DriverException {
        m24c.setHeaterMax(loop, power);
    }

    /*.
     * Get control type for loop
     */
    @Command(name = "getControlType", description = "Retrieve control type for loop")
    public String getControlType(@Argument(name = "loop", description = "loop number [1-4]") int loop)
                                 throws DriverException {
        return m24c.getControlType(loop);
    }

    /*.
     * Set control type for loop
     */
    @Command(name = "setControlType", description = "Set control type for loop")
    public void setControlType(@Argument(name = "loop", description = "loop number [1-4]") int loop,
                               @Argument(name = "type", description = "control type string") String type)
                             throws DriverException {
        m24c.setControlType(loop, type);
    }

    /*.
     * Get Temperature Reading For Channels A through D
     */
    @Command(name = "getTemp", description = "Retrieve temperature")
    public double getTemp(@Argument(name = "channel", description = "Temp Channel to read") String channel)
                          throws DriverException {
        return m24c.getTemp(stringToChar(channel));
    }

    /*.
     * Get Temperature Units Channels A through D
     */
    @Command(name = "getUnit", description = "Retrieve temperature units")
    public char getUnit(@Argument(name = "channel", description = "Temp Channel to read") String channel)
                        throws DriverException {
        return m24c.getUnit(stringToChar(channel));
    }

    /*.
     * Set Temperature Units for Channels A through D
     */
    @Command(name = "setUnit", description = "Set temperature units")
    public void setUnit(@Argument(name = "channel", description = "Temp Channel to set") String channel,
                        @Argument(name = "units", description = "Units in [K|C|F|S]") String unit)
                       throws DriverException {
        m24c.setUnit(stringToChar(channel), stringToChar(unit));
    }

    /*.
     * Gets the sensor type for a channel
     */
    @Command(name = "getType", description = "Get sensor type")
    public String getType(@Argument(name = "channel", description = "Channel ID") String channel)
                          throws DriverException {
        return m24c.getType(stringToChar(channel));
    }

    /*.
     * Sets the sensor type for a channel
     */
    @Command(name = "setType", description = "Set sensor type")
    public void setType(@Argument(name = "channel", description = "Channel ID") String channel,
                        @Argument(name = "type", description = "Sensor type") String type)
                        throws DriverException {
        m24c.setType(stringToChar(channel), type);
    }

    /*.
     * Gets whether OTD is enabled
     */
    @Command(name = "isOtdEnabled", description = "Get whether OTD system enabled")
    public boolean isOtdEnabled() throws DriverException {
        return m24c.isOtdEnabled();
    }

    /*.
     * Sets OTD enabled state
     */
    @Command(name = "enableOtd", description = "Set OTD system enabled state")
    public void enableOtd(@Argument(name = "enab", description = "Whether to enable OTD system") boolean enab)
                          throws DriverException {
        m24c.enableOtd(enab);
    }

    /*.
     * Gets OTD source channel
     */
    @Command(name = "getOtdSource", description = "Get source channel for the OTD system")
    public char getOtdSource() throws DriverException {
        return m24c.getOtdSource();
    }

    /*.
     * Sets OTD source channel
     */
    @Command(name = "setOtdSource", description = "Set source channel for the OTD system")
    public void setOtdSource(@Argument(name = "channel", description = "channel [A-D]") String channel)
                             throws DriverException {
        m24c.setOtdSource(stringToChar(channel));
    }

    /*.
     * Gets OTD temperature limit
     */
    @Command(name = "getOtdTemp", description = "Get temperature limit for the OTD system")
    public double getOtdTemp() throws DriverException {
        return m24c.getOtdTemp();
    }

    /*.
     * Sets OTD temperature limit
     */
    @Command(name = "setOtdTemp", description = "Set temperature limit for the OTD system")
    public void setOtdTemp(@Argument(name = "temp", description = "Temperature") double temp)
                           throws DriverException {
        m24c.setOtdTemp(temp);
    }

    /*.
     * Gets whether temperature is being controlled
     */
    @Command(name = "isInControl", description = "Get whether control is on")
    public boolean isInControl() throws DriverException {
        return m24c.isInControl();
    }

    /*.
     * Gets temperature control state
     */
    @Command(name = "getControl", description = "Get control state")
    public String getControl() throws DriverException {
        return m24c.getControl();
    }

    /*.
     * Sets temperature control on
     */
    @Command(name = "setToControl", description = "Set control on")
    public void setToControl() throws DriverException {
        m24c.setToControl();
    }

    /*.
     * Sets temperature control off
     */
    @Command(name = "stopControl", description = "Set control off")
    public void stopControl() throws DriverException {
        m24c.stopControl();
    }

    /*.
     * Convert string to char
     */
    private static char stringToChar(String str) throws DriverException {
        if (str.length() != 1) {
            throw new DriverException("String must be of length 1");
        }
        return str.charAt(0);
    }

}
