package org.lsst.ccs.drivers.cryocon;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.scpi.TestScpi;

/**
 * 
 * Program to test the Cryogenic Temperature Controller 24C device driver
 * 
 * @author Heather Kelly
 ** 
 */
public class TestM24C extends TestScpi {

    private final M24C tempDev;

   /**
    *  Constructor
    */
    public TestM24C()
    {
        super(new M24C());
        tempDev = (M24C)scpi;
    }

    /*.
     * Opens connection to a device with default port/baudrate.
     */
    @Command(name = "open", description = "Open connection to Model 24C")
    public void open(@Argument(name = "type", description = "Communications type") M24C.ConnType type,
                     @Argument(name = "identity", description = "Device identifier") String ident)
                     throws DriverException {
        tempDev.open(type, ident, 0);
    }

    /*.
     * Opens connection to a Default BNL Temp Control Model 24C
     */
    @Command(name = "openDefault", description = "Open connection to Default Model 24C")
    public void openDefault() throws DriverException {
        tempDev.openDefault();
    }

    /*.
     * Get MaxSetPoint for Loop [1-4]
     */
    @Command(name = "getMaxSetPoint", description = "Retrieve MaxSetPoint for loop")
    public double getMaxSetPoint(@Argument(name = "loop", description = "loop number [1-4]") int loop)
                                 throws DriverException {
        return tempDev.getMaxSetPoint(loop);
    }

    /*.
     * Set MaxSetPoint for Loop [1-4]
     */
    @Command(name = "setMaxSetPoint", description = "setMaxSetPoint for loop")
    public void setMaxSetPoint(@Argument(name = "loop", description = "loop number [1-4]") int loop,
                               @Argument(name = "temp", description = "temperature") double temp)
                               throws DriverException {
        tempDev.setMaxSetPoint(loop, temp);
    }

    /*.
     * Get SetPoint for Loop [1-4]
     */
    @Command(name = "getSetPoint", description = "Retrieve setPoint for loop")
    public double getSetPoint(@Argument(name = "loop", description = "loop number [1-4]") int loop)
                              throws DriverException {
        return tempDev.getSetPoint(loop);
    }

    /*.
     * Set SetPoint for Loop [1-4]
     */
    @Command(name = "setSetPoint", description = "setSetPoint for loop")
    public void setSetPoint(@Argument(name = "loop", description = "loop number [1-4]") int loop,
                            @Argument(name = "temp", description = "temperature") double temp)
                            throws DriverException {
        tempDev.setSetPoint(loop, temp);
    }

    /*.
     * Get Source for Loop returns Channels A through D
     */
    @Command(name = "getLoopSource", description = "Retrieve source channel for loop")
    public char getLoopSource(@Argument(name = "loop", description = "loop number [1-4]") int loop)
                              throws DriverException {
        return (tempDev.getSource(loop)).charAt(0);
    }

    /*.
     * Set Source Channel for Loop
     */
    @Command(name = "setLoopSource", description = "Set source channel for loop")
    public void setLoopSource(@Argument(name = "loop", description = "loop number [1-4]") int loop,
                              @Argument(name = "channel", description = "channel [A-D]") String channel)
                              throws DriverException {
        tempDev.setSource(loop, stringToChar(channel));
    }

    /*.
     * Get Temperature Reading For Channels A through D
     */
    @Command(name = "getTemp", description = "Retrieve temperature")
    public double getTemp(@Argument(name = "channel", description = "Temp Channel to read") String channel)
                          throws DriverException {
        return tempDev.getTemp(stringToChar(channel));
    }

    /*.
     * Get Temperature Units Channels A through D
     */
    @Command(name = "getUnit", description = "Retrieve temperature units")
    public char getUnit(@Argument(name = "channel", description = "Temp Channel to read") String channel)
                        throws DriverException {
        return tempDev.getUnit(stringToChar(channel));
    }

    /*.
     * Set Temperature Units for Channels A through D
     */
    @Command(name = "setUnit", description = "Set temperature units")
    public void setUnit(@Argument(name = "channel", description = "Temp Channel to set") String channel,
                        @Argument(name = "units", description = "Units in [K|C|F|S]") String unit)
                       throws DriverException {
        tempDev.setUnit(stringToChar(channel), stringToChar(unit));
    }

    /*.
     * Gets the sensor type for a channel
     */
    @Command(name = "getType", description = "Get sensor type")
    public String getType(@Argument(name = "channel", description = "Channel ID") String channel)
                          throws DriverException {
        return tempDev.getType(stringToChar(channel));
    }

    /*.
     * Sets the sensor type for a channel
     */
    @Command(name = "setType", description = "Set sensor type")
    public void setType(@Argument(name = "channel", description = "Channel ID") String channel,
                        @Argument(name = "type", description = "Sensor type") String type)
                        throws DriverException {
        tempDev.setType(stringToChar(channel), type);
    }

    /*.
     * Gets whether temperature is being controlled
     */
    @Command(name = "isControl", description = "Get whether control is on")
    public boolean isControl() throws DriverException {
        return tempDev.isInControl();
    }

    /*.
     * Sets temperature control on
     */
    @Command(name = "control", description = "Set control on")
    public void control() throws DriverException {
        tempDev.setToControl();
    }

    /*.
     * Convert string to char
     */
    private static char stringToChar(String str) throws DriverException {
        if (str.length() != 1) {
            throw new DriverException("String must be of length 1");
        }
        return str.charAt(0);
    }

}
