package org.lsst.ccs.drivers.cryocon;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.scpi.Scpi;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *
 * Enables access to a Cryogenic Temperature controller Model 24C
 *
 * @author Heather Kelly
 *
 */

/*
 * TODO define a common interface for temperature controllers. An issue is
 * temperature, which can be given in K, F, C, or sensor units depending on
 * controllers : how to unify?
 */
public class M24C extends Scpi {

    /**
     * Public constants
     */
    public static final int NUM_LOOPS = 4;

    /**
     * Private constants
     */
    private static final int DEFAULT_PORT = 5000, DEFAULT_BAUD = 9600;

    /**
     * Default connection settings at BNL
     */
    private static final Scpi.ConnType DEFAULT_TYPE = Scpi.ConnType.NET;
    private static final String BNL_DEFAULT_IP = "192.168.2.198";

    /**
     * Private fields
     */
    private static final Map<String, Integer> typeMap = new HashMap<>();
    static {
        typeMap.put("DIODE", 5);
        typeMap.put("PTC100", 20);
        typeMap.put("PTC1K", 21);
        typeMap.put("NONE", 0);
        typeMap.put("INTERNAL", 59);  // ???
        typeMap.put("DIODE1", 1);
        typeMap.put("DIODE2", 2);
        typeMap.put("DIODE3", 3);
        typeMap.put("DIODE4", 4);
        typeMap.put("TC70K", 45);
        typeMap.put("TC70E", 46);
        typeMap.put("TC70T", 47);
        typeMap.put("TC70AF", 48);
        typeMap.put("NTC10UA", 31);
        typeMap.put("ACR", 32);
    }
    private final Map<Integer, String> invTypeMap = new HashMap<>();
    private final Logger log = Logger.getLogger("lsst.ccs.drivers.cryocon");


    /**
     *  Constructor.
     */
    public M24C() {
        super();
        for (String type : typeMap.keySet()) {
            invTypeMap.put(typeMap.get(type), type);
        }
    }


    /**
     *  Constructor.
     *
     *  Deprecated; use explicit open instead
     *
     *  @param  type
     *  @param  ident
     *  @param  param
     *  @throws DriverException
     */
    @Deprecated
    public M24C(int type, String ident, int param) throws DriverException {
        this();
        open(type, ident, param);
    }


    /**
     *  Opens a connection to the device.
     *
     *  This is the open method that overrides the superclass open when extra
     *  functionality is needed.
     *
     *  @param  type   The enumerated type of connection to make
     *
     *  @param  ident  The device identifier:
     *                   host name or IP address for network;
     *                   [node:]serial number for FTDI device;
     *                   device name for serial
     *
     *  @param  parm1  The first device parameter:
     *                   port number for network;
     *                   baud rate for FTDI or serial
     *
     *  @param  parm2  The second device parameter:
     *                   unused for network;
     *                   encoded data characteristics for FTDI or serial:
     *                     0 sets 8-bit, no parity, 1 stop bit, no flow ctrl
     *
     *  @throws  DriverException
     */
    @Override
    public void open(ConnType type, String ident, int parm1, int parm2) throws DriverException {
        super.open(type, ident, parm1 == 0 ? type == ConnType.NET ? DEFAULT_PORT : DEFAULT_BAUD : parm1, parm2);
        //setNoErrorQueue(true);
        try {
            checkIdentification("Cryo-con", CHECK_EQUALS, "24C", CHECK_CONTAINS);
        }
        catch (DriverException e) {
            closeSilent();
            throw e;
        }
        log.debug("Opened Cryo-con M24C temperature controller");
    }


    /**
     * Provide open for BNL Default settings
     *
     *  @throws  DriverException
     */
    public void openDefault() throws DriverException {
        open(DEFAULT_TYPE, BNL_DEFAULT_IP, DEFAULT_PORT);
    }


    /**
     * Gets the maximum set point for a loop
     *
     * @param loop The loop number (1-4)
     * @return The maximum set point
     * @throws DriverException
     */
    public double getMaxSetPoint(int loop) throws DriverException {
        checkLoop(loop);
        return Double.parseDouble(readString("LOOP " + loop + ":MAXSET?").replaceAll("[a-zA-Z]", ""));
    }


    /**
     * Sets the maximum set point for a loop
     *
     * @param loop The loop number (1-4)
     * @param temp The temperature to set in the units specified via setUnit
     * @throws DriverException
     */
    public void setMaxSetPoint(int loop, double temp) throws DriverException {
        checkLoop(loop);
        writeCommand("LOOP " + loop + ":MAXSET " + temp);
    }


    /**
     * Gets the current set point for a loop
     *
     * @param loop The loop number (1-4)
     * @return The current set point
     * @throws DriverException
     */
    public double getSetPoint(int loop) throws DriverException {
        checkLoop(loop);
        return Double.parseDouble(readString("LOOP " + loop + ":SETPT?").replaceAll("[a-zA-Z]", ""));
    }


    /**
     * Sets the set point for a loop
     *
     * @param loop The loop number (1-4)
     * @param temp The temperature to set in the units specified via setUnit
     * @throws DriverException
     */
    public void setSetPoint(int loop, double temp) throws DriverException {
        checkLoop(loop);
        if (temp > getMaxSetPoint(loop)) {
            throw new DriverException("Temperature is above maximum set point: " + temp);
        }
        writeCommand("LOOP " + loop + ":SETPT " + temp);
    }


    /**
     * Gets the heater power level for a loop
     *
     * @param loop The loop number (1-4)
     * @return % of full power
     * @throws DriverException
     */
    public double getHtrRead(int loop) throws DriverException {
        checkLoop(loop);
        return(Double.parseDouble(readString("LOOP " + loop + ":HTRR?").replace("%", "")));
    }


    /**
     * Gets the PID P parameter for a loop
     *
     * @param loop The loop number (1-4)
     * @return The P parameter
     * @throws DriverException
     */
    public double getPID_P(int loop) throws DriverException {
        checkLoop(loop);
        return readDouble("LOOP " + loop + ":PGAIN?");
    }


    /**
     * Gets the PID I parameter for a loop
     *
     * @param loop The loop number (1-4)
     * @return The I parameter
     * @throws DriverException
     */
    public double getPID_I(int loop) throws DriverException {
        checkLoop(loop);
        return readDouble("LOOP " + loop + ":IGAIN?");
    }


    /**
     * Gets the PID D parameter for a loop
     *
     * @param loop The loop number (1-4)
     * @return The D parameter
     * @throws DriverException
     */
    public double getPID_D(int loop) throws DriverException {
        checkLoop(loop);
        return readDouble("LOOP " + loop + ":DGAIN?");
    }


    /**
     * Gets the channel source for a loop
     *
     * @param loop The loop number (1-4)
     * @return channel [A,D]
     * @throws DriverException
     */
    public String getSource(int loop) throws DriverException {
        checkLoop(loop);
        String resp = readString("LOOP " + loop + ":SOURCE?");
        if (resp.charAt(0) >= 'A' && resp.charAt(0) <= 'D') {
            return resp;
        } else {
            throw new DriverException("Invalid source response: " + resp);
        }
    }


    /**
     * Sets the channel source for a loop
     *
     * @param loop The loop number (1-4)
     * @param channel The channel ID (A-D)
     * @throws DriverException
     */
    public void setSource(int loop, char channel) throws DriverException {
        checkLoop(loop);
        checkChannel(channel);
        writeCommand("LOOP " + loop + ":SOURCE " + channel);
    }


    /**
     * Sets the heater range for a loop
     * 
     * @param loop The loop number (1-4)
     * @param range The range string: HI, MID or LOW
     * @throws DriverException
     */
    public void setHeaterRange(int loop, String range) throws DriverException {
        checkLoop(loop);
        String urange = range.toUpperCase();
        if (!urange.matches("HI|MID|LOW")) {
            throw new DriverException("Invalid range: " + range + " - must be HI, MID or LOW");
        }
        writeCommand("LOOP " + loop + ":RANGE " + range);
    }

    
    /**
     * Reads the temperature of a channel
     *
     * @param channel The channel ID (A-D)
     * @return The temperature in the units specified vis setUnit 
     * @throws DriverException
     */
    public double getTemp(char channel) throws DriverException {
        double temp = -999.;
//        double temp = Double.NaN;
        checkChannel(channel);
        String resp = readString("INPUT " + channel + ":TEMP?");
        if (resp.matches("\\.+|-+| +")) {
            log.debug("CryoCon channel " + channel + " appears not to be connected.");
        } else {
            temp = Double.parseDouble(resp.replaceAll("[a-zA-Z]", "")); // remove units before converting 
        }
        return temp;
    }


    /**
     * Gets the temperature units for a channel
     *
     * @param channel The channel ID (A-D)
     * @return Temperature unit: K, F, C, S (sensor units) or blank (not used)
     * @throws DriverException
     */
    public char getUnit(char channel) throws DriverException {
        checkChannel(channel);
        return readString("INPUT " + channel + ":UNIT?").charAt(0);
    }


    /**
     * Sets the temperature units for a channel
     *
     * @param channel The channel ID (A-D)
     * @param unit Temperature unit: K, F, C or S (sensor units)
     * @throws DriverException
     */
    public void setUnit(char channel, char unit) throws DriverException {
        checkChannel(channel);
        if (!String.valueOf(unit).toUpperCase().matches("[K|C|F|S]")) {
            throw new DriverException("Invalid unit value: " + unit);
        }
        writeCommand("INPUT " + channel + ":UNITS " + unit);
    }


    /**
     * Gets the sensor type for a channel
     *
     * @param channel The channel ID (A-D)
     * @return The sensor type string
     * @throws DriverException
     */
    public String getType(char channel) throws DriverException {
        checkChannel(channel);
        String type = invTypeMap.get(readInteger("INPUT " + channel + ":SENS?"));
        return type == null ? "UNKNOWN" : type;
    }


    /**
     * Sets the sensor type for a channel
     * 
     * @param channel The channel ID (A-D)
     * @param typ The type string: DIODE, ACR, PTC100, PTC1K, NTC10UA, TC70 or NONE
     * @throws DriverException
     */
    public void setType(char channel, String typ) throws DriverException {
        checkChannel(channel);
        Integer ityp = typeMap.get(typ.toUpperCase());
        if (ityp == null) {
            throw new DriverException("Invalid sensor type: " + typ);
        }
        writeCommand("INPUT " + channel + ":SENSOR " + ityp);
    }


    /**
     * Sets the device to control the temperature
     *
     * @throws DriverException
     */
    public void setToControl() throws DriverException {
        writeCommand("CONTROL");
    }


    /**
     * Gets whether the device is controlling the temperature
     *
     * @return Whether device is in control
     * @throws DriverException
     */
    public boolean isInControl() throws DriverException {
        return readString("CONTROL?").contains("ON");
    }


    /**
     * Reads a string response after sending a command.
     * 
     * Overrides the Scpi method in order to do specific error checking
     *
     * @param cmnd The command to send
     * @return The response string
     * @throws DriverException
     */
    @Override
    public String readString(String cmnd) throws DriverException {
        String resp = read(cmnd);
        if (resp.equals("NAK")) {
            throw new DriverException("Can't execute command: \"" + cmnd + "\"");
        }
        return resp;
    }


    /**
     * Sends a command.
     * 
     * Overrides the Scpi method in order to do specific error checking
     *
     * @param cmnd The command to send
     * @throws DriverException
     */
    @Override
    public void writeCommand(String cmnd) throws DriverException {
        if (read(cmnd + ";:*OPC?").equals("NAK")) {
            throw new DriverException("Can't execute command: \"" + cmnd + "\"");
        }
    }


    /**
     * Checks loop value for validity
     *
     * @param loop
     * @throws DriverException If loop is invalid
     */
    private void checkLoop(int loop) throws DriverException {
        if (loop < 1 || loop > 4) {
            throw new DriverException("Invalid loop number: " + loop);
        }
    }


    /**
     * Checks channel value for validity
     *
     * @param chan
     * @throws DriverException If channel is invalid
     */
    private void checkChannel(char chan) throws DriverException {
        char uchan = Character.toUpperCase(chan);
        if (uchan < 'A' || uchan > 'D') {
            throw new DriverException("Invalid channel: " + chan);
        }
    }

}
