package org.lsst.ccs.drivers.cryocon;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.scpi.Scpi;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *
 * Enables access to a Cryogenic Temperature controller Model 24C
 *
 * @author Heather Kelly
 *
 */

/*
 * TODO define a common interface for temperature controllers. An issue is
 * temperature, which can be given in K, F, C, or sensor units depending on
 * controllers : how to unify?
 */
public class M24C {

    /**
     * Public constants
     */
    Logger log = Logger.getLogger("lsst.drivers.cryocon");

    /**
     * Private constants
     */
    /**
     * Default connection settings at BNL
     */
    private final static int DEFAULT_TYPE = Scpi.CONN_TYPE_NETWORK;
    private final static int DEFAULT_PORT = 5000;
    private final static String BNL_DEFAULT_IP = "192.168.2.198";

    /**
     * Private fields
     */
    private final static Map<String, Integer> typeMap = new HashMap<>();

    static {
        typeMap.put("DIODE", 5);
        typeMap.put("PTC100", 20);
        typeMap.put("PTC1K", 21);
        typeMap.put("NONE", 0);
        typeMap.put("INTERNAL", 59);
    }

    private Scpi scpi;

    public M24C() throws DriverException {
        scpi = new Scpi();
    }

    public M24C(int type, String ident, int param)
            throws DriverException {
        scpi = new Scpi();
        this.open(type, ident, param);
    }

    public void open(int type, String ident, int param) throws DriverException {
        scpi.open(type, ident, param);
        log.debug("Opened Cryogenic Temperature Controller");
    }

    /**
     * Provide open for BNL Default settings
     */
    public void openDefault() throws DriverException {
        this.open(DEFAULT_TYPE, BNL_DEFAULT_IP, DEFAULT_PORT);
    }

    /**
     * Close the device
     *
     */
    public void close() throws DriverException {
        scpi.close();
    }

    /**
     * Gets the device ID
     *
     * @return The ID of the current open device
     */
    public String[] getIdentification() throws DriverException {
        String[] id = scpi.getIdentification();
        return id;
    }

    /**
     * @return Maximum Set Point for the specified loop [1,4]
     */
    public double getMaxSetPoint(int loop) throws DriverException {
        if ((loop < 1) || (loop > 4)) {
            throw new DriverException(
                    "M24C::getMaxSetPoint loop is invalid " + loop);
        }
        String cmd = "LOOP " + Integer.toString(loop) + ":MAXSET?";
        String resp = scpi.read(cmd);
        return Double.parseDouble(resp.replaceAll("[a-zA-Z]", ""));
    }

    /**
     * Sets the Maximum Set Point for loop = [1,4], using the input temp. The
     * units of the temperature may be K, F, C, or S (sensor units) and can be
     * checked with the getUnit command.
     */
    public void setMaxSetPoint(int loop, double temp) throws DriverException {
        if ((loop < 1) || (loop > 4)) {
            throw new DriverException(
                    "M24C::setMaxSetPoint loop is invalid " + loop);
        }
        String cmd = "LOOP " + Integer.toString(loop) + ":MAXSET "
                + Double.toString(temp) + ";:*OPC?";
        String resp = scpi.read(cmd);
        int retVal = Integer.parseInt(resp);
        if (retVal != 1) {
            throw new DriverException(
                    "M24C::setMaxSetPoint returned error code "
                    + resp);
        }
        return;
    }

    /**
     * @return current SetPoint for loop = [1,4]
     */
    public double getSetPoint(int loop) throws DriverException {
        if ((loop < 1) || (loop > 4)) {
            throw new DriverException(
                    "M24C::getSetPoint loop is invalid " + loop);
        }
        String cmd = "LOOP " + Integer.toString(loop) + ":SETPT?";
        String resp = scpi.read(cmd);
        return Double.parseDouble(resp.replaceAll("[a-zA-Z]", ""));
    }

    /**
     * Sets the set point for the loop =[1,4], using temp where the temp units a
     * are in K, F, C, or S (sensor units), using setUnit and getUnit to set and
     * check the temp units
     */
    public void setSetPoint(int loop, double temp) throws DriverException {
        if ((loop < 1) || (loop > 4)) {
            throw new DriverException(
                    "M24C::setSetPoint loop is invalid " + loop);
        }
        if (temp > getMaxSetPoint(loop)) {
            throw new DriverException(
                    "M24C::setSetPoint requested temp is too high check MAXSET "
                    + temp);
        }
        String cmd = "LOOP " + Integer.toString(loop) + ":SETPT "
                + Double.toString(temp) + ";:*OPC?";
        String resp = scpi.read(cmd);
        int retVal = Integer.parseInt(resp);
        if (retVal != 1) {
            throw new DriverException(
                    "M24C::setSetPoint returned error code " + resp);
        }
        return;

    }

    /**
     * Checks on % heater power associated with a specific loop = [1,4]
     *
     * @return % of full power
     */
    public double getHtrRead(int loop) throws DriverException {
        if ((loop < 1) || (loop > 4)) {
            throw new DriverException(
                    "M24C::getSource loop is invalid " + loop);
        }
        String cmd = "LOOP " + Integer.toString(loop) + ":HTRREAD?";
        String resp = scpi.read(cmd);
        return(Double.parseDouble(resp.replace("%", " ")));
    }

    /**
     * Gets [P]ID parameter associated with a specific loop = [1,4]
     *
     * @return P parameter
     */
    public double getPID_P(int loop) throws DriverException {
        if ((loop < 1) || (loop > 4)) {
            throw new DriverException(
                    "M24C::getSource loop is invalid " + loop);
        }
        String cmd = "LOOP " + Integer.toString(loop) + ":PGAIN?";
        String resp = scpi.read(cmd);
        return(Double.parseDouble(resp));
    }

    /**
     * Gets P[I]D parameter associated with a specific loop = [1,4]
     *
     * @return I parameter
     */
    public double getPID_I(int loop) throws DriverException {
        if ((loop < 1) || (loop > 4)) {
            throw new DriverException(
                    "M24C::getSource loop is invalid " + loop);
        }
        String cmd = "LOOP " + Integer.toString(loop) + ":IGAIN?";
        String resp = scpi.read(cmd);
        return(Double.parseDouble(resp));
    }

    /**
     * Gets PI[D] parameter associated with a specific loop = [1,4]
     *
     * @return D parameter
     */
    public double getPID_D(int loop) throws DriverException {
        if ((loop < 1) || (loop > 4)) {
            throw new DriverException(
                    "M24C::getSource loop is invalid " + loop);
        }
        String cmd = "LOOP " + Integer.toString(loop) + ":DGAIN?";
        String resp = scpi.read(cmd);
        return(Double.parseDouble(resp));
    }

    /**
     * Checks on the channel associated with a specific loop = [1,4]
     *
     * @return channel [A,D]
     */
    public String getSource(int loop) throws DriverException {
        if ((loop < 1) || (loop > 4)) {
            throw new DriverException(
                    "M24C::getSource loop is invalid " + loop);
        }
        String cmd = "LOOP " + Integer.toString(loop) + ":SOURCE?";
        String resp = scpi.read(cmd);
        if (resp.charAt(0) >= 'A' && resp.charAt(0) <= 'D') {
            return resp;
        } else {
            throw new DriverException(
                    "M24C::getSource invalid response " + resp);
        }
    }

    /**
     * Set the channel [A,D] source for a loop [1,4]
     */
    public void setSource(int loop, char channel) throws DriverException {
        if ((loop < 1) || (loop > 4)) {
            throw new DriverException(
                    "M24C::setSource loop is invalid " + loop);
        }
        if (channel >= 'A' && channel <= 'D') {
            String cmd = "LOOP " + Integer.toString(loop) + ":SOURCE "
                    + channel + ";:*OPC?";
            String resp = scpi.read(cmd);
            int retVal = Integer.parseInt(resp);
            if (retVal != 1) {
                throw new DriverException(
                        "M24C::setSource returned error code " + resp);
            }
            return;
        } else {
            throw new DriverException(
                    "M24C::setSource channel is invalid " + channel);
        }
    }

    /**
     * Set the heater range for the given loop
     */
    public void setHeaterRange(int loop, String range) throws DriverException {
        if ((loop < 1) || (loop > 4)) {
            throw new DriverException(
                    "M24C::setHeaterRange loop is invalid " + loop);
        }
        String urange = range.toUpperCase();
        if (!urange.contains("HI") && !urange.contains("MID") && !urange.contains("LOW")) {
            throw new DriverException(
                    "M24C::setHeaterRange invalid range specified. Allowed values are Hi, MID, LOW. Requested value =  "
                    + urange);
        }
        String cmd = "LOOP " + Integer.toString(loop) + ":RANGE "
                + urange + ";:*OPC?";
        String resp = scpi.read(cmd);
        int retVal = Integer.parseInt(resp);
        if (retVal != 1) {
            throw new DriverException(
                    "M24C::setSetHeaterRange returned error code " + resp);
        }
        return;

    }

    /**
     * Queries the temperature of a channel [A,D]
     *
     * @return temperature in units of K, F, C, or S (sensor units) depending on
     * the channel units which can checked and set via getUnit and setUnit
     */
    public double getTemp(char channel) throws DriverException {
        double temp = -999.;
//        double temp = Double.NaN;
        if (channel >= 'A' && channel <= 'D') {
            String cmd = "INPUT " + channel + ":TEMP?";
            String resp = scpi.read(cmd);
            if (resp.indexOf("...") >= 0 || resp.indexOf("----") >= 0) {
                log.debug("CryoCon channel " + channel + " appears not to be connected.");
            } else {
                temp = Double.parseDouble(resp.replaceAll("[a-zA-Z]", "")); // remove units before converting 
            }
        } else {
            throw new DriverException("Invalid channel request " + channel);
        }
        return (temp);
    }

    /**
     * Queries the temperatur units associated with channel [A,D]
     *
     * @return temperatue unit K, F, C, S (sensor units)
     */
    public char getUnit(char channel) throws DriverException {
        char upperCh = Character.toUpperCase(channel);
        if (upperCh >= 'A' && upperCh <= 'D') {
            String cmd = "INPUT " + upperCh + ":UNIT?";
            String resp = scpi.read(cmd);
            if (!resp.matches("[K|C|F|S]")) {
                throw new DriverException("Invalid response for UNIT?" + resp);
            }
            return resp.charAt(0);
        } else {
            throw new DriverException("Invalid channel request " + channel);
        }
    }

    /**
     * Sets the temperature units [K, F, C, or S (sensor units) associated with
     * a channel [A,D]
     */
    public void setUnit(char channel, char unit) throws DriverException {
        Character upperCh = Character.toUpperCase(channel);
        Character upperUn = Character.toUpperCase(unit);
        if (upperCh >= 'A' && upperCh <= 'D') {
            if (!upperUn.toString().matches("[K|C|F|S]")) {
                throw new DriverException("Invalid Unit request " + unit);
            }
            String cmd = "INPUT " + upperCh.charValue() + ":UNITS "
                    + upperUn.charValue() + ";:*OPC?";
            String resp = scpi.read(cmd);
            int retVal = Integer.parseInt(resp);
            if (retVal != 1) {
                throw new DriverException("Invalid response for UNIT?"
                        + resp);
            }
            return;
        } else {
            throw new DriverException("Invalid channel request " + channel);
        }
    }

    /**
     * Sets the calibration curve type DIODE | ACR | PTC100 | PTC1K |NTC10UA |
     * ACR | TC70 | NONE associated with a channel [A,D]
     */
    public void setType(char channel, String typ) throws DriverException {
        Character upperCh = Character.toUpperCase(channel);

        if (upperCh >= 'A' && upperCh <= 'D') {
//            if (!typ.matches("DIODE|ACR|PTC100|PTC1K|NTC10UA|ACR|TC70|NONE"))
//5,0,21,22,0,0,0,0
            Integer ityp = typeMap.get(typ.toUpperCase());
            if (ityp == null) {
                throw new DriverException("Invalid Type request " + typ);
            }
            String cmd = "INPUT " + upperCh.charValue() + ":SENSOR "
                    + ityp + ";:*OPC?";
            String resp = scpi.read(cmd);
//            System.out.println("Output from cmd=("+cmd+") is resp = ("+resp+")");
            int retVal = Integer.parseInt(resp);
            if (retVal != 1) {
                throw new DriverException("Invalid response for TYPE?"
                        + resp);
            }
            return;
        } else {
            throw new DriverException("Invalid channel request " + channel);
        }
    }

    /**
     * setToControl
     */
    public void setToControl() throws DriverException {
        String cmd = "CONTROL:*OPC?";
        String resp = scpi.read(cmd);
        int retVal = Integer.parseInt(resp);
        if (retVal != 1) {
            throw new DriverException("Invalid response for TYPE?"
                    + resp);
        }
        return;
    }

    /**
     * isInControl
     */
    public boolean isInControl() throws DriverException {
        String cmd = "CONTROL?";
        String resp = scpi.read(cmd);
        boolean inControl = false;
        if (resp.contains("ON")) inControl = true;
        return(inControl);
    }
}
