package org.lsst.ccs.drivers.cryocon;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 * 
 * Program to test the Cryogenic Temperature Contorller 24C device driver
 * 
 * @author Heather Kelly
 ** 
 */
public class TestM24C {

    private M24C tempDev;

    /**
     * 
     * Opens connection to a device.
     */
    @Command(name = "open", description = "Open connection to Model 24C")
    public void open(
            @Argument(name = "type", description = "Communications type") int type,
            @Argument(name = "identity", description = "Device identifier") String ident,
            @Argument(name = "parameter", description = "Device parameter") int param)
            throws DriverException {
        tempDev = new M24C(type, ident, param);
    }

    /**
     * 
     * Opens connection to a Default BNL Temp Control Model 24C
     * 
     */
    @Command(name = "openDefault", description = "Open connection to Default Model 24C")
    public void openDefault() throws DriverException {
        tempDev = new M24C();
        tempDev.openDefault();
    }

    /**
     * 
     * Closes connection to a device.
     * 
     */
    @Command(name = "close", description = "Close connection to temp control 24C")
    public void close() throws DriverException {
        tempDev.close();
    }

    /**
     * 
     * Displays Temp Control 24C identification.
     * 
     */
    @Command(name = "showident", description = "Show Temp Control 24C identification")
    public String showIdentification() throws DriverException {
        String[] ident = tempDev.getIdentification();
        return "Manufacturer:    " + ident[0] + "\n" + "Model name:      "
                + ident[1] + "\n" + "DCI F/W version: " + ident[2] + "\n"
                + "AI F/W version:  " + ident[3];
    }

    /**
     * 
     * Get Source for Loop returns Channels A through D
     * 
     */
    @Command(name = "getMaxSetPoint", description = "Retrieve MaxSetPoint for loop")
    public double getMaxSetPoint(
            @Argument(name = "loop", description = "loop number [1-4]") int loop)
            throws DriverException {
        return tempDev.getMaxSetPoint(loop);
    }

    /**
     * 
     * Set MaxSetPoint for Loop [1-4]
     * 
     */
    @Command(name = "setMaxSetPoint", description = "setMaxSetPoint for loop")
    public void setMaxSetPoint(
            @Argument(name = "loop", description = "loop number [1-4]") int loop,
            @Argument(name = "temp", description = "temperature") double temp)
            throws DriverException {
        tempDev.setMaxSetPoint(loop, temp);
        return;
    }

    /**
     * 
     * 
     * Get Source for Loop returns Channels A through D
     * 
     */
    @Command(name = "getSetPoint", description = "Retrieve setPoint for loop")
    public double getSetPoint(
            @Argument(name = "loop", description = "loop number [1-4]") int loop)
            throws DriverException {
        return tempDev.getSetPoint(loop);
    }

    /**
     * 
     * Set SetPoint for Loop [1-4]
     * 
     */
    @Command(name = "setSetPoint", description = "setSetPoint for loop")
    public void setSetPoint(
            @Argument(name = "loop", description = "loop number [1-4]") int loop,
            @Argument(name = "temp", description = "temperature") double temp)
            throws DriverException {
        tempDev.setSetPoint(loop, temp);
        return;
    }

    /**
     * 
     * Get Source for Loop returns Channels A through D
     * 
     */
    @Command(name = "getLoopSource", description = "Retrieve source channel for loop")
    public char getLoopSource(
            @Argument(name = "loop", description = "loop number [1-4]") int loop)
            throws DriverException {
        return (tempDev.getSource(loop)).charAt(0);
    }

    /**
     * 
     * Set Source Channel for Loop
     * 
     */
    @Command(name = "setLoopSource", description = "Set source channel for loop")
    public void setLoopSource(
            @Argument(name = "loop", description = "loop number [1-4]") int loop,
            @Argument(name = "channel", description = "channel [A-D]") String channel)
            throws DriverException {
        tempDev.setSource(loop, channel.charAt(0));
        return;
    }

    /**
     * 
     * Get Temperature Reading For Channels A through D
     * 
     */
    @Command(name = "getTemp", description = "Retrieve temperature info")
    public double getTemp(
            @Argument(name = "channel", description = "Temp Channel to read") String channel)
            throws DriverException {
        char ch = channel.charAt(0);
        return tempDev.getTemp(ch);
    }

    /**
     ** 
     ** Get Temperature Units Channels A through D
     ** 
     */
    @Command(name = "getUnit", description = "Retrieve temperature units")
    public char getUnit(
            @Argument(name = "channel", description = "Temp Channel to read") String channel)
            throws DriverException {
        char ch = channel.charAt(0);
        return tempDev.getUnit(ch);
    }

    /**
     * 
     * Set Temperature Units for Channels A through D
     * 
     */
    @Command(name = "setUnit", description = "Set temperature units")
    public void setUnit(
            @Argument(name = "channel", description = "Temp Channel to set") String channel,
            @Argument(name = "units", description = "Units in [K|C|F|S]") String unit)
            throws DriverException {
        char ch = channel.charAt(0);
        char un = unit.charAt(0);
        tempDev.setUnit(ch, un);
        return;
    }
}
