package org.lsst.ccs.drivers.cryocon;

import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.scpi.Scpi;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * 
 * Enables access to a Cryogenic Temperature controller Model 24C
 * 
 * @author Heather Kelly
 * 
 */

/*
 * TODO define a common interface for temperature controllers. An issue is
 * temperature, which can be given in K, F, C, or sensor units depending on
 * controllers : how to unify?
 */

public class M24C {

    /**
     * Public constants
     */

    Logger log = Logger.getLogger("lsst.drivers.cryocon");

    /**
     * Private constants
     */

    /** Default connection settings at BNL */
    private final static int DEFAULT_TYPE = Scpi.CONN_TYPE_NETWORK;
    private final static int DEFAULT_PORT = 5000;
    private final static String BNL_DEFAULT_IP = "192.168.2.198";

    /**
     * Private fields
     */

    private Scpi scpi;

    public M24C() throws DriverException {
        scpi = new Scpi();
    }

    public M24C(int type, String ident, int param)
            throws DriverException {
        scpi = new Scpi();
        this.open(type, ident, param);
    }

    public void open(int type, String ident, int param) throws DriverException {
        scpi.open(type, ident, param);
        log.debug("Opened Cryogenic Temperature Controller");
    }

    /**
     *  Provide open for BNL Default settings
     */
    public void openDefault() throws DriverException {
        this.open(DEFAULT_TYPE, BNL_DEFAULT_IP, DEFAULT_PORT);
    }

    /**
     * Close the device
     * 
     */
    public void close() throws DriverException {
        scpi.close();
    }

    /**
     * Gets the device ID
     * 
     * @return The ID of the current open device
     */
    public String[] getIdentification() throws DriverException {
        String[] id = scpi.getIdentification();
        return id;
    }

/**
 * @return Maximum Set Point for the specified loop [1,4]
 */
    public double getMaxSetPoint(int loop) throws DriverException {
        if ((loop < 1) || (loop > 4))
            throw new DriverException(
                "M24C::getMaxSetPoint loop is invalid " + loop);
        String cmd = "LOOP " + Integer.toString(loop) + ":MAXSET?";
        String resp = scpi.read(cmd);
        return Double.parseDouble(resp.replaceAll("[a-zA-Z]",""));
    }

/**
 * Sets the Maximum Set Point for loop = [1,4], using the input temp.  
 * The units of the temperature may be K, F, C, or S (sensor units)
 * and can be checked with the getUnit command.
 */
    public void setMaxSetPoint(int loop, double temp) throws DriverException {
        if ((loop < 1) || (loop > 4))
            throw new DriverException(
               "M24C::setMaxSetPoint loop is invalid " + loop);
        String cmd = "LOOP " + Integer.toString(loop) + ":MAXSET "
                    + Double.toString(temp) + ";:*OPC?";
        String resp = scpi.read(cmd);
        int retVal = Integer.parseInt(resp);
        if (retVal != 1)
             throw new DriverException(
                  "M24C::setMaxSetPoint returned error code "
                            + resp);
        return;
    }

/**
 * @return current SetPoint for loop = [1,4]
 */
    public double getSetPoint(int loop) throws DriverException {
        if ((loop < 1) || (loop > 4))
            throw new DriverException(
                   "M24C::getSetPoint loop is invalid " + loop);
        String cmd = "LOOP " + Integer.toString(loop) + ":SETPT?";
        String resp = scpi.read(cmd);
        return Double.parseDouble(resp.replaceAll("[a-zA-Z]",""));
    }

/**
 * Sets the set point for the loop =[1,4], using temp where the temp units a
 * are in K, F, C, or S (sensor units), using setUnit and getUnit to set 
 * and check the temp units
 */
    public void setSetPoint(int loop, double temp) throws DriverException {
        if ((loop < 1) || (loop > 4))
            throw new DriverException(
                "M24C::setSetPoint loop is invalid " + loop);
        if (temp > getMaxSetPoint(loop))
            throw new DriverException(
                "M24C::setSetPoint requested temp is too high check MAXSET "
                       + temp);
        String cmd = "LOOP " + Integer.toString(loop) + ":SETPT "
                 + Double.toString(temp) + ";:*OPC?";
        String resp = scpi.read(cmd);
        int retVal = Integer.parseInt(resp);
        if (retVal != 1)
            throw new DriverException(
                "M24C::setSetPoint returned error code " + resp);
        return;

    }

/**
 * Checks on the channel associated with a specific loop = [1,4]
 * @return channel [A,D]
 */
    public String getSource(int loop) throws DriverException {
        if ((loop < 1) || (loop > 4))
            throw new DriverException(
                "M24C::getSource loop is invalid " + loop);
        String cmd = "LOOP " + Integer.toString(loop) + ":SOURCE?";
        String resp = scpi.read(cmd);
        if (resp.charAt(0) >= 'A' && resp.charAt(0) <= 'D')
            return resp;
        else
            throw new DriverException(
               "M24C::getSource invalid response " + resp);
    }

/**
 * Set the channel [A,D] source for a loop [1,4]
 */
    public void setSource(int loop, char channel) throws DriverException {
        if ((loop < 1) || (loop > 4))
            throw new DriverException(
                "M24C::setSource loop is invalid " + loop);
        if (channel >= 'A' && channel <= 'D') {
            String cmd = "LOOP " + Integer.toString(loop) + ":SOURCE "
                    + channel + ";:*OPC?";
            String resp = scpi.read(cmd);
            int retVal = Integer.parseInt(resp);
            if (retVal != 1)
                throw new DriverException(
                    "M24C::setSource returned error code " + resp);
            return;
        } else {
            throw new DriverException(
                "M24C::setSource channel is invalid " + channel);
    }
}

/**
 * Queries the temperature of a channel [A,D]
 * @return temperature in units of K, F, C, or S (sensor units) depending on
 * the channel units which can checked and set via getUnit and setUnit
 */
    public double getTemp(char channel) throws DriverException {
        double temp = -999.;
        if (channel >= 'A' && channel <= 'D') {
            String cmd = "INPUT " + channel + ":TEMP?";
            String resp = scpi.read(cmd);
            if (resp.contains("...")) {
                log.debug("CryoCon channel " + channel + " appears not to be connected.");
            } else {
                 temp = Double.parseDouble(resp.replaceAll("[a-zA-Z]","")); // remove units before converting 
            }
        } else {
            throw new DriverException("Invalid channel request " + channel);
        }      
        return(temp);
    }

/**
 * Queries the temperatur units associated with channel [A,D]
 * @return temperatue unit K, F, C, S (sensor units)
 */
    public char getUnit(char channel) throws DriverException {
        char upperCh = Character.toUpperCase(channel);
        if (upperCh >= 'A' && upperCh <= 'D') {
            String cmd = "INPUT " + upperCh + ":UNIT?";
            String resp = scpi.read(cmd);
            if (!resp.matches("[K|C|F|S]"))
                throw new DriverException("Invalid response for UNIT?" + resp);
            return resp.charAt(0);
        } else {
            throw new DriverException("Invalid channel request " + channel);
        }
    }

/**
 * Sets the temperature units [K, F, C, or S (sensor units)  associated 
 * with a channel [A,D]
 */
    public void setUnit(char channel, char unit) throws DriverException {
        Character upperCh = Character.toUpperCase(channel);
        Character upperUn = Character.toUpperCase(unit);
        if (upperCh >= 'A' && upperCh <= 'D') {
            if (!upperUn.toString().matches("[K|C|F|S]"))
                throw new DriverException("Invalid Unit request " + unit);
            String cmd = "INPUT " + upperCh.charValue() + ":UNITS "
                    + upperUn.charValue() + ";:*OPC?";
            String resp = scpi.read(cmd);
            int retVal = Integer.parseInt(resp);
            if (retVal != 1)
                throw new DriverException("Invalid response for UNIT?"
                     + resp);
            return;
        } else {
            throw new DriverException("Invalid channel request " + channel);
        }   
    }
}
