package org.lsst.ccs.drivers.chiller;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 * Chiller test program driver
 */
public class TestChiller {

    private final Chiller chiller;

    public TestChiller() {
        chiller = new Chiller();
    }

   /**
    * Opens connection to chiller
    *
    * @mparam host The host name
    * @param port The port number
    * @throws DriverException
    */
    @Command(name = "open", description = "Open connection to chiller")
    public void open(@Argument(description = "Host name") String host,
            @Argument(description = "Port number", defaultValue = "9760") int port) throws DriverException {
        chiller.open(host, port);
    }

   /**
    * Closes connection to a device.
    *
    * @throws DriverException
    */
    @Command(name = "close", description = "Close connection to chiller")
    public void close() throws DriverException {
        chiller.close();
    }

   /**
    *  Set debug mode
    *
    *  @param  debug
    */
    @Command(name = "setDebug", description = "set debug mode true/false")
	public void setDebug(@Argument(description = "true/false") 
                             boolean debug) {
	    chiller.setDebug(debug);
    }

   /**
    *  List queries
    *
    *  @return  Table listing enumeraed queries with descroptions
    */
    @Command(name = "listQueries", description = "List available queries")
    public String listQueries() {
        return chiller.listQueries();
    }

   /**
    *  List parameter-setting commands
    *
    *  @return  Table listing enumeraed parameter-setting commands
    */
    @Command(name = "listSetParams", 
             description = "List implemented parameter-setting commands")    
    public String listSetParams() {
        Chiller.SetParam[] val = Chiller.SetParam.values();
        String table = "Implemented parameter-setting commands:";
        for (int i = 0; i < val.length; i++) {
            table += ("\n  " + val[i].getDescription());
        }
        return table;
    }

   /**
    *  List status bits
    *
    *  @return  Table listing enumeraed status bits with descroptions
    */
    @Command(name = "listStatusBits", description = "List status register bits")
    public String listStatusBits() {
        return chiller.listStatusBits();
    }

   /*  Chiller query commands   */

   /**
    *  Send query to Chiller and show response
    *
    *  @param  quantity (enumerated Query)
    *  @return String
    *  @throws DriverException
    */
    @Command(name = "queryChiller", description = "send specified query to chiller")
    public String queryChiller(@Argument(description = "Read Enumerated quantity from Chiller") Chiller.Query quantity) throws DriverException {
        String response = chiller.queryChiller(quantity);
        return response;
    }

   /**
    *  Show a temperature value (extracted from query response)
    *
    *  @param  Chiller.Query (wprks for temperature readings and set-points)
    *  @return String representation of extracted value
    *  @throws DriverException
    */
    @Command(name = "showTemperature", description = "numerical value of a temperature reading or set-point")
    public String showTemperature(@Argument(description = "which quantity")
	Chiller.Query quantity) throws DriverException {
        String temp = "Example of getTemperature " + quantity.toString() + 
	    ":  " + chiller.getTemperature(quantity);
        return temp;
    }

   /**
    *  Show a flow-rate value (extracted from query response)
    *
    *  @param  Chiller.Query 
    *  @return String representation of extracted value
    *  @throws DriverException
    */
    @Command(name = "showFlow", description = "numerical value of a flow reading or set-point")
    public String showFlow(@Argument(description = "which quantity")
	Chiller.Query quantity) throws DriverException {
        String flow = "Example of getFlow " + quantity.toString() + 
	    ":  " + chiller.getFlow(quantity);
        return flow;
    }

   /**
    *  Show a pressure value (extracted from query response)
    *
    *  @param  Chiller.Query 
    *  @return String representation of extracted value
    *  @throws DriverException
    */
    @Command(name = "showPressure", description = "numerical value of a pressure reading")
    public String showPressure(@Argument(description = "which quantity")
	Chiller.Query quantity) throws DriverException {
        String pressure = "Example of getPresssure " + quantity.toString() + 
	    ":  " + chiller.getPressure(quantity);
        return pressure;
    }

   /**
    *  Show a lifetime value (extracted from query response)
    *
    *  @param  Chiller.Life 
    *  @return String representation of extracted value
    *  @throws DriverException
    */
    @Command(name = "showLifetime", description = "numerical value of lifetime")
    public String showLifetime(@Argument(description = "which quantity")
	Chiller.Life quantity) throws DriverException {
        String lifetime = "Example of getLifetime " + quantity.getDescription()
	    + ":  " + chiller.getLifetime(quantity);
        return lifetime;
    }

   /**
    *  Show heating and cooling percentages (extracted from query)
    *
    *  @return  String representation of extracted values
    *  @throws DriverException
    */
    @Command(name = "showHeatCool", description = "numerical values of heat/cool percentages")
	public String showHeatCool() throws DriverException {
	double percentages[] = chiller.getHeatCool();
        String heatcool = "Heat, Cool percentages = " + percentages[0]
            + ", " + percentages[1];
        return heatcool;
    }

   /**
    *  Show event register bits
    *
    *  @return String listing any set bits
    *  @throws DriverException
    */
    @Command(name = "showEvtReg", description = "Show decoded Event Register")
    public String showEvtReg() throws DriverException {
        return chiller.decodeEvtReg();
    }

   /**
    *  Show status register bits
    *
    *  @return String listing any set bits
    *  @throws DriverException
    */
    @Command(name = "showStatusReg", description="Show decoded Status Register")
    public String showStatusReg() throws DriverException {
        return chiller.decodeStatusReg();
    }

   /**
    *  Show error and warning words in hex Ascii
    *
    *  @return String
    *  @throws DriverException
    */
    @Command(name = "showErrorWords", 
             description = "Show error and warning words in hex")
    public String showErrorWords() throws DriverException {
	return chiller.getErrorWords().toString();
    }

   /*  Commands to read or set chiller parameters   */

   /**
    *  Read a specified Chiller setup parameter
    *
    *  @param  index of setup parameter
    *  @return String
    *  @throws DriverException
    */
    @Command(name = "readParameter", description = "Read value of specified parameter")
    public String readParameter(@Argument(description = "index of parameter")
     int index) throws DriverException {
        return chiller.readParameter(index);
    }

   /**
    *  Set a specified Chiller setup parameter
    *
    *  @param  index of setup parameter
    *  @param  value to be set (as a String)
    *  @throws DriverException
    */
    @Command(name = "setAnyParameter", description = "Set value of specified parameter")
    public void setAnyParameter(@Argument(description = "index of parameter")
			        int index, @Argument(description = 
                                "value to be set") String value) 
     throws DriverException {
        chiller.setParameter(index, value);
        return;
    }

   /**
    *  Set and read back a specified Chiller setup parameter
    *
    *  @param  index of setup parameter
    *  @param  value to be set (as a String)
    *  @return read-back value
    *  @throws DriverException
    */
    @Command(name = "setAndReadAnyParameter", 
             description = "Set value of specified paramete and read it backr")
    public String setAndReadAnyParameter(
     @Argument(description = "index of parameter") int index,
     @Argument(description = "value to set") String value) 
     throws DriverException {
        return chiller.setAndReadParameter(index, value);
    }

   /**
    *  Command to execute a parameter-setting command, enumerared in
    *  Chiller.SetParam, checking that value gets set as requested.
    *
    *  @param  SetParam (specifies command to Chiller controller)
    *  @param  Arguments to send to Chiller
    *  @throws DriverException
    */
    @Command(name = "setParams", 
             description = "Invoke Chiller command tp set parameter(s)")
    public void setParams(
     @Argument(description = "enumerated command") Chiller.SetParam setParam, 
     @Argument(description = "parameter value(s)") double... values)
     throws DriverException {
	chiller.setParamCommand(setParam, values);
        return;
    }

   /**
    *  Select DUT vs Normal temperature-control ode
    *
    *  @param  boolean true for DUT, false for Normal
    *  @throws DriverException
    */
    @Command(name = "selectDUT", 
             description = "Select DUT vs Normal temperature-control mode")
    public void selectDUT(@Argument(description="true for DUT") boolean dut) 
     throws DriverException {
        setParams(Chiller.SetParam.TEMP_CTRL_MODE, dut ? 1. : 0.);
    }

   /*  Chiller control commands   */

   /**
    *  Set chiller temperature and go there using default ramp
    *
    *  @param  double temperature in degrees
    *  @throws DriverException
    */
    @Command(name = "setTemperature", 
	     description = "go to temperature setting using default ramp")
    public void setTemperature(@Argument(description="tempeature in degrees")
			       double temperature) throws DriverException {
        chiller.setTemperature(temperature);
    }

   /**
    *  Set chiller temperature and go there using provided ramp rate
    *
    *  @param  double temperature in degrees
    *  @param  double ramp in degrees/minute
    *  @throws DriverException
    */
    @Command(name = "setTemperatureWithRaRamp", 
	     description = "go to temperature setting using provided ramp")
    public void setTemperatureWithRamp(
        @Argument(description="tempeature in degrees") double temperature, 
        @Argument(description="ramp in degrees/min") double ramp) 
        throws DriverException {
        chiller.setTemperatureWithRamp(temperature, ramp);
    }

   /**
    *  Set chiller flow rate
    *
    *  @param  double flow (e.g., gallons per monute)
    *  @throws DriverException
    */
    @Command(name = "setFlow", 
	     description = "go to temperature setting using default ramp")
    public void setFlow(@Argument(description="flow rate in gpm")
			double flow) throws DriverException {
        chiller.setFlow(flow);
    }

   /**
    *  Clear chiller error and warning bits
    *
    *  @throws DriverException
    */
    @Command(name = "clearErrors",
             description = "clear chiller errpr and warning words")
    public void clearErrors() throws DriverException {
	chiller.clearErrors();
    }

   /**
    * Send non-Query command string to chiller. Debug-mode only.
    *
    * @param  command (String)
    * @param  args (arguments, if any)
    * @throws DriverException
    */
    @Command(name = "sendDebugCommand", description = "Send non-query command to chiller, debug-mode")
    public void sendDebugCommand(@Argument(description = "command String")
        String command, @Argument(description = "arguments to command") 
        String... args) throws DriverException {
        chiller.sendDirectCommand(command, args);
    }
}
