package org.lsst.ccs.drivers.chiller;

import java.io.OutputStream;
import java.time.Duration;
import org.lsst.ccs.drivers.ascii.Ascii;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;

/**
 * Driver for inTEST Chiller.
 */
public class Chiller extends Ascii {

    private static final int DEFAULT_PORT = 9760;
    // Timeouts in ms
    private static final int TIMEOUT = 200;       // Timeout for read response
    private static final int TIMEOUT_SET = 1000;  // Timeout for settings
    // Tolerance for setting temperature and flow setpoints
    private static final double SETPOINT_TOLERANCE = 0.1;  // degrees
    private static final double SETFLOW_TOLERANCE = 0.1;   // gpm
    // Maximum index of Chiller setup ("F") parameters
    private static final int MAX_SETUP_PARAM = 77;  
    // Label identifying source of messages
    private final static String Lbl = "Chiller ";

    private final Duration intervalCheckSet = Duration.ofMillis(10);

    private int telnetPort = DEFAULT_PORT;
    private int timeout = TIMEOUT;
    private boolean debug = false;
    private boolean first = true;
    private Query[] qval;

   /**
    *  Enumeration of all "Immediate" Query/Read commands to Chiller,
    ^  except QFA and REA (both handled separately).
    */
    public enum Query {

        IDENTITY     ("*IDN?", "Controller Identity",             1, false),
        LOOP_COUNT   ("LCT",   "Loop Count (Program Mode)",       1, false),
	T_PROBE2     ("PT 2",  "Temperature Probe 2 (type F35)",  1, true),
        LAT_CMND     ("QC",    "Last Command String",             1, false),
        ERR_BITS_OLD ("QCE",   "Error1, Error2, Warning1",        1, false),
        DIAG_INFO    ("QDI",   "Diagnostics Info",                2, false),
        ERROR_BITS   ("QEW",   "Error1, Error2, Warn1, Warn2",    1, false),
        // PARAMETERS   ("QFA",    "Setup Parameter",                1, false);
	TANK_P_SET   ("QFA 73","Tank pressure setpoint",          1, false),
        LIFETIME     ("QLP",   "Life Time Parameters",            4, false),
        REMOTE_MODE  ("QM",    "1 = Immediate, 2 = Program",      1, false),
        SERIAL_NO    ("QN",    "Controller Serial Number",        1, false),
        PGM_STATE    ("QPG",   "Remote Program Running State",    1, false), 
	HEAT_COOL    ("QPL",   "Heat and Cool Percents of Max",   1, false),
        TEMP_RANGE   ("QR",    "Controller Temperature Range",    1, false),
        SET_POINT    ("QS",    "Probe Number and Set Point",      1, true ),
        DYNAMIC_SET  ("QSC",   "Probe No. and Control Set Point", 1, true ), 
        SYS_INFO     ("QSI",   "System Information",              7, false),
        FIRMWARE     ("QV",    "Firmware Version",                1, false),
	WARN2        ("QWT",   "Warning2 (bits)",                 1, false),
        FLOW_RATE    ("RCF",   "Flow Rate (GPM or LPM)",          1, false),
        STATUS       ("RCS",   "Chiller Status (hex)",            1, false),
	// EVENT_BITS   ("REA",   "Event Bits (hex)",                1, false),
        INPUT_BITS1  ("RID 1", "5V Digital Inputs (binary)",      1, false),
        INPUT_BITS2  ("RID 2", "24V Digital Inputs (binary)",     1, false),
        PRESSURE_IN  ("RIP",   "Input Pressure",                  1, false),
        OUTPUT_BITS1 ("ROD 1", "5V Digital Outputs (binary)",     1, false),
        OUTPUT_BITS2 ("ROD 2", "24V Digital Outputs (binary)",    1, false),
        PRESSURE_OUT ("ROP",   "Output Pressure",                 1, false),
	PID_PARAM    ("RP",    "PID Constants (F0, F10 to F12)",  1, false),
	FLOW_SETPT   ("RPS",   "Pressure or Flow Setpoint",       1, false),
	STATE_BITS   ("RSA",   "Chiller State Bits (hex)",        1, false),
	GPIB_TIMEOUT ("RTM",   "GPIB timeout (F44)",              1, false),
        PRESSURE_TANK("RTP",   "Tank Pressure",                   1, false),
        TEMPERATURE  ("TP 1",  "Coolant Output Temperature",      1, true ),
        T_CONDENSER  ("TP 5",  "Temperature CondensorOut",        1, true ),
        T_TXV_BULB   ("TP 6",  "Temperture TXV_Bulb",             1, true ),
        T_STAGE2EVAP ("TP 7",  "Temperature Stage2Evap",          1, true );

        private String command;     // Command to be sent to chiller
        private String description; // Description of commands
        private int expectedLines;  // expected number of lines in response
        private boolean isTemp;     // true if returns temperature as last field

        Query(String command, String description, int expectedLines,
              boolean isTemp) {            
            this.command = command;
            this.description = description;
            this.expectedLines = expectedLines;
            this.isTemp = isTemp;
        }

        public String getCommand() {return command;}
        public String getDescription() {return description;}
	public int getExpectedLines() {return expectedLines;}
	public boolean getIsTemp() {return isTemp;}

    }

   /**
    *  Enumeration of bits in Event Register
    */
    public enum EventRegister {
        BAD_ARG    (0x01, "Bad_Argument "),
        PGM_FULL   (0x02, "PgmBuffer_0.75Full "),
        BAD_CMND   (0x04, "UnrecognizedCmnd "),  
        DWELLTIME  (0x08, "DwellTime_Done "),	   
        SETPOINT   (0x10, "SetPoint_Reached "),  
        CMND_OVFL  (0x20, "CmndBufferOverrun  "),  
        UNUSED_6   (0x40, "Unused_Bit6 "),  
	UNUSED_7   (0x80, "Unused_Bit7 ");

        private int mask;
        private String descr;

        EventRegister(int mask, String descr) {
            this.mask = mask;
            this.descr = descr;
        }

        public static String decode(int register) {
            String decoded = "EventRegister:  ";
            EventRegister[] bits = EventRegister.values();
            int nbits = bits.length;
            for (int i = 0; i < nbits; i++) {
                if ((register & bits[i].mask) != 0) {
                    decoded += (bits[i].toString() + " ");
                }
            }
            return decoded;
        }
    }

   /**
    *  Enumeration of bits in Status Register
    */
    public enum StatusRegister {
        ERROR      (0x01, "Error(s) present "),
        WARNING    (0x02, "Warning(s) present "),
        PGM_FULL   (0x04, "Program Buffer over 0.75 full "),  
        T_CONTROL  (0x08, "RunState: Controlling T "),	   
        AT_SETPT   (0x10, "At Temperature SetPoint "),  
        SRQ_SET    (0x20, "SRQ Set "),  
        CMPRS_ON   (0x40, "Compressors on "),  
	UNUSED_7   (0x80, "Unused Bit7 ");

        private int mask;
        private String descr;

        StatusRegister(int mask, String descr) {
            this.mask = mask;
            this.descr = descr;
        }

        public int getMask() {return mask;}
        public String getDescr() {return descr;}

        public static String decode(int register) {
            String decoded = "StatusRegister = " + Integer.toHexString(register) + "x:";
            StatusRegister[] stat = StatusRegister.values();
            int nbits = stat.length;
            for (int i = 0; i < nbits; i++) {
                String st = ((register & stat[i].mask) != 0) ? "Yes" : "No ";
                decoded += ("   " + stat[i] + " " + st);
            }
            return decoded;
        }
    }

   /**
    *  Enumeration of quantities returned by LIFETIME query
    */
    public enum Life {
        CONTROL   ("Controller Hours"),
	COMPRESS  ("Compressor One Hours"),
	PUMP      ("Pump Hours"),
	VALVE     ("Valve Activation Count");

        private String descr;

        Life(String descr) {
            this.descr = descr;
        }

        public String getDescription() {return descr;}
    }

   /**
    *  Enumeration of chiller parameters to be checked after Set commands
    */
    public enum FParam {
        PID_KP       ( 0, 1.0, 1.,    64. , "", "PID proportional coeff"),
	COLD_PERIOD  ( 2, 1.0, 1.,    32. , "s","Cold valve period (s)"),
	CLR_KI_SETPT ( 5, 1.0, 0.,    1.,   "", "If 1, clear Ki at setpoint"),
	PID_KI       (10, 1.0, 0.,    3200, "", "PID integral coeff"),
	PID_KD       (11, 1.0, 0.,    3200, "", "PID differential coeff"),
	PID_GAINFAC  (12, 1.0, -99.,  99.,  "", "PID cool (<0)/heat (>0) factor"),
	HEAT_CAPACITY(15, 1.0, 1.,    100., "", "Heat capacity, % of max"),
	DUT_TMIN     (27, 0.1, -70.0, 30.0 ,"\u00b0C", "DUT lower temp limit"), 
	DUT_TMAX     (28, 0.1, -70.0, 30.0 ,"\u00b0C", "DUT upper temp limit"),
	DUT_DT_LO    (29, 0.1, 1.0,   300.0,"\u00b0C","DUT lower DeltaT limit"),
	DUT_DT_HI    (30, 0.1, 1.0,   300.0,"\u00b0C","DUT upper DeltaT limit"),
	SETPT_TOL    (31, 0.1, 0.1,   10.0, "\u00b0C", "SetPoint tolerance (deg-C)"),
	SETPT_SETTLE (32, 1.0, 0.,    59. , "s","Min SetPoint stable time (s)"),
        THERMSL_MASS (34, 1.0, 1.,    1000.,"", "Thermal mass (DUT damping)"),
	REFRIG_OFF   (36, 1.0, 5.,    10.,  "min", "Refrig-off timer (min)"),
	RAMP_DEFAULT (37, 0.1, 0.0,   500.0,"\u00b0C/min", "Default ramp rate (deg/min)"),
	FLOWMETER_LO (38, 0.1, 0.0,   99.0, "gal/min", "Low-flow-meter alarm thresh"),
	T_CTRL_MODE  (40, 1.0, 0.,    1.,   "", "T-control Normal (0)/DUT (1)"),
	FLOW_ERR_MASK(58, 1.0, 1.,    180., "s","Flow-switch error mask time (s)"),
	DUT_KP       (59, 0.1, 0.1,   64.0, "", "DUT proportional coeff"),
	DUT_KI       (60, 0.1, 0.0,   64.0, "", "DUT integral coeff"),
	COOL_CAPACITY(64, 1.0, 1.,    100., "", "Cool capacity, % of max"),
        BUBBLEV_TIME (72, 1.0, 1.,    10.,  "min","Bubble valve open duration"),
        TANK_SETPOINT(73, 1.0, 0.0,   50.0, "psig", "Tank pressure setpoint");

        private int number;      // parameter number
        private double tol;      // tolerance (least significant digit)
        private double min;      // minimum value
        private double max;      // maximum value
        private String units;    // units (for publication)
        private String descr;    // description of parameter

        FParam(int number, double tol, double min, double max, String units, String descr) {
            this.number = number;
            this.tol = tol;
            this.min = min;
            this.max = max;
            this.units = units;
            this.descr = descr;
        }

        public int getNumber() {return number;}
        public double getTolerance() {return tol;}
        public double getMin() {return min;}
        public double getMax() {return max;}
        public String getUnits() {return units;}
        public String getDescription() {return descr;}

    }

   /**
    *  Subclass to hold error and warning words from Chiller Controller
    */
    public static class ErrorWords {
        private int err1, err2, warn1, warn2;
        public ErrorWords(int err1, int err2, int warn1, int warn2) {
            this.err1 = err1;
            this.err2 = err2;
            this.warn1 = warn1;
            this.warn2 = warn2;
        }

        public int getError1() {return err1;}
        public int getError2() {return err2;}
        public int getWarning1() {return warn1;}
        public int getWarning2() {return warn2;}

        @Override
	public String toString() {
            String err = 
                "Error1 = " + Integer.toHexString(err1) + "x " +
                "Error2 = " + Integer.toHexString(err2) + "x " +
                "Warning1 = " + Integer.toHexString(warn1) + "x " +
                "Warning2 = " + Integer.toHexString(warn2) + "x ";
            return err;
        }

        @Override
	public boolean equals(Object y) {
            ErrorWords x = (ErrorWords) y;
            boolean eq = (err1 == x.getError1() && err2 == x.getError2() &&
			  warn1 == x.getWarning1() && warn2 == x.getWarning2());
	    return eq;
        }
    }

   /**
    *  Constructor.
    */
    public Chiller() {
        qval = Query.values();
    }

    /**
     * Open a connection to the chiller
     *
     * @param  host  The host to connect to
     * @param  port  Telnet port
     * @throws DriverException 
     */
    public void open(String host, int port) throws DriverException {
        super.open(ConnType.NET, host, port);
        setTimeout(timeout);
        setTerminator(Terminator.CRLF);

        /**
         *   Initialize some controller settings
         */

        try {
            // if (first) {write("DC");}    // Soft reset of controller
            // Temporarily skip this reset.

            write("SI");    // Set immediate mode
            write("DP2");   // Two decimal places for temperature readings
            write("DS");    // Disable SRQ (not used)

            // Fixed F# parameters

            setParameter( 1, "2");       // Second probe is for DUT
            setParameter(16, "0");       // Temperature units Celsius
            setParameter(39, "0");       // Flow units GPM
            setParameter(47, "0");       // Pressure units PSI
            setParameter(35, "5");       // Select digital input for DUT mode

            first = false;
        }
        catch (DriverException e) {
            closeSilent();
            throw e;
        }
    }
    
    /**
     * Open a connection to the chiller using default port
     *
     * @param  host  The host to connect to
     * @throws DriverException 
     */
    public void open(String host) throws DriverException {
        open(host, DEFAULT_PORT);
    }

   /** 
   *  Close connection.  (Uses parent method.)
    *
    * @throws DriverException
    */

   /**
    *  Set debug mode true or false
    *
    *  @param  on_off
    */
    public void setDebug(boolean on_off) {
        debug = on_off;
    }

   /**
    *  List enumerared queries
    *
    *  @return  Table of queries abd descriptions
    */
    public String listQueries() {
        String table = "List of all queries for Chiller \n";
        for (int i = 0; i < qval.length; i++) {
            table += String.format("\n   %-15s  (%-5s)     %s", qval[i],
                                   qval[i].getCommand(),
                                   qval[i].getDescription());
        }
        table += "\n";
        return table;
    }

   /**
    *  List Status Register bits
    *
    *  @return  Table of status bits and descriptions
    */
    public String listStatusBits() {
        StatusRegister[] stat = StatusRegister.values();
        String table = "List of status bits \n";
        for (int i = 0; i < stat.length; i++) {
            table += String.format("\n   %-10s   %s", stat[i], 
                                   stat[i].getDescr());
        }
        return table;
    }

   /**
    *  List enumerated parameters
    *
    *  @return  Table of enumerated parameters
    */
    public String listNamedParameters() {
        String table = "List of named Chiller setup parameters \n";
        FParam[] pars = FParam.values();
        for (FParam par : pars) {
            table += String.format("\n   %-15s  (F%02d)     %s", par,
                                   par.getNumber(), par.getDescription());
        }
        table += "\n";
        return table;
    }

   /**
    ^  Send an enumerated query to Chiller and receive response
    *
    *  @param  query   enumerated command identifier
    *  @return value   value of quantity requested
    *  @throws DriverException
    */
    public synchronized String queryChiller(Query query) throws DriverException
    {
        String reply = read(query.getCommand());
        int expect = query.getExpectedLines();
        if (expect > 1) {
            for (int nlines = 1; nlines < expect; nlines++) {
		reply += ("\n" + read());
	    }
        }
        return reply;
    }

   /**
    *  Return any temperature (reading or set point)n as a double
    *
    *  @param  Query enumeraion of desired temperature reading
    *  @return temperature
    *  @throws DriverException
    */
    @SuppressWarnings("SleepWhileInLoop")
    public double getTemperature(Query query) throws DriverException {
        if (!query.getIsTemp()) {
            throw new IllegalArgumentException("Quantity is not a temperature");
        }
        String response = queryChiller(query);
	String substr = response.substring(response.lastIndexOf(' ')+1);
        double value = Double.parseDouble(substr);
        return value;
    }

   /**
    *  Return flow rate (reading or setpoint)
    *
    *  @param  Query enumeraion of desired flow reading
    *  @return flow
    *  @throws DriverException
    */
    public double getFlow(Query query) throws DriverException {
        if (query != Query.FLOW_RATE && query != Query.FLOW_SETPT) {
            throw new IllegalArgumentException("Arg must be a flow quantity");
        }
        String response = queryChiller(query);
	String substr = response.substring(response.lastIndexOf(' ')+1);
        double value = Double.parseDouble(substr);
        return value;
    }

   /**
    *  Return a pressure reading
    *
    *  @param  Query enumeraion of desired pressure reading
    *  @return pressure
    *  @throws DriverException
    */
    public double getPressure(Query query) throws DriverException {
        if (query != Query.PRESSURE_IN && query != Query.PRESSURE_OUT &&
            query != Query.PRESSURE_TANK && query != Query.TANK_P_SET) {
            throw new IllegalArgumentException("Arg must be pressure quantity");
        }
        String response = queryChiller(query);
	String substr = response.substring(response.lastIndexOf(' ')+1);
        double value = Double.parseDouble(substr);
        return value;
    }

   /**
    *  Return heating and cooling as percentages (of maximum capacity?).
    *  (Heat is duty cycle of heater; Cool is duty cycle of cold valve.)
    *  No more than one of these should be non-zero at any one time.
    *
    *  @return  percentages[heat,cool]
    *  @throws DriverException
    */
    public double[] getHeatCool() throws DriverException {
        String[] resp = queryChiller(Query.HEAT_COOL).split("%");
        double[] percentages = new double[2];
        for (int i = 0; i < 2; i++) {
            String substr = resp[i].substring(resp[i].lastIndexOf("=") + 1);
            percentages[i] = Double.parseDouble(substr);
        }
        return percentages;
    }

   /**
    *  Return a Lifetime quantity
    *
    *  @param  Enumerated Life
    *  @return value (int)
    *  @throws DriverException
    */
    public int getLifetime(Life life) throws DriverException {
        String[] lifetime = queryChiller(Query.LIFETIME).split("\n");
        String line = lifetime[life.ordinal()];
        String substr = line.substring(line.lastIndexOf(' ')+1);
        int value = Integer.parseInt(substr);
        return value;
    }
        
   /**
    *  Decode event register
    *
    *  @return String showing which bits are set
    *  @throws DriverException
    */
    public String decodeEvtReg() throws DriverException {
        String resp = queryEvtReg();
        String substr = resp.substring(resp.length()-2, resp.length());
        int register = Integer.parseUnsignedInt(substr, 16);
        return resp + "  " + EventRegister.decode(register);
    }

   /**
    *  Get status register
    *
    *  @return  Status register as int value
    *  @throws DriverException
    */
    public int getStatusReg() throws DriverException {
        String resp = queryChiller(Query.STATE_BITS);
        String substr = resp.substring(resp.length()-2, resp.length());
        return Integer.parseUnsignedInt(substr, 16);
    }

   /**
    *  Decode status register
    *
    *  @return String showing which bits are set
    *  @throws DriverException
    */
    public String decodeStatusReg() throws DriverException {
        return StatusRegister.decode(getStatusReg());
    }

   /**
    *  Extract and return error and warning words
    *
    *  @throws DriverException
    */
    public ErrorWords getErrorWords() throws DriverException  {
        String errors = queryChiller(Query.ERROR_BITS);
        int idx1 = errors.indexOf(' ');
        int idx2 = errors.indexOf(' ', idx1+1);
        int idx3 = errors.indexOf(' ', idx2+1);
        int idx4 = errors.indexOf(' ', idx3+1);
        int err1 = Integer.parseInt(errors.substring(idx1+1,idx2));
        int err2 = Integer.parseInt(errors.substring(idx2+1,idx3));
        int warn1 = Integer.parseInt(errors.substring(idx3+1,idx4));
        int warn2 = Integer.parseInt(errors.substring(idx4+1));
        return new ErrorWords(err1, err2, warn1, warn2);
    }

   /**
    ^  Send event-register query (REA) to Chiller and receive response
    *  Treated separately from generic query, because reading clears value.
    *
    *  @return reply   value of event register response
    *  @throws DriverException
    */
    private synchronized String queryEvtReg() throws DriverException {
        String reply = read("REA");
        return reply;
    }

   /**
    *  Read a chiller setup parameter
    *
    *  @param  index
    *  @return value (as a String)
    *  @throws DriverExceptiom
    */
    public synchronized String readParameter(int index) throws DriverException {
        if (index >= 0 && index <= MAX_SETUP_PARAM) {
            String command = "QFA " + index;
            String reply = read(command);
            String[] lines = reply.split("/n");
            if (lines.length != 1) {
                throw new DriverException("Expected 1 line in reply, received "
                                          + lines.length);
            }
            return reply;
        }
        throw new IllegalArgumentException("Index must be in range 0 to " + 
                                           MAX_SETUP_PARAM);
    }

   /**
    *  Read setup parameter identified by enumerated name
    *
    *  @param  FParam
    *  @return value (as a String)
    *  @throws DriverExceptiom
    */
    public String readNamedParameter(FParam param) throws DriverException {
        return readParameter(param.getNumber());
    }

   /**
    *  Set a chiller setup parameter.
    *  For use internally and in testing only. 
    *  (It lacks validation of value and checking of result.)
    *
    *  @param  index
    *  @param  value (as a String)
    *  @throws DriverExceptiom
    */
    void setParameter(int index, String value) throws DriverException {
        if (index >= 0 && index <= MAX_SETUP_PARAM) {
            // Truncate value to at most one character after decimal point
            int point = value.lastIndexOf('.');
	    String command = "SPA " + index + " " + ((point > -1 && value.length()-point > 2) ? value.substring(0, point + 2) : value);       
            write(command);
            return;
        }
        throw new IllegalArgumentException("Index must be in range 0 to " +
                                           MAX_SETUP_PARAM);
    }

   /**
    *  Set and read back a chiller setup parameter
    *  For use in testing only.
    *
    *  @param  index
    *  @param  value (as a String)
    *  @throws DriverExceptiom
    */
    String setAndReadParameter(int index, String value) 
     throws DriverException {
        setParameter(index, value);
        try {
            Thread.sleep(TIMEOUT_SET);
        } catch (InterruptedException ex) {
            throw new RuntimeException("Interrupt between se and read",ex);
        }

        return readParameter(index);
    }

   /**
    *  Implement parameter-setting command (enumerated in FParam).
    *  Checks value and waits for set values (or times out);
    *
    *  @param  FParam (named CHiller parameter to set)
    *  @param  double (value to be set)
    *  @throws DriverException
    */
    @SuppressWarnings("SleepWhileInLoop")
    public void setParamCommand(FParam param, double value) 
        throws DriverException {

        /* Check value of parameter */
        if ((value < param.getMin())||(value > param.getMax())) {
            throw new IllegalArgumentException(String.format("Parameter %s must be between %f and %f", param, param.getMin(), param.getMax()));
        }

        /* Check precision of input value - must not be less than tol */
        double testp = value/param.getTolerance();
        double testd = (double) Math.round(testp);
        // Allow for rounding error in ratio
        if (Math.abs(testd - testp) >= 0.0000001) {
            throw new IllegalArgumentException(String.format("Precision of value must not be finer than %.2f", param.getTolerance()));
        }
        // If this passes, setParameter truncates excess trailing 0's
					   
        /* Send command to Chiller controller */
        setParameter(param.getNumber(), Double.toString(value));

       /**
        *  Check parameter value against requesting setting
        *  Repeat until within tolerances, or a maximum time is reached.
        */
        long start = System.currentTimeMillis();
        long dt = 0;
        while (dt < TIMEOUT_SET) {
            if (debug) System.out.println("In wait loop, dt = " + dt);
            try {
                Thread.sleep(intervalCheckSet.toMillis());
            } catch (InterruptedException ex) {
                throw new RuntimeException("Unexpected interrupt while waiting in Chiller.setParamCommand",ex);
            }
	    String str = readNamedParameter(param);
            String substr = str.substring(str.lastIndexOf(' ')+1);
            double val = Double.parseDouble(substr);
	    if (debug) System.out.println("param " + param.getNumber() + 
	        "  val read = " + val + "  set value = " + value);
	    dt = System.currentTimeMillis() - start;
            if (Math.abs(val - value) < param.getTolerance()) {
  	        if (debug) System.out.println("End of loop, dt = " + dt);
                return;
	    } 
        }
        throw new DriverTimeoutException("timeout, dt = " + dt + ", while waiting for parameter setting");
    }

   /**
    *  Go to a temperature using default ramp.
    *  Waits for setpoint value to match request and then returns.
    *
    *  @param  double temperature value (degrees C)
    *  @throws Driver Exception
    */
    public void setTemperature(double value) throws DriverException {
        String cmndString = "GT " + Double.toString(value);
        write(cmndString);
        checkSetPoint(value);
        return;
    }

   /**
    *  Go to a temperature using specified ramp.
    *  Waits for setpoint value to match request and then returns.
    *
    *  @param  double temperature value (degrees C)
    *  @param  double ramp (degrees C per minute)
    *  @throws Driver Exception
    */
    public void setTemperatureWithRamp(double value, double ramp) throws DriverException {
        String cmndString = "RR " + Double.toString(value) + " " +
	    Double.toString(ramp);
        write(cmndString);
        checkSetPoint(value);
        return;
    }

   /**
    *  Check setpoint value against requested temperature value.
    *  Repeat until within tolerance or until a maximum time.
    *
    *  @param  double requested temperature value
    *  @throws DriverException
    */
    private void checkSetPoint(double value) throws DriverException {
        long start = System.currentTimeMillis();
        long dt = 0;
        while (dt < TIMEOUT_SET) {
            try {
                Thread.sleep(intervalCheckSet.toMillis());
            } catch (InterruptedException ex) {
                throw new RuntimeException("Unexpected interrupt while waiting in Chiller.setParamCommand",ex);
            }
            double setpt = getTemperature(Query.SET_POINT);
            if (Math.abs(setpt - value) < SETPOINT_TOLERANCE) {
                return;
            }
	    dt = System.currentTimeMillis() - start;
        }
        throw new DriverTimeoutException("timeout while waiting for temperature setpoimt");
    }

   /**
    *  Stop controlling temperature
    *
    *  @throws DriverException
    */
    public void quitControl() throws DriverException {
        write("QU");
        return;
    }

   /**
    *  Set chilled fluid flow rate.
    *  Waits for setpoint value to match request and then returns.
    *
    *  @param  double flow-rate value (gallons per minute)
    *  @throws Driver Exception
    */
    public void setFlow(double value) throws DriverException {
        String cmndString = "PSP " + String.format("%.1f",value);
        write(cmndString);
        long start = System.currentTimeMillis();
        long dt = 0;
        while (dt < TIMEOUT_SET) {
            try {
                Thread.sleep(intervalCheckSet.toMillis());
            } catch (InterruptedException ex) {
                throw new RuntimeException("Unexpected interrupt while waiting in Chiller.setParamCommand",ex);
            }
            double setpt = getFlow(Query.FLOW_SETPT);
            if (Math.abs(setpt - value) < SETFLOW_TOLERANCE) {
                return;
            }
	    dt = System.currentTimeMillis() - start;
        }
        throw new DriverTimeoutException("timeout while waiting for flow setpoimt");
    }

   /**
    *  Send digital DUT temperature reading to Chiller
    *
    *  @param  double temperature in degrees-C
    *  @throws DriverException
    */
    public void writeDUT(double temperature) throws DriverException {
        String cmnd = "SDT " + String.format("%.1f",temperature);
        write(cmnd);
    }

   /**
    *  Clear chiller error and warning bits.
    *  These bits are latched; if the causing condition for a bit persists,
    *  the bit will get reasserted after clearing.
    *
    *  @throws DriverException
    */
    public void clearErrors() throws DriverException {
        write("CLE");
    }

   /**
    *  Clear error screen on chiller-controller gui
    *
    *  @throws DriverException
    */
    public void clearGuiErrorScreen() throws DriverException {
        write("CES");
    }

   /**
    *  Lock or unlock GUI
    *
    *  @param  boolean lock/unlock if true/false
    *  @throws DriverException
    */
    public void lockGui(boolean lock) throws DriverException {
        String cmnd = lock ? "FLO" : "FLF";
        write(cmnd);
    }

   /**
    *  Save all current chiller F# parameters to active non-volatile memory
    *
    *  @throws DriverException
    */
    public void saveParams() throws DriverException {
        write("UP");
    }        

   /**
    *  Restore all chiller F# parameters from active non-volatile memory
    *
    *  @throws DriverException
    */
    public void loadParams() throws DriverException {
        write("RUP");
    }        

   /**
    *  Directly send command string to chiller/
    *  For temporary use, to allow using non-implemented commands, allowed
    *  only in debug mode.  Do not use for commands expecting a response.
    *
    *  @param  chiller command String
    *  @param  args (if any)
    *  @throws DeiverExceptiom
    */
    public void sendDirectCommand(String command, String... args) 
     throws DriverException {
        if (debug) {
            boolean notQuery = true;
            for (int i = 0; i < qval.length; i++) { 
                String test = qval[i].getCommand();
                int idx = test.indexOf(' ');
                if (idx > 0) {
                    test = test.substring(0,idx);
                }
                if (command.equals(test)) {
                    notQuery = false;
                    break;
                }
            }
	    if (command.equals("QFA") || command.equals("TE")) {
                notQuery = false;
            }
            if (notQuery) {
                write(makeCommandString(command, args));
            } else {
                throw new DriverException("not allowed for query commands");
            }
        } else {
            throw new DriverException("allowed only in Debug mode");
	}
    }

   /**
    *  Makes a command string (borrowed from TestAscii)
    *
    *  @param  command  The command word
    *  @param  args     The command arguments
    *  @return The generated command string
    */
    protected static String makeCommandString(String command, String... args)
    {
        StringBuilder cmnd = new StringBuilder(command);
        for (String arg : args) {
            cmnd.append(' ').append(arg);
        }
        return cmnd.toString();
    }

}
