package org.lsst.ccs.drivers.chiller;

import java.io.OutputStream;
import java.time.Duration;
import org.lsst.ccs.drivers.ascii.Ascii;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;

/**
 * Driver for inTEST Chiller.
 */
public class Chiller extends Ascii {

    private static final int DEFAULT_PORT = 9760;
    private static final int TIMEOUT = 200;     // ms
    // Maximum index of Chiller setup ("F") parameters
    private static final int MAX_SETUP_PARAM = 77;  
    // Label identifying source of messages
    private final static String Lbl = "Chiller ";

    private final Duration intervalCheckSet = Duration.ofMillis(10);

    private int telnetPort = DEFAULT_PORT;
    private int timeout = TIMEOUT;
    private boolean debug = false;
    private boolean first = true;
    private Query[] qval;

   /**
    *  Enumeration of all "Immediate" Query/Read commands to Chiller,
    ^  except QFA and REA (both handled separately).
    */
    public enum Query {

        IDENTITY    ("*IDN?", "Controller Identity",             1, false),
        LOOP_COUNT  ("LCT",   "Loop Count (Program Mode)",       1, false),
        LAT_CMND    ("QC",    "Last Command String",             1, false),
        ERROR_BITS  ("QCE",   "Error1, Error2, Warning (hex)",   1, false),
        DIAG_INFO   ("QDI",   "Diagnostics Info",                2, false),
        // PARAMETERS  ("QFA",    "Setup Parameter",                1, false);
        LIFETIME    ("QLP",   "Life Time Parameters",            4, false),
        REMOTE_MODE ("QM",    "1 = Immediate, 2 = Program",      1, false),
        SERIAL_NO   ("QN",    "Controller Serial Number",        1, false),
        PGM_STATE   ("QPG",   "Remote Program Running State",    1, false), 
	HEAT_COOL   ("QPL",   "Heat and Cool Percents of Max",   1, false),
        TEMP_RANGE  ("QR",    "Controller Temperature Range",    1, false),
        SET_POINT   ("QS",    "Probe Number and Set Point",      1, true ),
        DYNAMIC_SET ("QSC",   "Probe No. and Control Set Point", 1, true ), 
        SYS_INFO    ("QSI",   "System Information",              7, false),
        FIRMWARE    ("QV",    "Firmware Version",                1, false),
        FLOW_RATE   ("RCF",   "Flow Rate (GPM or LPM)",          1, false),
        STATUS      ("RCS",   "Chiller Status (hex)",            1, false),
	// EVENT_BITS  ("REA",   "Event Bits (hex)",                1, false),
        INPUT_BITS1 ("RID 1", "5V Digital Inputs (binary)",      1, false),
        INPUT_BITS2 ("RID 2", "24V Digital Inputs (binary)",     1, false),
        PRESSURE_IN ("RIP",   "Input Pressure",                  1, false),
        OUTPUT_BITS1("ROD 1", "5V Digital Outputs (binary)",     1, false),
        OUTPUT_BITS2("ROD 2", "24V Digital Outputs (binary)",    1, false),
        PRESSURE_OUT("ROP",   "Output Pressure",                 1, false),
	PID_PARAM   ("RP",    "PID Constants (F0, F10 to F12)",  1, false),
	FLOW_SETPT  ("RPS",   "Pressure or Flow Setpoint",       1, false),
	STATE_BITS  ("RSA",   "Chiller State Bits (hex)",        1, false),
	GPIB_TIMEOUT("RTM",   "GPIB timeout (F44)",              1, false),
        TEMPERATURE ("TP 1",  "Coolant Output Temperature",      1, true ),
        T_CONDENSER ("TP 5",  "Temperature CondensorOut",        1, true ),
        T_TXV_BULB  ("TP 6",  "Temperture TXV_Bulb",             1, true ),
        T_STAGE2EVAP("TP 7",  "Temperature Stage2Evap",          1, true );

        private String command;     // Command to be sent to chiller
        private String description; // Description of commands
        private int expectedLines;  // expected number of lines in response
        private boolean isTemp;     // true if returns temperature as last field

        Query(String command, String description, int expectedLines,
              boolean isTemp) {            
            this.command = command;
            this.description = description;
            this.expectedLines = expectedLines;
            this.isTemp = isTemp;
        }

        public String getCommand() {return command;}
        public String getDescription() {return description;}
	public int getExpectedLines() {return expectedLines;}
	public boolean getIsTemp() {return isTemp;}

    }

   /**
    *  Enumeration of bits in Event Register
    */
    public enum EventRegister {
        BAD_ARG    (0x01, "Bad_Argument "),
        PGM_FULL   (0x02, "PgmBuffer_0.75Full "),
        BAD_CMND   (0x04, "UnrecognizedCmnd "),  
        DWELLTIME  (0x08, "DwellTime_Done "),	   
        SETPOINT   (0x10, "SetPoint_Reached "),  
        CMND_OVFL  (0x20, "CmndBufferOverrun  "),  
        UNUSED_6   (0x40, "Unused_Bit6 "),  
	UNUSED_7   (0x80, "Unused_Bit7 ");

        private int mask;
        private String descr;

        EventRegister(int mask, String descr) {
            this.mask = mask;
            this.descr = descr;
        }

        public static String decode(int register) {
            String decoded = "EventRegister:  ";
            EventRegister[] bits = EventRegister.values();
            int nbits = bits.length;
            for (int i = 0; i < nbits; i++) {
                if ((register & bits[i].mask) != 0) {
                    decoded += (bits[i].toString() + " ");
                }
            }
            return decoded;
        }
    }

   /**
    *  Enumeration of bits in Status Register
    */
    public enum StatusRegister {
        ERROR      (0x01, "Error(s) present "),
        WARNING    (0x02, "Warning(s) present "),
        PGM_FULL   (0x04, "Program Buffer over 0.75 full "),  
        T_CONTROL  (0x08, "RunState: Controlling T "),	   
        AT_SETPT   (0x10, "At Temperature SetPoint "),  
        SRQ_SET    (0x20, "SRQ Set "),  
        CMPRS_ON   (0x40, "Compressors on "),  
	UNUSED_7   (0x80, "Unused Bit7 ");

        private int mask;
        private String descr;

        StatusRegister(int mask, String descr) {
            this.mask = mask;
            this.descr = descr;
        }

        public int getMask() {return mask;}
        public String getDescr() {return descr;}

        public static String decode(int register) {
            String decoded = "StatusRegister = " + Integer.toHexString(register) + "x:";
            StatusRegister[] stat = StatusRegister.values();
            int nbits = stat.length;
            for (int i = 0; i < nbits; i++) {
                String st = ((register & stat[i].mask) != 0) ? "Yes" : "No ";
                decoded += ("   " + stat[i] + " " + st);
            }
            return decoded;
        }
    }

   /**
    *  Enumeration of quantities returned by LIFETIME query
    */
    public enum Life {
        CONTROL   ("Controller Hours"),
	COMPRESS  ("Compressor One Hours"),
	PUMP      ("Pump Hours"),
	VALVE     ("Valve Activation Count");

        private String descr;

        Life(String descr) {
            this.descr = descr;
        }

        public String getDescription() {return descr;}
    }

   /**
    *  Enumeration of chiller parameters to be checked after Set commands
    */
    public enum FParam {
        F00 ( 0, 1.0, 1.,     64. ,  "PID proportional coeff"),
	F02 ( 2, 1.0, 1.,     32. ,  "Cold valve period (s)"),
	F10 (10, 1.0, 0.,     3200,  "PID integral coeff"),
	F11 (11, 1.0, 0.,     3200,  "PID difeential coeff"),
	F12 (12, 1.0, -99.,   99. ,  "PID heater factor"),
	F27 (27, 0.1, -70.0,  30.0 , "DUT lower temp limit"), 
	F28 (28, 0.1, -70.0,  30.0 , "DUT upper temp limit"),
	F29 (29, 0.1, 1.0,    300.0, "DUT lower DeltaT limit"),
	F30 (30, 0.1, -300.0, 300.0, "DUT upper DeltaT limit"),
	F31 (31, 0.1, 0.1,    10.0,  "SetPoint tolerance (deg-C)"),
	F32 (32, 1.0, 0.,     59. ,  "Min SetPoint stable time (s)"),
        F34 (34, 1.0, 1.,     1000,  "Thermal mass (damping)"),
	F37 (37, 0.1, 0.0,    500.0, "Default ramp rate (deg/min)"),
	F40 (40, 1.0, 0.,     1.,    "T-control Normal (0) or DUT (1)"),
	F59 (50, 0.1, 0.1,    64.0,  "DUT proportional coeff"),
	F60 (60, 0.1, 0.0,    64.0,  "DUT integral coeff");

        private int number;      // parameter number
        private double tol;      // tolerance (least significant digit)
        private double min;      // minimum value
        private double max;      // maximum value
        private String descr;    // description of parameter

        FParam(int number, double tol, double min, double max, String descr) {
            this.number = number;
            this.tol = tol;
            this.min = min;
            this.max = max;
            this.descr = descr;
        }

        public int getNumber() {return number;}
        public double getTolerance() {return tol;}
        public double getMin() {return min;}
        public double getMax() {return max;}
        public String getDescription() {return descr;}

    }

   /**
    *  Enumeration of Chiller Commans which set parameters
    *
    *  Some commands set more than one parameter,
    *  The information included can be used to verify settings
    */
    public enum SetParam {
        COLD_PERIOD    ("CV",  1, new FParam[]{FParam.F02}),
	SETTLING_BAND  ("SB",  2, new FParam[]{FParam.F31,FParam.F32}),
	DUT_VS_RTD     ("SD",  2, new FParam[]{FParam.F29,FParam.F30}),
        DUT_T_LIMITS   ("SL",  2, new FParam[]{FParam.F27,FParam.F28}),
        RAMP_RATE      ("SR",  1, new FParam[]{FParam.F37}),
        THERMAL_MASS   ("TM",  1, new FParam[]{FParam.F34}),
        DUT_INT_COEFF  ("WDI", 1, new FParam[]{FParam.F60}),
        DUT_PROP_COEFF ("WDP", 1, new FParam[]{FParam.F59}),
        PID_PARAMS     ("WP",  4, new FParam[]{FParam.F00,FParam.F10,FParam.F11,FParam.F12});

        private String command;     // Command to send to Chiller
        private int nparam;         // Number of parameters
        private FParam[] params;    // Array of parameters

	SetParam(String command, int nparam, FParam[] params) {
            this.command = command;
            this.nparam = nparam;
            this.params = params;
        }

        public String getCommand() {return command;}
        public int getNparam() {return nparam;}
        public FParam[] getParams() {return params;}
        public String getDescription() {
            String descr = String.format("%-16s %d param:", this, nparam);
            for (int i = 0; i < nparam; i++) {
                descr += ("  " + params[i].getDescription());
                if (i != (nparam - 1)) {descr += ",";}
            }
            return descr;
        }
    }

   /**
    *  Subclass to hold error and warning words from Chiller Controller
    */
    public static class ErrorWords {
        private int err1, err2, warn1, warn2;
        public ErrorWords(int err1, int err2, int warn1, int warn2) {
            this.err1 = err1;
            this.err2 = err2;
            this.warn1 = warn1;
            this.warn2 = warn2;
        }

        public int getError1() {return err1;}
        public int getError2() {return err2;}
        public int getWarning1() {return warn1;}
        public int getWarning2() {return warn2;}

        @Override
	public String toString() {
            String err = 
                "Error1 = " + Integer.toHexString(err1) + "x " +
                "Error2 = " + Integer.toHexString(err2) + "x " +
                "Warning1 = " + Integer.toHexString(warn1) + "x " +
                "Warning2 = " + Integer.toHexString(warn2) + "x ";
            return err;
        }

        @Override
	public boolean equals(Object y) {
            ErrorWords x = (ErrorWords) y;
            boolean eq = (err1 == x.getError1() && err2 == x.getError2() &&
			  warn1 == x.getWarning1() && warn2 == x.getWarning2());
	    return eq;
        }
    }

   /**
    *  Constructor.
    */
    public Chiller() {
        qval = Query.values();
    }

    /**
     * Open a connection to the chiller
     *
     * @param  host  The host to connect to
     * @param  port  Telnet port
     * @throws DriverException 
     */
    public void open(String host, int port) throws DriverException {
        super.open(ConnType.NET, host, port);
        setTimeout(timeout);
        setTerminator(Terminator.CRLF);

        /**
         *   Initialize some controller settings
         */

        try {
            // if (first) {write("DC");}    // Soft reset of controller
            // Temporarily skip this reset.
            write("SI");    // Set immediate mode
            write("DP2");   // Two decimal places for temperature readings
            write("DS");    // Disable SRQ (not used)

            //Fixed parameters

            setParameter(16, "0");       // Temperature units Celsius
            setParameter(39, "0");       // Flow units GPM
            setParameter(47, "0");       // Pressure units PSI

            first = false;
        }
        catch (DriverException e) {
            closeSilent();
            throw e;
        }
    }
    
    /**
     * Open a connection to the chiller using default port
     *
     * @param  host  The host to connect to
     * @throws DriverException 
     */
    public void open(String host) throws DriverException {
        open(host, DEFAULT_PORT);
    }

   /** 
   *  Close connection.  (Uses parent method.)
    *
    * @throws DriverException
    */

   /**
    *  Set debug mode true or false
    *
    *  @param  on_off
    */
    public void setDebug(boolean on_off) {
        debug = on_off;
    }

   /**
    *  List enumerared queries
    *
    *  @return  Table of queries abd descriptions
    */
    public String listQueries() {
        String table = "List of all queries for Chiller \n";
        for (int i = 0; i < qval.length; i++) {
            table += String.format("\n   %-15s  (%-5s)     %s", qval[i],
                                   qval[i].getCommand(),
                                   qval[i].getDescription());
       }
       table += "\n";
       return table;
    }

   /**
    *  List Status Register bits
    *
    *  @return  Table of status bits and descriptions
    */
    public String listStatusBits() {
        StatusRegister[] stat = StatusRegister.values();
        String table = "List of status bits \n";
        for (int i = 0; i < stat.length; i++) {
            table += String.format("\n   %-10s   %s", stat[i], 
                                   stat[i].getDescr());
        }
        return table;
    }

   /**
    ^  Send an enumerated query to Chiller and receive response
    *
    *  @param  query   enumerated command identifier
    *  @return value   value of quantity requested
    *  @throws DriverException
    */
    public String queryChiller(Query query) throws DriverException {
        String reply = read(query.getCommand());
        int expect = query.getExpectedLines();
        if (expect > 1) {
            for (int nlines = 1; nlines < expect; nlines++) {
		reply += ("\n" + read());
	    }
        }
        return reply;
    }

   /**
    ^  Send event-register query (REA) to Chiller and receive response
    *  Treated separately from generic query, because reading clears value.
    *
    *  @return reply   value of event register response
    *  @throws DriverException
    */
    private String queryEvtReg() throws DriverException {
        String reply = read("REA");
        return reply;
    }

   /**
    *  Read a chiller setup parameter
    *
    *  @param  index
    *  @return value (as a String)
    *  @throws DriverExceptiom
    */
    public String readParameter(int index) throws DriverException {
        if (index >= 0 && index <= MAX_SETUP_PARAM) {
            String command = "QFA " + index;
            String reply = read(command);
            String[] lines = reply.split("/n");
            if (lines.length != 1) {
                throw new DriverException("Expected 1 line in reply, received "
                                          + lines.length);
            }
            return reply;
        }
        throw new IllegalArgumentException("Index must be in range 0 to " + 
                                           MAX_SETUP_PARAM);
    }

   /**
    *  Set a chiller setup parameter.
    *  For use internally and in testing only. 
    *  (It lacks validation of value and checking of result.)
    *
    *  @param  index
    *  @param  value (as a String)
    *  @throws DriverExceptiom
    */
    void setParameter(int index, String value) throws DriverException {
        if (index >= 0 && index <= MAX_SETUP_PARAM) {
            String command = "SPA " + index + " " + value;
            write(command);
            return;
        }
        throw new IllegalArgumentException("Index must be in range 0 to " +
                                           MAX_SETUP_PARAM);
    }

   /**
    *  Set and read back a chiller setup parameter
    *  For use in testing only.
    *
    *  @param  index
    *  @param  value (as a String)
    *  @throws DriverExceptiom
    */
    String setAndReadParameter(int index, String value) 
     throws DriverException {
        setParameter(index, value);
        return readParameter(index);
    }

   /**
    *  Implement parameter-setting command (enumerated in SetParam).
    *  Checks arguments and waits for set values (or times out);
    *
    *  @param  SetParam (specifies command to Chiller controller)
    *  @param  Arguments to send to Chiller
    *  @throws DriverException
    */
    @SuppressWarnings("SleepWhileInLoop")
    public void setParamCommand(SetParam setParam, double... values) 
        throws DriverException {

        /* Check number and values of parameters */
        if (values.length != setParam.getNparam()) {
            throw new IllegalArgumentException(setParam + " requires " 
            + Integer.toString(setParam.getNparam()) + " parameter(s)");
        }
        for (int i = 0; i < values.length; i++) {
            FParam fParam = setParam.getParams()[i];
            if ((values[i] < fParam.getMin())||(values[i] > fParam.getMax())) {
                throw new IllegalArgumentException(String.format("Parameter %d must be between %f and %f", i, fParam.getMin(), fParam.getMax()));
            }
        }
					   
        /* Send command to Chiller controller */
        String command = setParam.getCommand();
        for (int i = 0; i < values.length; i++) {
            command += (" " + Double.toString(values[i]));
        }
        write(command);

       /**
        *  Check parameter values against requesting settings.
        *  Repeat until all within tolerances, or until a maximum time.
        */
        long start = System.currentTimeMillis();
        long dt = 0;
        int istart = 0;
        while (dt < 5*timeout) {
            if (debug) System.out.println("In wait loop, dt = " + dt);
            try {
                Thread.sleep(intervalCheckSet.toMillis());
            } catch (InterruptedException ex) {
                throw new RuntimeException("Unexpected interrupt while waiting in Chiller.setParamCommand",ex);
            }
            for (int i = istart; i < values.length; i++) {
                String str = readParameter(setParam.getParams()[i].getNumber());
                String substr = str.substring(str.lastIndexOf(' ')+1);
                double val = Double.parseDouble(substr);
                if (debug) System.out.println("param " + i + "  val read = " + 
                 val + "  set value = " + values[i]);
                if (Math.abs(val - values[i]) >= 
                    setParam.getParams()[i].getTolerance()) {
                    break;
                } else if (i == values.length - 1) {
                    return;
                } else {
                    istart++;    // If parameter i okay, skip on next loop
                }
            }
	    dt = System.currentTimeMillis() - start;
	    if (debug) System.out.println("End of loop, dt = " + dt);
        }
        throw new DriverTimeoutException("timeout while waiting for parameter settings");
    }

   /**
    *  Return any temperature (reading or set point)n as a double
    *
    *  @param  Query enumeraion of desired temperature reading
    *  @return temperature
    *  @throws DriverException
    */
    public double getTemperature(Query query) throws DriverException {
        if (!query.getIsTemp()) {
            throw new IllegalArgumentException("Quantity is not a temperature");
        }
        String response = queryChiller(query);
	String substr = response.substring(response.lastIndexOf(' ')+1);
        double value = Double.parseDouble(substr);
        return value;
    }

   /**
    *  Return flow rate (reading or setpoint)
    *
    *  @param  Query enumeraion of desired flow reading
    *  @return flow
    *  @throws DriverException
    */
    public double getFlow(Query query) throws DriverException {
        if (query != Query.FLOW_RATE && query != Query.FLOW_SETPT) {
            throw new IllegalArgumentException("Arg must be a flow quantity");
        }
        String response = queryChiller(query);
	String substr = response.substring(response.lastIndexOf(' ')+1);
        double value = Double.parseDouble(substr);
        return value;
    }

   /**
    *  Return a pressure reading
    *
    *  @param  Query enumeraion of desired pressure reading
    *  @return pressure
    *  @throws DriverException
    */
    public double getPressure(Query query) throws DriverException {
        if (query != Query.PRESSURE_IN && query != Query.PRESSURE_OUT) {
            throw new IllegalArgumentException("Arg must be pressure quantity");
        }
        String response = queryChiller(query);
	String substr = response.substring(response.lastIndexOf(' ')+1);
        double value = Double.parseDouble(substr);
        return value;
    }

   /**
    *  Return heating and cooling as percentages (of maximum capacity?).
    *  (Heat is duty cycle of heater; Cool is duty cycle of cold valve.)
    *  No more than one of these should be non-zero at any one time.
    *
    *  @return  percentages[heat,cool]
    *  @throws DriverException
    */
    public double[] getHeatCool() throws DriverException {
        String[] resp = queryChiller(Query.HEAT_COOL).split("%");
        double[] percentages = new double[2];
        for (int i = 0; i < 2; i++) {
            String substr = resp[i].substring(resp[i].lastIndexOf("=") + 1);
            percentages[i] = Double.parseDouble(substr);
        }
        return percentages;
    }

   /**
    *  Return a Lifetime quantity
    *
    *  @param  Enumerated Life
    *  @return value (int)
    *  @throws DriverException
    */
    public int getLifetime(Life life) throws DriverException {
        String[] lifetime = queryChiller(Query.LIFETIME).split("\n");
        String line = lifetime[life.ordinal()];
        String substr = line.substring(line.lastIndexOf(' ')+1);
        int value = Integer.parseInt(substr);
        return value;
    }
        
   /**
    *  Decode event register
    *
    *  @return String showing which bits are set
    *  @throws DriverException
    */
    public String decodeEvtReg() throws DriverException {
        String resp = queryEvtReg();
        String substr = resp.substring(resp.length()-2, resp.length());
        int register = Integer.parseUnsignedInt(substr, 16);
        return resp + "  " + EventRegister.decode(register);
    }

   /**
    *  Get status register
    *
    *  @return  Status register as int value
    *  @throws DriverException
    */
    public int getStatusReg() throws DriverException {
        String resp = queryChiller(Query.STATE_BITS);
        String substr = resp.substring(resp.length()-2, resp.length());
        return Integer.parseUnsignedInt(substr, 16);
    }

   /**
    *  Decode status register
    *
    *  @return String showing which bits are set
    *  @throws DriverException
    */
    public String decodeStatusReg() throws DriverException {
        return StatusRegister.decode(getStatusReg());
    }

   /**
    *  Extract and return error and warning words
    *
    *  @throws DriverException
    */
    public ErrorWords getErrorWords() throws DriverException  {
        String errors = queryChiller(Query.ERROR_BITS);
        int idx1 = errors.indexOf(' ');
        int idx2 = errors.indexOf(' ', idx1+1);
        int idx3 = errors.indexOf(' ', idx2+1);
        int err1 = Integer.parseInt(errors.substring(idx1+1,idx2));
        int err2 = Integer.parseInt(errors.substring(idx2+1,idx3));
        int warn1 = Integer.parseInt(errors.substring(idx3+1));
        int warn2 = 0;    // Until we can read this from Chiller
        return new ErrorWords(err1, err2, warn1, warn2);
    }

   /**
    *  Directly send command string to chiller/
    *  For temporary use, to allow using non-implemented commands, allowed
    *  only in debug mode.  Do not use for commands expecting a response.
    *
    *  @param  chiller command String
    *  @param  args (if any)
    *  @throws DeiverExceptiom
    */
    public void sendDirectCommand(String command, String... args) 
     throws DriverException {
        if (debug) {
            boolean notQuery = true;
            for (int i = 0; i < qval.length; i++) { 
                String test = qval[i].getCommand();
                int idx = test.indexOf(' ');
                if (idx > 0) {
                    test = test.substring(0,idx);
                }
                if (command.equals(test)) {
                    notQuery = false;
                    break;
                }
            }
	    if (command.equals("QFA") || command.equals("TE")) {
                notQuery = false;
            }
            if (notQuery) {
                write(makeCommandString(command, args));
            } else {
                throw new DriverException("not allowed for query commands");
            }
        } else {
            throw new DriverException("allowed only in Debug mode");
	}
    }


   /**
    *  Makes a command string (borrowed from TestAscii)
    *
    *  @param  command  The command word
    *  @param  args     The command arguments
    *  @return The generated command string
    */
    protected static String makeCommandString(String command, String... args)
    {
        StringBuilder cmnd = new StringBuilder(command);
        for (String arg : args) {
            cmnd.append(' ').append(arg);
        }
        return cmnd.toString();
    }

}
