package org.lsst.ccs.drivers.chiller;

import org.lsst.ccs.drivers.ascii.Ascii;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;

/**
 * Driver for inTEST Chiller.
 */
public class Chiller extends Ascii {

    private static final int DEFAULT_PORT = 9760;
    private static final int TIMEOUT = 100;     // ms
    // Maximum index of Chiller setup ("F") parameters
    private static final int MAX_SETUP_PARAM = 71;  
    // Label identifying source of messages
    private final static String Lbl = "Chiller ";

    private int telnetPort = DEFAULT_PORT;
    private int timeout = TIMEOUT;
    private boolean debug = false;
    private Query[] qval;
    private int error1Word, error2Word, warningWord;

   /**
    *  Enumeration of all "Immediate" Query/Read commands to Chiller,
    ^  except QFA and (until it is nderstood) RP.
    */
    public enum Query {

        IDENTITY    ("*IDN?", "Controller Identity",             1, false),
        LOOP_COUNT  ("LCT",   "Loop Count (Program Mode)",       1, false),
        LAT_CMND    ("QC",    "Last Command String",             1, false),
        ERROR_BITS  ("QCE",   "Error1, Error2, Warning (hex)",   1, false),
        DIAG_INFO   ("QDI",   "Diagnostics Info",                2, false),
        // PARAMETERS  ("QFA",    "Setup Parameter",                1, false);
        LIFETIME    ("QLP",   "Life Time Parameters",            4, false),
        REMOTE_MODE ("QM",    "1 = Immediate, 2 = Program",      1, false),
        SERIAL_NO   ("QN",    "Controller Serial Number",        1, false),
        PGM_STATE   ("QPG",   "Remote Program Running State",    1, false), 
        TEMP_RANGE  ("QR",    "Controller Temperature Range",    1, false),
        SET_POINT   ("QS",    "Probe Number and Set Point",      1, true ),
        DYNAMIC_SET ("QSC",   "Probe No. and Control Set Point", 1, true ), 
        SYS_INFO    ("QSI",   "System Information",              7, false),
        FIRMWARE    ("QV",    "Firmware Version",                1, false),
        FLOW_RATE   ("RCF",   "Flow Rate (GPM or LPM)",          1, false),
        STATUS      ("RCS",   "Chiller Status (hex)",            1, false),
        EVENT_BITS  ("REA",   "Event Bits (hex)",                1, false),
        INPUT_BITS1 ("RID 1", "5V Digital Inputs (binary)",      1, false),
        INPUT_BITS2 ("RID 2", "24V Digital Inputs (binary)",     1, false),
        OUTPUT_BITS1("ROD 1", "5V Digital Outputs (binary)",     1, false),
        OUTPUT_BITS2("ROD 2", "24V Digital Outputs (binary)",    1, false),
        PRESSURE    ("ROP",   "Output Pressure",                 1, false),
        // PID_PARAM   ("RP",    "PID Constants (F1, F10, F11)",    1, false);
	FLOW_SETPT  ("RPS",   "Pressure or Flow Setpoint",       1, false),
	STATE_BITS  ("RSA",   "Chiller State Bits (hex)",        1, false),
	GPIB_TIMEOUT("RTM",   "GPIB timeout (F44)",              1, false),
        TEMPERATURE ("TP 1",  "Coolant Output Temperature",      1, true ),
        T_CONDENSER ("TP 5",  "Temperature CondensorOut",        1, true ),
        T_TXV_BULB  ("TP 6",  "Temperture TXV_Bulb",             1, true ),
        T_STAGE2EVAP("TP 7",  "Temperature Stage2Evap",          1, true );

        private String command;     // Command to be sent to chiller
        private String description; // Description of commands
        private int expectedLines;  // expected number of lines in response
        private boolean isTemp;     // true if returns temperature as last field

        Query(String command, String description, int expectedLines,
              boolean isTemp) {            
            this.command = command;
            this.description = description;
            this.expectedLines = expectedLines;
            this.isTemp = isTemp;
        }

        public String getCommand() {return command;}
        public String getDescription() {return description;}
	public int getExpectedLines() {return expectedLines;}
	public boolean getIsTemp() {return isTemp;}

    }

   /**
    *  Enumeration of bits in Event Register
    */
    public enum EventRegister {
        BAD_ARG    (0x01, "Bad_Argument "),
        PGM_FULL   (0x02, "PgmBuffer_0.75Full "),
        BAD_CMND   (0x04, "UnrecognizedCmnd "),  
        DWELLTIME  (0x08, "DwellTime_Done "),	   
        SETPOINT   (0x10, "SetPoint_Reached "),  
        CMND_OVFL  (0x20, "CmndBufferOverrun  "),  
        UNUSED_6   (0x40, "Unused_Bit6 "),  
	UNUSED_7   (0x80, "Unused_Bit7 ");

        private int mask;
        private String descr;

        EventRegister(int mask, String descr) {
            this.mask = mask;
            this.descr = descr;
        }

        public static String decode(int register) {
            String decoded = "EventRegister:  ";
            EventRegister[] bits = EventRegister.values();
            int nbits = bits.length;
            for (int i = 0; i < nbits; i++) {
                if ((register & bits[i].mask) != 0) {
                    decoded += (bits[i].toString() + " ");
                }
            }
            return decoded;
        }
    }

   /**
    *  Enumeration of bits in Status Register
    */
    public enum StatusRegister {
        ERROR      (0x01, "ErrorPresent "),
        WARNING    (0x02, "WarningPresent "),
        PGM_FULL   (0x04, "PgmBuff_0.75Full "),  
        RUN_STATE  (0x08, "RunningState "),	   
        SETPOINT   (0x10, "At_SetPoint "),  
        SRQ_SET    (0x20, "SRQ_Set "),  
        UNUSED_6   (0x40, "Unused_Bit6 "),  
	UNUSED_7   (0x80, "Unused_Bit7 ");

        private int mask;
        private String descr;

        StatusRegister(int mask, String descr) {
            this.mask = mask;
            this.descr = descr;
        }

        public static String decode(int register) {
            String decoded = "StatusRegister:  ";
            StatusRegister[] bits = StatusRegister.values();
            int nbits = bits.length;
            for (int i = 0; i < nbits; i++) {
                if ((register & bits[i].mask) != 0) {
                    decoded += (bits[i].toString() + " ");
                }
            }
            return decoded;
        }
    }

   /**
    *  Enumeration of quantities returned by LIFETIME query
    */
    public enum Life {
        CONTROL   ("Controller Hours"),
	COMPRESS  ("Compressor One Hours"),
	PUMP      ("Pump Hours"),
	VALVE     ("Valve Activation Count");

        private String descr;

        Life(String descr) {
            this.descr = descr;
        }

        public String getDescription() {return descr;}
    }

   /**
    *  Constructor.
    */
    public Chiller() {
        qval = Query.values();
    }

    /**
     * Open a connection to the chiller
     *
     * @param  host  The host to connect to
     * @param  port  Telnet port
     * @throws DriverException 
     */
    public void open(String host, int port) throws DriverException {
        super.open(ConnType.NET, host, port);
        setTimeout(timeout);
        setTerminator(Terminator.CRLF);

        /**
         *   Initialize some controller settings
         */

        try {
            write("DC");    // Soft reset of controller
            write("SI");    // Set immediate mode
            write("DP2");   // Two decimal places for temperature readings

            //Fixed parameters, not subject to outside change
        }
        catch (DriverException e) {
            closeSilent();
            throw e;
        }
    }
    
    /**
     * Open a connection to the chiller using default port
     *
     * @param  host  The host to connect to
     * @throws DriverException 
     */
    public void open(String host) throws DriverException {
        open(host, DEFAULT_PORT);
    }

   /** 
   *  Close connection.  (Uses parent method.)
    *
    * @throws DriverException
    */

   /**
    *  Set debug mode true or false
    *
    *  @param  on_off
    */
    public void setDebug(boolean on_off) {
        debug = on_off;
    }

   /**
    *  List enumerared queries
    *
    *  @return  Table of queries abd descriptions
    */
    public String listQueries() {
        String table = "List of all queries for Chiller \n";
        for (int i = 0; i < qval.length; i++) {
            table += String.format("\n   %-15s  (%-5s)     %s", qval[i],
                                   qval[i].getCommand(),
                                   qval[i].getDescription());
       }
       table += "\n";
       return table;
    }

   /**
    ^  Send query to Chiller and receive response
    *
    *  @param  query   enumerated command identifier
    *  @return value   value of quantity requested
    *  @throws DriverException
    */
    public String queryChiller(Query query) throws DriverException {
        String reply = read(query.getCommand());
        int expect = query.getExpectedLines();
        if (expect > 1) {
            for (int nlines = 1; nlines < expect; nlines++) {
		reply += ("\n" + read());
	    }
        }
        return reply;
    }

   /**
    *  Read a chiller setup parameter
    *
    *  @param  index
    *  @return value (as a String)
    *  @throws DriverExceptiom
    */
    public String readParameter(int index) throws DriverException {
        if (index >= 0 && index <= MAX_SETUP_PARAM) {
            String command = "QFA " + index;
            String reply = read(command);
            String[] lines = reply.split("/n");
            if (lines.length != 1) {
                throw new DriverException("Expected 1 line in reply, received "
                                          + lines.length);
            }
            return reply;
        }
        throw new IllegalArgumentException("Index must be in range 0 to " + 
                                           MAX_SETUP_PARAM);
    }

   /**
    *  Set a chiller setup parameter
    *
    *  @param  index
    *  @param  value (as a String)
    *  @throws DriverExceptiom
    */
    public void setParameter(int index, String value) throws DriverException {
        if (index >= 0 && index <= MAX_SETUP_PARAM) {
            String command = "SPA " + index + " " + value;
            write(command);
            return;
        }
        throw new IllegalArgumentException("Index must be in range 0 to " +
                                           MAX_SETUP_PARAM);
    }

   /**
    *  Set and read back a chiller setup parameter
    *
    *  @param  index
    *  @param  value (as a String)
    *  @throws DriverExceptiom
    */
    public String setAndReadParameter(int index, String value) 
     throws DriverException {
        setParameter(index, value);
        return readParameter(index);
    }

   /**
    *  Return any temperature (reading or set point)n as a double
    *
    *  @param  Query enumeraion of desired temperature reading
    *  @return temperature
    *  @throws DriverException
    */
    public double getTemperature(Query query) throws DriverException {
        if (!query.getIsTemp()) {
            throw new IllegalArgumentException("Quantity is not a temperature");
        }
        String response = queryChiller(query);
	String substr = response.substring(response.lastIndexOf(' ')+1);
        double value = Double.parseDouble(substr);
        return value;
    }

   /**
    *  Return flow rate (reading or setpoint)
    *
    *  @param  Query enumeraion of desired flow reading
    *  @return flow
    *  @throws DriverException
    */
    public int getFlow(Query query) throws DriverException {
        if (query != Query.FLOW_RATE && query != Query.FLOW_SETPT) {
            throw new IllegalArgumentException("Arg must be a flow quantity");
        }
        String response = queryChiller(query);
	String substr = response.substring(response.lastIndexOf(' ')+1);
        int value = Integer.parseInt(substr);
        return value;
    }

   /**
    *  Return a Lifetime quantity
    *
    *  @param  Enumerated Life
    *  @return value (int)
    *  @throws DriverException
    */
    public int getLifetime(Life life) throws DriverException {
        String[] lifetime = queryChiller(Query.LIFETIME).split("\n");
        String line = lifetime[life.ordinal()];
        String substr = line.substring(line.lastIndexOf(' ')+1);
        int value = Integer.parseInt(substr);
        return value;
    }
        
   /**
    *  Decode event register
    *
    *  @return String showing which bits are set
    *  @throws DriverException
    */
    public String decodeEvtReg() throws DriverException {
        String resp = queryChiller(Query.EVENT_BITS);
        String substr = resp.substring(resp.length()-2, resp.length());
        int register = Integer.parseUnsignedInt(substr, 16);
        return resp + "  " + EventRegister.decode(register);
    }

   /**
    *  Decode status register
    *
    *  @return String showing which bits are set
    *  @throws DriverException
    */
    public String decodeStatusReg() throws DriverException {
        String resp = queryChiller(Query.STATE_BITS);
        String substr = resp.substring(resp.length()-2, resp.length());
        int register = Integer.parseUnsignedInt(substr, 16);
        return resp + "  " + StatusRegister.decode(register);
    }

   /**
    *  Extract error words (error1, error2, warning)
    *
    *  @throws DriverException
    */
    private void extractErrorWords() throws DriverException  {
        String errors = queryChiller(Query.ERROR_BITS);
        int idx1 = errors.indexOf(' ');
        int idx2 = errors.indexOf(' ', idx1+1);
        int idx3 = errors.indexOf(' ', idx2+1);
        error1Word = Integer.parseInt(errors.substring(idx1+1,idx2));
        error2Word = Integer.parseInt(errors.substring(idx2+1,idx3));
        warningWord = Integer.parseInt(errors.substring(idx3+1));
    }

   /**
    *  Show error words in hex Ascii
    *
    *  @return String
    *  @throws DriverException
    */
    public String showErrors() throws DriverException {
        extractErrorWords();
        String err = "Error1 (hex) = " + Integer.toHexString(error1Word) +
            ", Error2 (hex) = " + Integer.toHexString(error2Word) +
	    ", Warning (hex) = " + Integer.toHexString(warningWord);
        return err;
    }

   /**
    *  Directly send command string to chiller/
    *  For temporary use, to allow using non-implemented commands, allowed
    *  only in debuf mode.  Do not use for commands expecting a response.
    *
    *  @param  chiller command String
    *  @param  args (if any)
    *  @throws DeiverExceptiom
    */
    public void sendDirectCommand(String command, String... args) 
     throws DriverException {
        if (debug) {
            boolean notQuery = true;
            for (int i = 0; i < qval.length; i++) { 
                String test = qval[i].getCommand();
                int idx = test.indexOf(' ');
                if (idx > 0) {
                    test = test.substring(0,idx);
                }
                if (command.equals(test)) {
                    notQuery = false;
                    break;
                }
            }
	    if (command.equals("QFA") || command.equals("TE")) {
                notQuery = false;
            }
            if (notQuery) {
                write(makeCommandString(command, args));
            } else {
                throw new DriverException("not allowed for query commands");
            }
        } else {
            throw new DriverException("allowed only in Debug mode");
	}
    }


   /**
    *  Makes a command string (borrowed from TestAscii)
    *
    *  @param  command  The command word
    *  @param  args     The command arguments
    *  @return The generated command string
    */
    protected static String makeCommandString(String command, String... args)
    {
        StringBuilder cmnd = new StringBuilder(command);
        for (String arg : args) {
            cmnd.append(' ').append(arg);
        }
        return cmnd.toString();
    }

}
