package org.lsst.ccs.drivers.ccob;

import java.io.IOException;
import java.io.PrintStream;
import org.kohsuke.args4j.CmdLineException;
import org.kohsuke.args4j.CmdLineParser;
import org.kohsuke.args4j.Option;
import org.lsst.ccs.command.CommandSetBuilder;
import org.lsst.ccs.command.CompositeCommandSet;
import org.lsst.ccs.drivers.ccob.jni.CcobJNI;
import org.lsst.ccs.shell.JLineShell;

/**
 * Program to test the CCOB drivers
 * 
 * @author emarin & saxton
 */
public class TestCcob {
    
    /**
     *  Inner class for holding command line options.
     */
    private static class Options {

        @Option(name="-c", metaVar="<connType>", usage="Connection type: usb (default), socket, jni or sim")
        private ConnType type = ConnType.USB;

        @Option(name="-i", metaVar="<index>", usage="USB device index")
        private int index = 0;

        @Option(name="-n", metaVar="<node>", usage="Node name")
        private String node = "localhost";

        @Option(name="-s", metaVar="<socket>", usage="Socket number")
        private int port = 1557;

        @Option(name="-help", usage="Displays this help")
        private boolean showHelp;

    }

    public enum ConnType {USB, SOCKET, JNI, SIM}
    private final static PrintStream out = System.out;


    /**
     *  Main program.
     *
     *  @param  args  Command-line arguments
     *  @throws IOException
     */
    public static void main(String[] args) throws IOException {
        Options optns = new Options();
        CmdLineParser parser = new CmdLineParser(optns);
        try {
            parser.parseArgument(args);
        }
        catch (CmdLineException e) {
            if (!optns.showHelp) {
                out.println(e.getMessage());
            }
            showHelp(parser);
            System.exit(0);
        }
        if (optns.showHelp) {
            showHelp(parser);
            System.exit(0);
        }
        CCOBInterface ccob = null;
        switch (optns.type) {
        case USB:
            ccob = new CCOBUsb(optns.index); break;
        case SOCKET:
            ccob = new CCOBSocketDriver(optns.node, optns.port); break;
        case JNI:
            ccob = new CcobJNI(); break;
        case SIM:
            ccob = new CCOBSim(); break;
        }
        CompositeCommandSet compositeSet = new CompositeCommandSet();
        compositeSet.add(new CommandSetBuilder().buildCommandSet(new CCOBCommands(ccob)));
        JLineShell shell = new JLineShell(compositeSet, ">> ");
        shell.run();
        System.exit(0);    
    }


    /**
     *  Displays help.
     */
    private static void showHelp(CmdLineParser parser)
    {
        out.println("Usage: TestCcob [options...]");
        out.println("Options:");
        parser.printUsage(out);
    }
    
}
