package org.lsst.ccs.drivers.ccob;

import java.io.Closeable;
import java.io.IOException;
import java.net.Socket;
import java.nio.charset.Charset;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 * An implementation of the CCOB driver which uses sockets to communicate with a
 * CCOBServer
 *
 * @author CCS
 */
public class CCOBSocketDriver implements CCOBInterface, Closeable {
    private final static Charset DEFAULT_CHARSET = Charset.forName("US-ASCII");
    private Socket socket;
    private final String host;
    private final int port;

    public CCOBSocketDriver(String host, int port) {
        this.host = host;
        this.port = port;
    }

    @Override
    public void close() throws IOException {
        socket.close();
    }
    
    @Override
    public void init() throws DriverException {
        try {
            socket = new Socket(host,port);
        } catch (IOException ex) {
            throw new DriverException(ex);
        }
        write("Init");
        readAndCheckOK();
    }

    @Override
    public void stop() throws DriverException {
        write("Stop");
        readAndCheckOK();
        try {
            close();
        } catch (IOException ex) {
            throw new DriverException(ex);
        }
    }

    @Override
    public String version() throws DriverException {
        write("Version");
        return read();
    }

    @Override
    public String date() throws DriverException {
        write("Date");
        return read();
    }

    @Override
    public String misc() throws DriverException {
        write("Misc");
        return read();
    }

    @Override
    public String manu() throws DriverException {
        write("Manu");
        return read();
    }

    @Override
    public String device() throws DriverException {
        write("Device");
        return read();
    }

    @Override
    public void selectLed(LED led) throws DriverException {
        write("SelectLed "+led.ordinal());
        readAndCheckOK();
    }

    @Override
    public void setLedCurrent(double current) throws DriverException {
        write("SetLedCurrent "+current);
        readAndCheckOK();
    }

    @Override
    public void setExposureTime(double time) throws DriverException {
        write("SetExposureTime "+time);
        readAndCheckOK();
    }

    @Override
    public void pulse() throws DriverException {
        write("PulseLed");
        readAndCheckOK();
    }

    @Override
    public void shutter() throws DriverException {
        write("UseShutter");
        readAndCheckOK();
    }

    @Override
    public void startExposure() throws DriverException {
        write("StartExposure");
        readAndCheckOK();
    }

    @Override
    public boolean pollEnd() throws DriverException {
        write("PollEnd");
        String response = read();
        if (response.equals("1")) return true;
        if (response.equals("0")) return false;
        throw new DriverException("Unexpected response from server: " + response);
    }


    
    @Override
    public CCOBDataRead getAdcValues() throws DriverException {
        write("GetAdcValues");
	String response = read();	
	CCOBDataRead myData = new CCOBDataRead(response);         
	return myData;  	

    }
    
    @Override
    public double getAdcPhotoDiode() throws DriverException {
        write("GetAdcPhotoDiode");
	String response = read();	
        return Double.parseDouble(response);
    }
    
    private void write(String command) throws DriverException {
        try {
            socket.getOutputStream().write(command.getBytes(DEFAULT_CHARSET));
        } catch (IOException ex) {
            throw new DriverException("Error writing to socket", ex);
        }
    }

    private String read() throws DriverException {
        try {
            // DANGER: Assumes full response will be read into buffer, and that
            // response will not be > 256 characters
            byte[] buffer = new byte[256];
            socket.getInputStream().read(buffer);
            return new String(buffer, DEFAULT_CHARSET);
        } catch (IOException ex) {
            throw new DriverException("Error reading from socket", ex);
        }
    }

    private void readAndCheckOK() throws DriverException {
        String response = read();
        if (!response.startsWith("OK")) {
            throw new DriverException("Unexpected response from server: " + response);
        }
    }

    @Override
    public void ledOn(int Led, int On) throws DriverException {
        write("LedOn "+Led+" "+(On > 0 ?"1":"0"));
        readAndCheckOK();
    }

    @Override
    public boolean readButton(int Button) throws DriverException {
        write("ReadButton "+Button);
        String response = read();
        if (response.equals("1")) return true;
        if (response.equals("0")) return false;
        throw new DriverException("Unexpected response from server: " + response);
    }
}
