package org.lsst.ccs.drivers.ccob;

import org.lsst.ccs.drivers.commons.DriverException;

/**
 * A simulation implementation of the CCOB driver
 *
 * @author CCS
 */
public class CCOBSim implements CCOBInterface {

    private double ledCurr;


    /**
     *  Initializes the connection to a device
     *
     *  @throws  DriverException
     */
    @Override
    public void init() throws DriverException {
    }


    /**
     *  Stops the CCOB and closes the connection
     *
     *  @throws  DriverException
     */
    @Override
    public void stop() throws DriverException {
    }


    /**
     *  Gets the miscellaneous string
     *
     *  @return  The string
     *  @throws  DriverException
     */
    @Override
    public String misc() throws DriverException {
        return "XXXXXXXX";
    }


    /**
     *  Gets the manufacturer string
     *
     *  @return  The string
     *  @throws  DriverException
     */
    @Override
    public String manu() throws DriverException {
        return "CCS";
    }


    /**
     *  Gets the date string
     *
     *  @return  The string
     *  @throws  DriverException
     */
    @Override
    public String date() throws DriverException {
        return "1Jan2018";
    }


    /**
     *  Gets the version string
     *
     *  @return  The string
     *  @throws  DriverException
     */
    @Override
    public String version() throws DriverException {
        return "1.0";
    }


    /**
     *  Gets the device string
     *
     *  @return  The string
     *  @throws  DriverException
     */
    @Override
    public String device() throws DriverException {
        return "Sim";
    }


    /**
     *  Selects the LED
     *
     *  @param  led  The enumerated LED
     *  @throws  DriverException
     */
    @Override
    public void selectLed(LED led) throws DriverException {
    }


    /**
     *  Sets the LED current
     *
     *  @param  current  The current value
     *  @throws  DriverException
     */
    @Override
    public void setLedCurrent(double current) throws DriverException {
        ledCurr = current;
    }


    /**
     *  Sets the exposure tine
     *
     *  @param  time  The exposure time
     *  @throws  DriverException
     */
    @Override
    public void setExposureTime(double time) throws DriverException {
    }


    /**
     *  Sets LED pulse mode
     *
     *  @throws  DriverException
     */
    @Override
    public void pulse() throws DriverException {
    }


    /**
     *  Sets shutter mode
     *
     *  @throws  DriverException
     */
    @Override
    public void shutter() throws DriverException {
    }


    /**
     *  Starts exposure
     *
     *  @throws  DriverException
     */
    @Override
    public void startExposure() throws DriverException {
    }


    /**
     *  Polls for exposure end
     *
     *  @return  Whether ended
     *  @throws  DriverException
     */
    @Override
    public boolean pollEnd() throws DriverException {
        return true;
    }

    
    /**
     *  Reads the converted ADC values.
     *
     *  @return  Eight-element array of values
     *  @throws  DriverException
     */
    @Override
    public CCOBDataRead getAdcValues() throws DriverException {
        double[] values = new double[CCOBDataRead.NUM_ADCS];
        values[CCOBDataRead.ADC_LED_CURRENT] = ledCurr;
        values[CCOBDataRead.ADC_LED_VOLTAGE] = 0.0;
        values[CCOBDataRead.ADC_LED_VREF] = 0.0;
        values[CCOBDataRead.ADC_LED_TEMP1] = 25.0;
        values[CCOBDataRead.ADC_LED_TEMP2] = 25.1;
        values[CCOBDataRead.ADC_BOARD_TEMP] = 24.0;
        values[CCOBDataRead.ADC_SPHERE_TEMP] = 30.0;
        values[CCOBDataRead.ADC_PD_CURRENT] = 0.0;
        return new CCOBDataRead(values);
    }


    /**
     *  Reads the photodiode voltage.
     *
     *  @return  The PD voltage
     *  @throws  DriverException
     */
    @Override
    public double getAdcPhotoDiode() throws DriverException {
        return 0.0;
    }
    
 
    /**
     *  Turns on a board LED.
     *
     *  @param  led  The LED number (1 - 4)
     *  @param  on   The on state: non-0 = on, 0 = off
     *  @throws  DriverException
     */
    @Override
    public void ledOn(int led, int on) throws DriverException {
        if (led < 1 || led > 4) {
            throwException("Invalid board LED number");
        }
    }


    /**
     *  Reads the state of a button.
     *
     *  @param  button  The button number (0 - 4)
     *  @throws  DriverException
     */
    @Override
    public boolean readButton(int button) throws DriverException {
        if (button < 0 || button > 4) {
            throwException("Invalid button number");
        }
        return (button & 1) == 0;
    }


    /**
     *  Throws an exception.
     *
     *  @param  msg  The message text
     *  @throws  DriverException
     */
    private void throwException(String msg) throws DriverException
    {
        throw new DriverException("CCOB error: " + msg);
    }

}
