package org.lsst.ccs.drivers.ccob;

import org.lsst.ccs.drivers.commons.DriverException;

/**
 * A high-level interface to the CCOB designed to be implementable using either
 * sockets or JNI or UsbLib
 *
 * @author CCS
 */
public interface CCOBInterface {

    enum LED { CARD, BLUE, RED, UV, NM750, NM850, NM960 }

    /**
     * Initializes the interface, should be called before anything else
     *
     * @throws org.lsst.ccs.drivers.commons.DriverException If the
     * initialization fails
     */
    public void init() throws DriverException;

    /**
     * Closes the interface, should be called at end or before you restart a cycle
     *
     * @throws DriverException
     */
    public void stop() throws DriverException;

    /**
     * The version of the firmware
     *
     * @return
     * @throws org.lsst.ccs.drivers.commons.DriverException
     */
    public String version() throws DriverException;

    /**
     * The date of compilation of the firmware
     *
     * @return
     * @throws org.lsst.ccs.drivers.commons.DriverException
     */
    public String date() throws DriverException;

    /**
     * Additional firmware details
     *
     * @return
     * @throws org.lsst.ccs.drivers.commons.DriverException
     */
    public String misc() throws DriverException;

    /**
     * Manufacturer/developer info from the firmware
     *
     * @return
     * @throws org.lsst.ccs.drivers.commons.DriverException
     */
    public String manu() throws DriverException;

    /**
     * Identify the device type
     * 
     * @return The name of the device
     * @throws org.lsst.ccs.drivers.commons.DriverException
     */
    public String device() throws DriverException;

    /**
     * Select which LED will be turned on (by startExposure, pulseLED etc)
     *
     * @param led
     * @throws DriverException
     */
    public void selectLed(LED led) throws DriverException;

    /**
     * Sets the current (for all LEDs)
     *
     * @param current Current, in range 0-0.273 (Amps)
     * @throws DriverException
     */
    public void setLedCurrent(double current) throws DriverException;
    
    /**
     * Duration for use in pulsed mode
     * @param time Exposure time (undefined units) (0-65535)
     * @throws DriverException 
     */
    
    public void setExposureTime(double time) throws DriverException;

    /**
     * Set the mode to pulsed, timed by setExposureTime
     * @throws org.lsst.ccs.drivers.commons.DriverException
     */
    public void pulse() throws DriverException;
    
    /**
     * Set the mode to shutter (unlimited time, remember to stop it)
     * @throws org.lsst.ccs.drivers.commons.DriverException
     */
    public void shutter() throws DriverException;
    
    /**
     * Turn on the LED based on exposure time and LED current
     * @throws org.lsst.ccs.drivers.commons.DriverException
     */
    public void startExposure() throws DriverException;
    
    /**
     * Check to see if the exposure has finished
     * @return
     * @throws org.lsst.ccs.drivers.commons.DriverException
     */
    public boolean pollEnd() throws DriverException;
    
    /**
     * Return housekeeping ADC values. Some are latched on specific moments
     * @return
     * @throws DriverException 
     */
    public CCOBDataRead getAdcValues() throws DriverException;

    /**
     * Return photodiode ADC values
     * @return
     * @throws DriverException 
     */
    public double getAdcPhotoDiode() throws DriverException;

    /**
     * Turn one of the 4 onboard LED on or off
     * @param led LED number
     * @param on On state
     * @throws DriverException 
     */
    public void ledOn(int led, int on) throws DriverException;

    /**
     * Reads the state of one of the 5 buttons
     * @param button Button number
     * @return
     * @throws DriverException 
     */
    public boolean readButton(int button) throws DriverException;

    @Deprecated
    public default void selectLED(LED led) throws DriverException {
        selectLed(led);
    }

    @Deprecated
    public default void setLEDCurrent(double current) throws DriverException {
        setLedCurrent(current);
    }
}
