package org.lsst.ccs.drivers.ccob;

import java.io.Serializable;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 * Constructor CCOBDataRead(String ).
 * 
 * To use construct on String returned from CCOB (call it ccob_readback)
 * CCOBData mydata ccob_readback)
 *
 *  To get the original string readback from CCOB 
 *   String asString = mydata.getInputString();
 *
 *  To get readback from CCOB as an array of strings 
 *  String [] asStringArray = mydata.getInputStringAsArray();
 *
 * To get readback from CCOB as an array of doubles
 * double [] asDoubletArray = mydata.getDoubleArray()
 *
 * Instead or as well get the individual parameters
 * 
 * get the led current
 * double ledI = mydata.getLedCurrent();
 * 
 * get the photodiode current
 * double phdeI = mydata.getPhotodiodeCurrent();
 *
 *   
 * get the sphere temperature
 * double tSph = mydata.getTempSphere();
 *
 * get board temperature
 * double tBrd = mydata.getTempBrd();
 *
 *  get led1 temperature 
 *  double tLed1 = mydata.getTempLed1();
 *
 *  get led2 temperature  
 *  double tLed2 = mydata.getTempLed2();
 *
 *  get led voltage
 *  double ledV = mydata.getLedVoltage();
 *
 *  get led reference voltate  
 *  double ledVref = mydata.getLedVref();
 *
 *  
 *
 * @author Farrukh Azfar. 
 */

public class CCOBDataRead implements Serializable {

    public static final int
        ADC_LED_VOLTAGE = 0,
        ADC_LED_CURRENT = 1,
        ADC_LED_VREF    = 2,
        ADC_LED_TEMP1   = 3,
        ADC_LED_TEMP2   = 4,
        ADC_PD_CURRENT  = 5,
        ADC_SPHERE_TEMP = 6,
        ADC_BOARD_TEMP  = 7,
        NUM_ADCS        = 8;

    // these are returned from private member functions ...

    // the parameters themselves 
    private double ledVoltage, ledCurrent, ledVref, tempLed1, tempLed2, photodiodeCurrent, tempSphere, tempBrd;
    private double [] asDoubleArray;
    private String [] asStringArray;
    private String originalString;
  

    // return functions ...

     // currents
 
    public double getLedCurrent() {         return ledCurrent;}
    public double getPhotodiodeCurrent() {  return photodiodeCurrent;}

    // temperatures

    public double getTempSphere() {         return tempSphere;}
    public double getTempBrd() {            return tempBrd;}
    public double getTempLed1() {           return tempLed1;}
    public double getTempLed2() {           return tempLed2;}
     

    // voltages

    public double getLedVoltage() {return ledVoltage;}
    public double getLedVref() {return ledVref;}

    // original String

    public String getInputString() {return originalString;}

    // original String as array 
    public String [] getInputStringAsArray() {return asStringArray;}
    
    // as double array 
    public double [] getDoubleArray() {return asDoubleArray;}

    // toString 
    @Override
    public  String toString() { 
	return String.format("\tLED Voltage = %g\n"+
                             "\tLED Current = %g\n"+
                             "\tLED Vref    = %g\n"+
                             "\tLED Temp1   = %g\n"+
                             "\tLED Temp2   = %g\n"+
                             "\tPhotodiode Current = %g\n"+
                             "\tSphere temperature = %g\n"+
                             "\tBoard temperature  = %g", 
                             ledVoltage, ledCurrent, ledVref, tempLed1, 
                             tempLed2, photodiodeCurrent, tempSphere, tempBrd);
    }
    
    public CCOBDataRead(double[] data) throws DriverException {
        if (data.length < NUM_ADCS) {
            throw new DriverException("CCOB ADC data array is too short");
        }
        asDoubleArray = new double[NUM_ADCS];
        System.arraycopy(data, 0, asDoubleArray, 0, NUM_ADCS);
        ledVoltage = asDoubleArray[ADC_LED_VOLTAGE]; 
	ledCurrent = asDoubleArray[ADC_LED_CURRENT]; 
	ledVref = asDoubleArray[ADC_LED_VREF]; 
	tempLed1 = asDoubleArray[ADC_LED_TEMP1];  
	tempLed2 = asDoubleArray[ADC_LED_TEMP2];  
	photodiodeCurrent = asDoubleArray[ADC_PD_CURRENT]; 
	tempSphere = asDoubleArray[ADC_SPHERE_TEMP];
        tempBrd = asDoubleArray[ADC_BOARD_TEMP];
        
        asStringArray = new String[NUM_ADCS];
        originalString = "";
        for (int i = 0; i < NUM_ADCS; i++) {
            asStringArray[i] = String.valueOf(asDoubleArray[i]);
            originalString += asStringArray[i] + ", ";
        }
    }

    public CCOBDataRead(String response) throws DriverException {
        originalString = response;
 	// set everything that could possibly be returned 
	asStringArray = originalString.split(" +");
        if (asStringArray.length < NUM_ADCS) {
            throw new DriverException("CCOB ADC data string is invalid");
        }
	asDoubleArray = new double[(asStringArray).length];      

	for(int j = 0; j< (asStringArray.length); j++) {
	    asDoubleArray[j] = Double.valueOf(asStringArray[j]);
	}

	// only way to do this ... 
        ledVoltage = asDoubleArray[ADC_LED_VOLTAGE]; 
	ledCurrent = asDoubleArray[ADC_LED_CURRENT]; 
	ledVref = asDoubleArray[ADC_LED_VREF]; 
	tempLed1 = asDoubleArray[ADC_LED_TEMP1];  
	tempLed2 = asDoubleArray[ADC_LED_TEMP2];  
	photodiodeCurrent = asDoubleArray[ADC_PD_CURRENT]; 
	tempSphere = asDoubleArray[ADC_SPHERE_TEMP];
        tempBrd = asDoubleArray[ADC_BOARD_TEMP];
	
    } // end constructor on string -

    private static final long serialVersionUID = -5659896462402886077L;
}
