package org.lsst.ccs.drivers.ccob;

import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.usb.UsbDevice;
import org.lsst.ccs.drivers.usb.UsbLib;
import org.lsst.ccs.utilities.conv.Convert;

/**
 * An implementation of the CCOB driver which uses the CCS USB driver
 *
 * @author CCS
 */
public class CCOBUsb implements CCOBInterface {

    private static final int
        CCOB_VID = 0x04d8,
        CCOB_DID = 0x003f,
        EP_IN = 0x81,
        EP_OUT = 0x01,
        BUFF_SIZE = 64,
        READ_TIMEOUT = 1000,
        DAC_MAX = 65535;

    private static final byte
        CID_INIT = 0,
        CID_STOP = 1,
        CID_MISC = 11,
        CID_MANU = 12,
        CID_DATE = 13,
        CID_VERSION = 14,
        CID_DEVICE = 15,
        CID_SELECT_LED = 16,
        CID_SET_LED_CURRENT = 17,
        CID_PULSE_LED = 18,
        CID_USE_SHUTTER = 19,
        CID_SET_EXPOSURE_TIME = 20,
        CID_START_EXPOSURE = 21,
        CID_POLL_END = 22,
        CID_SET_HEATER_CURRENT = 32,
        CID_SET_TEMP_SETPOINT = 33,
        CID_SET_PID_PARAM = 34,
        CID_START_PID = 33,
        CID_STOP_PID = 34,
        CID_GET_ADC_VALUES = 48,
        CID_LED_1_OFF = 49,
        CID_LED_1_ON = 50,
        CID_LED_2_OFF = 51,
        CID_LED_2_ON = 52,
        CID_LED_3_OFF = 53,
        CID_LED_3_ON = 54,
        CID_LED_4_OFF = 55,
        CID_LED_4_ON = 56,
        CID_GET_ADC_PHOTODIODE = 64,
        CID_SW0 = 80,
        CID_SW1 = 81,
        CID_SW2 = 82,
        CID_SW3 = 83,
        CID_SW4 = 84,
        CID_RESP = (byte)0x80,
        CID_ERROR = (byte)0xff;

    private static final int
        ADC_TYPE_VOLTAGE = 0,
        ADC_TYPE_CURRENT = 1,
        ADC_TYPE_TEMP    = 2;

    private static final double
        CNV_ADC_VOLTS = 4.096 / 65535,
        CNV_ADC_AMPS  = 0.36813 / 65535,
        CNV_TEMP_A = 3.9083e-3,
        CNV_TEMP_B = -5.775e-7,
        CNV_DAC_AMPS = DAC_MAX / 0.273,
        CNV_DAC_TIME = 21.318e-6;

    private static final int[] adcType = new int[CCOBDataRead.NUM_ADCS];
    static {
        adcType[CCOBDataRead.ADC_LED_VOLTAGE] = ADC_TYPE_VOLTAGE;
        adcType[CCOBDataRead.ADC_LED_CURRENT] = ADC_TYPE_CURRENT;
        adcType[CCOBDataRead.ADC_LED_VREF]    = ADC_TYPE_VOLTAGE;
        adcType[CCOBDataRead.ADC_LED_TEMP1]   = ADC_TYPE_TEMP;
        adcType[CCOBDataRead.ADC_LED_TEMP2]   = ADC_TYPE_TEMP;
        adcType[CCOBDataRead.ADC_PD_CURRENT]  = ADC_TYPE_CURRENT;
        adcType[CCOBDataRead.ADC_SPHERE_TEMP] = ADC_TYPE_TEMP;
        adcType[CCOBDataRead.ADC_BOARD_TEMP]  = ADC_TYPE_TEMP;
    }

    private static final byte[] ledCmnds = {CID_LED_1_OFF, CID_LED_1_ON,
                                            CID_LED_2_OFF, CID_LED_2_ON,
                                            CID_LED_3_OFF, CID_LED_3_ON,
                                            CID_LED_4_OFF, CID_LED_4_ON};

    private static final byte[] switchCmnds = {CID_SW0, CID_SW1, CID_SW2, CID_SW3, CID_SW4};

    private final int index;
    private UsbDevice usb;
    private boolean debug = false;


    /**
     *  Constructor.
     */
    public CCOBUsb() {
        this(0);
    }


    /**
     *  Constructor.
     *
     *  @param  index  The index of the device in the list of matching ones
     */
    public CCOBUsb(int index) {
        this.index = index;
    }


    /**
     *  Initializes the connection to a device
     *
     *  @throws  DriverException
     */
    @Override
    public void init() throws DriverException {
        if (usb != null) {
            throwException("Device is already connected");
        }
        usb = UsbLib.getDevice(CCOB_VID, CCOB_DID, null, index);
        if (usb == null) {
            throwException("Device not found");
        }
        try {
            usb.claimInterface(0, true);
        }
        catch (DriverException e) {
            usb.close();
            usb = null;
            throw e;
        }
        writeCommand(CID_INIT);
    }


    /**
     *  Stops the CCOB and closes the connection
     *
     *  @throws  DriverException
     */
    @Override
    public void stop() throws DriverException {
        writeCommand(CID_STOP);
        usb.releaseInterface(0);
        usb.close();
        usb = null;
    }


    /**
     *  Gets the miscellaneous string
     *
     *  @return  The string
     *  @throws  DriverException
     */
    @Override
    public String misc() throws DriverException {
        return readString(CID_MISC);
    }


    /**
     *  Gets the manufacturer string
     *
     *  @return  The string
     *  @throws  DriverException
     */
    @Override
    public String manu() throws DriverException {
        return readString(CID_MANU);
    }


    /**
     *  Gets the date string
     *
     *  @return  The string
     *  @throws  DriverException
     */
    @Override
    public String date() throws DriverException {
        return readString(CID_DATE);
    }


    /**
     *  Gets the version string
     *
     *  @return  The string
     *  @throws  DriverException
     */
    @Override
    public String version() throws DriverException {
        return readString(CID_VERSION);
    }


    /**
     *  Gets the device string
     *
     *  @return  The string
     *  @throws  DriverException
     */
    @Override
    public String device() throws DriverException {
        return readString(CID_DEVICE);
    }


    /**
     *  Selects the LED
     *
     *  @param  led  The enumerated LED
     *  @throws  DriverException
     */
    @Override
    public void selectLed(LED led) throws DriverException {
        read(new byte[]{2, CID_SELECT_LED, (byte)led.ordinal()});
    }


    /**
     *  Sets the LED current
     *
     *  @param  current  The current value
     *  @throws  DriverException
     */
    @Override
    public void setLedCurrent(double current) throws DriverException {
        byte[] cmnd = {4, CID_SET_LED_CURRENT, 0, 0, 0};
        int dac = (int)(CNV_DAC_AMPS * current);
        Convert.shortToBytes((short)Math.max(Math.min(dac, DAC_MAX), 0), cmnd, 3);
        read(cmnd);
    }


    /**
     *  Sets the exposure tine
     *
     *  @param  time  The exposure time
     *  @throws  DriverException
     */
    @Override
    public void setExposureTime(double time) throws DriverException {
        byte[] cmnd = {3, CID_SET_EXPOSURE_TIME, 0, 0};
        int dac = DAC_MAX - (int)(time / CNV_DAC_TIME);
        Convert.shortToBytes((short)Math.max(Math.min(dac, DAC_MAX), 0), cmnd, 2);
        read(cmnd);
    }


    /**
     *  Sets LED pulse mode
     *
     *  @throws  DriverException
     */
    @Override
    public void pulse() throws DriverException {
        writeCommand(CID_PULSE_LED);
    }


    /**
     *  Sets shutter mode
     *
     *  @throws  DriverException
     */
    @Override
    public void shutter() throws DriverException {
        writeCommand(CID_USE_SHUTTER);
    }


    /**
     *  Starts exposure
     *
     *  @throws  DriverException
     */
    @Override
    public void startExposure() throws DriverException {
        writeCommand(CID_START_EXPOSURE);
    }


    /**
     *  Polls for exposure end
     *
     *  @return  Whether ended
     *  @throws  DriverException
     */
    @Override
    public boolean pollEnd() throws DriverException {
        byte[] resp = read(new byte[]{1, CID_POLL_END});
        if (resp[0] < 2) {
            throwException("Response to pollEnd is too short");
        }
        return resp[2] != 0;
    }

    
    /**
     *  Reads the converted ADC values.
     *
     *  @return  Eight-element array of values
     *  @throws  DriverException
     */
    @Override
    public CCOBDataRead getAdcValues() throws DriverException {
        byte[] resp = read(new byte[]{1, CID_GET_ADC_VALUES});
        if (resp[0] < 2 * CCOBDataRead.NUM_ADCS + 1) {
            throwException("Response to getAdcValues is too short");
        }
        double[] data = new double[CCOBDataRead.NUM_ADCS];
        for (int j = 0; j < CCOBDataRead.NUM_ADCS; j++) {
            int raw = Convert.bytesToShortBE(resp, 2 * j + 2) & 0xffff;
            switch (adcType[j]) {
            case ADC_TYPE_VOLTAGE:
                data[j] = CNV_ADC_VOLTS * raw; break;
            case ADC_TYPE_CURRENT:
                data[j] = CNV_ADC_AMPS * raw; break;
            case ADC_TYPE_TEMP:
                data[j] = convertAdcTemp(raw); break;
            }
        }
        return new CCOBDataRead(data);
    }


    /**
     *  Reads the photodiode voltage.
     *
     *  @return  The PD voltage
     *  @throws  DriverException
     */
    @Override
    public double getAdcPhotoDiode() throws DriverException {
        byte[] resp = read(new byte[]{1, CID_GET_ADC_PHOTODIODE});
        if (resp[0] < 3) {
            throwException("Response to getAdcPhotoDiode is too short");
        }
        return CNV_ADC_VOLTS * (Convert.bytesToShortBE(resp, 2) & 0xffff);
    }
    
 
    /**
     *  Turns on a board LED.
     *
     *  @param  led  The LED number (1 - 4)
     *  @param  on   The on state: non-0 = on, 0 = off
     *  @throws  DriverException
     */
    @Override
    public void ledOn(int led, int on) throws DriverException {
        if (led < 1 || led > 4) {
            throwException("Invalid board LED number");
        }
        writeCommand(ledCmnds[2 * (led - 1) + (on == 0 ? 0 : 1)]);
    }


    /**
     *  Reads the state of a button.
     *
     *  @param  button  The button number (0 - 4)
     *  @throws  DriverException
     */
    @Override
    public boolean readButton(int button) throws DriverException {
        if (button < 0 || button > 4) {
            throwException("Invalid button number");
        }
        byte[] resp = read(new byte[]{1, switchCmnds[button]});
        if (resp[0] < 2) {
            throwException("Response to readButton is too short");
        }
        return resp[2] == 0;
    }


    /**
     *  Converts ADC counts to temperature.
     *
     *  @param  adc  The raw ADC value
     *  @return  The converted temperature
     */
    private static double convertAdcTemp(int adc)
    {
        return (Math.sqrt(CNV_TEMP_A * CNV_TEMP_A + 4 * CNV_TEMP_B * (CNV_ADC_VOLTS * adc - 1.0)) - CNV_TEMP_A)
                  / (2 * CNV_TEMP_B);
    }


    /**
     *  Writes a command with no arguments
     *
     *  @param  code  The command code
     *  @throws  DriverException
     */
    private void writeCommand(byte code) throws DriverException
    {
        read(new byte[]{1, code});
    }


    /**
     *  Reads an information string
     *
     *  @param  code  The command code
     *  @return  The string
     *  @throws  DriverException
     */
    private String readString(byte code) throws DriverException
    {
        byte[] resp = read(new byte[]{1, code});
        return new String(resp, 2, resp[0] - 2);  // Remove trailing zero
    }


    /**
     *  Writes a command and reads the response.
     *
     *  This is used by all read and write operations.
     *
     *  @param  cmnd  The command buffer
     *  @return  The response buffer
     *  @throws  DriverException
     */
    private synchronized byte[] read(byte[] cmnd) throws DriverException
    {
        if (usb == null) {
            throwException("Device is not connected");
        }
        usb.interruptWrite(EP_OUT, cmnd, 0, cmnd.length, 0);
        byte[] resp = new byte[BUFF_SIZE];
        int leng = usb.interruptRead(EP_IN, resp, 0, resp.length, READ_TIMEOUT);
        if (debug) {
            System.out.format("Response (%s):", leng);
            for (int j = 0; j < 20; j++) {
                System.out.format(" %02x", resp[j] & 0xff);
            }
            System.out.println();
        }
        if (leng < 2 || resp[0] < 1) {
            throwException("Response is too short");
        }
        if ((cmnd[1] | CID_RESP) != resp[1]) {
            throwException("Response doesn't match command");
        }
        return resp;
    }


    /**
     *  Throws an exception.
     *
     *  @param  msg  The message text
     *  @throws  DriverException
     */
    private void throwException(String msg) throws DriverException
    {
        throw new DriverException("CCOB error: " + msg);
    }

}
