package org.lsst.ccs.drivers.ccob;

import java.io.IOException;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 * Adds convenience commands on top of the driver implementation.
 *
 * @author CCS
 */
public class CCOBCommands implements CCOBInterface {

    private CCOBInterface driver;
    
    public CCOBCommands(CCOBInterface driver) {
        this.driver = driver;
    }

    public void setDriver(CCOBInterface driver) {
        this.driver = driver;
    }

    @Command(description="Close connection")
    public void close() throws IOException, DriverException {
        driver.stop();
    }

    @Command(description="Open connection and initialize CCOB")
    @Override
    public void init() throws DriverException {
        checkSocket().init();
    }

    @Command(description="Stop CCOB and close connection")
    @Override
    public void stop() throws DriverException {
        checkSocket().stop();
    }

    @Command(description="Get version string")
    @Override
    public String version() throws DriverException {
        return checkSocket().version();
    }

    @Command(description="Get date string")
    @Override
    public String date() throws DriverException {
        return checkSocket().date();
    }

    @Command(description="Get miscellaneous string")
    @Override
    public String misc() throws DriverException {
        return checkSocket().misc();
    }

    @Command(description="Get manufacturer string")
    @Override
    public String manu() throws DriverException {
        return checkSocket().manu();
    }

    @Command(description="Get device string")
    @Override
    public String device() throws DriverException {
        return checkSocket().device();
    }

    @Command(description="Select the LED")
    @Override
    public void selectLed(@Argument(name = "led", description = "LED name")
                          LED led) throws DriverException {
        checkSocket().selectLed(led);
    }

    @Command(description="Set the LED current")
    @Override
    public void setLedCurrent(@Argument(description = "LED current (amps)")
                              double current) throws DriverException {
        checkSocket().setLedCurrent(current);

    }

    @Command(description="Set the exposure time")
    @Override
    public void setExposureTime(@Argument(description = "Exposure time (secs)")
                                double time) throws DriverException {
        checkSocket().setExposureTime(time);
    }

    @Command(description="Set pulse mode")
    @Override
    public void pulse() throws DriverException {
        checkSocket().pulse();
    }

    @Command(description="Set shutter mode")
    @Override
    public void shutter() throws DriverException {
        checkSocket().shutter();
    }

    @Command(description="Start exposure")
    @Override
    public void startExposure() throws DriverException {
        checkSocket().startExposure();
    }

    @Command(description="Poll for end of exposure")
    @Override
    public boolean pollEnd() throws DriverException {
        return checkSocket().pollEnd();
    }

    @Command(description="Read ADC values")
    @Override
    public CCOBDataRead getAdcValues() throws DriverException {
        return checkSocket().getAdcValues();
    }

    @Command(description="Read photodiode voltage")
    @Override
    public double getAdcPhotoDiode() throws DriverException {
        return checkSocket().getAdcPhotoDiode();
    }

    @Command(description="Read LED current")
    public double getLedCurrent() throws DriverException {
        return getAdcValues().getLedCurrent();
    }

    @Command(description="Read LED voltage")
    public double getLedVoltage() throws DriverException {
        return getAdcValues().getLedVoltage();
    }

    @Command(description="Read LED reference voltage")
    public double getLedRefVoltage() throws DriverException {
        return getAdcValues().getLedVref();
    }

    @Command(description="Read sphere temperature")
    public double getSphereTemperature() throws DriverException {
        return getAdcValues().getTempSphere();
    }

    @Command(description="Read board temperature")
    public double getBoardTemperature() throws DriverException {
        return getAdcValues().getTempBrd();
    }

    @Command(description="Read LED temperature 1")
    public double getLed1Temperature() throws DriverException {
        return getAdcValues().getTempLed1();
    }

    @Command(description="Read LED temperature 2")
    public double getLed2Temperature() throws DriverException {
        return getAdcValues().getTempLed2();
    }

    @Command(description="Set board LED state")
    @Override
    public void ledOn(@Argument(description = "LED number (1 - 4)")
                      int led,
                      @Argument(description = "LED state (1 = on, 0 = off)")
                      int on) throws DriverException {
        checkSocket().ledOn(led, on);
    }

    @Command(description="Get button state")
    @Override
    public boolean readButton(@Argument(description = "Button number (0 - 4)")
                              int Button) throws DriverException {
        return checkSocket().readButton(Button);
    }

    private CCOBInterface checkSocket() {
        if (driver == null) {
            throw new RuntimeException("Socket not open");
        }
        return driver;
    }

}
