package org.lsst.ccs.drivers.canopenjni;

import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *
 * @author emarin
 */
public class CanFestivalJNI implements CanOpenInterface {

    private static final Logger log = Logger.getLogger("org.lsst.ccs.drivers.canopenjni");

    static {
        System.out.println("**** Loading the library ");
        try {
            System.loadLibrary("canopenJNI");
        } catch (Exception e) {
            e.printStackTrace();
        }
        System.out.println("**** Done Loading the library ");
    }

    private long sdoTimeout = 5000;
    private long pdoTimeout = 500;

    private BootMessageListener bml = (nodeId) -> {
        System.out.println("boot : " + Integer.toHexString(nodeId));
    };

    private EmergencyMessageListener eml = (nodeId, errCode, errReg) -> {
        System.out.println("emergency message for node 0x" + Integer.toHexString(nodeId) + " :(errCode 0x"
                + Integer.toHexString(errCode) + " errReg 0x" + Integer.toHexString(errReg) + ")");
    };

    /**
     * Enables reception of the PDO identified by cob_id at the specified index.
     *
     * @param cobId
     */
    @Override
    public native void addReceivedPDO(int cobId);

    /**
     * Reset all RxPDOs previously added.
     */
    @Override
    public native void clearReceivedPDOs();

    /**
     *
     * @param master  1
     * @param baud    "125K" or 1M
     * @param busName 0 or 1
     * @param nodeID  the master nodeId : 8
     */
    @Command
    @Override
    public native void init(int master, String baud, String busName, int nodeID);

    @Override
    public void setEmergencyMessageListener(EmergencyMessageListener eml) {
        this.eml = eml;
    }

    // Called back from jni
    private void onEmergencyMessage(int nodeid, int errCode, int errReg) {
        eml.onEmergencyMessage(nodeid, errCode, errReg);
    }

    @Override
    public void setBootMessageListener(BootMessageListener bml) {
        this.bml = bml;
    }

    // Called back from jni
    private void onSlaveBootup(int nodeid) {
        bml.onBootMessage(nodeid);
    }

    @Override
    @Command
    public PDOData sync() throws DriverException, DriverTimeoutException, ConcurrentCallException {
        PDOData rd = new PDOData();
        long beginTime = System.currentTimeMillis();
        log.info("sync sent to CANbus");

        try {
            rd = new SyncCommandWrapper<PDOData>((pdocb, r) -> {
                syncAsync(pdocb, r);
            }, rd, () -> {
                clearCallBack(0, 0);
            }, pdoTimeout).get();
            if (rd.errCode != 0) {
                throw new DriverException(String.valueOf(rd.errCode));
            }
        } catch (DriverTimeoutException ex) {
            log.info("sync timed out, returning partial PDO data", ex);
            rd.errCode = 0x99;
        }
        long duration = System.currentTimeMillis() - beginTime;
        log.info("data returned from CANbus after sync - duration = " + duration);
        return rd;
    }

    /**
     * Sends a SYNC request.
     */
    private native void syncAsync(CallbackListener cb, PDOData data);

    @Command
    @Override
    public native int scan();

    @Override
    @Command
    @Deprecated
    public String info(int nodeID) throws DriverException {
        long deviceType = rsdo(nodeID, 0x1000, 0x00);
        long vendorID = rsdo(nodeID, 0x1018, 0x01);
        //TODO get rid of these ugly tests when sdo.c bug has been fixed.
        if (vendorID == deviceType) {
            log.warning("error for vendor, has to read again for nodeID=" + nodeID);
            vendorID = rsdo(nodeID, 0x1018, 0x01);
        }
        long productCode = rsdo(nodeID, 0x1018, 0x02);
        if (productCode == vendorID) {
            log.warning("error for productCode, has to read again for nodeID=" + nodeID);
            productCode = rsdo(nodeID, 0x1018, 0x02);
        }
        long revNumber = rsdo(nodeID, 0x1018, 0x03);
        if (revNumber == productCode) {
            log.warning("error for revNumber, has to be read again for nodeID=" + nodeID);
            revNumber = rsdo(nodeID, 0x1018, 0x03);
        }
        long serialNb = rsdo(nodeID, 0x1018, 0x04);
        if (serialNb == revNumber) {
            log.warning("error for serialNb, has to be read again for nodeID=" + nodeID);
            serialNb = rsdo(nodeID, 0x1018, 0x04);
        }

        StringBuilder res = new StringBuilder("info," + nodeID);

        return res.append(",").append(Long.toHexString(deviceType)).append(",").append(Long.toHexString(vendorID))
                .append(",").append(Long.toHexString(productCode)).append(",").append(Long.toHexString(revNumber))
                .append(",").append(Long.toHexString(serialNb)).toString();
    }

    @Override
    @Command
    public synchronized void wsdo(int nodeId, int index, int subindex, int size, long data)
            throws DriverException, DriverTimeoutException, ConcurrentCallException {
        ReturnData rd = new ReturnData();
        rd.nodeId = nodeId;
        rd.otherData = nodeId + ":" + index + "/" + subindex;

        ReturnData rd2 = new SyncCommandWrapper<ReturnData>((sdocb, r) -> {
            log.info("> wsdo async " + nodeId + " " + index + " " + subindex);
            wsdoAsync(nodeId, index, subindex, size, data, sdocb, r);
            log.info("< wsdo async " + nodeId + " " + index + " " + subindex);
        }, rd, () -> {
            if (nodeId != rd.nodeId) {
                log.error("nodeid discrepency call " + nodeId + " return " + rd.nodeId);
            }
            log.info("> wsdo clear " + nodeId + " " + index + " " + subindex + " ret " + rd.nodeId);
            clearCallBack(nodeId, 1);
        }, sdoTimeout).get();

        if (rd.errCode != 0) {
            throw new SDOException(rd.errCode, rd.abortCode);
        }
    }

    /**
     * Write device entry
     *
     * @param nodeID   the node id
     * @param index    the index
     * @param subindex the subindex
     * @param size     the size of the data
     * @param data     the data.
     * @param cb       the callback
     */
    private native void wsdoAsync(int nodeID, int index, int subindex, int size, long data, CallbackListener cb,
            ReturnData rd);

    @Override
    @Command
    public synchronized long rsdo(int nodeId, int index, int subindex)
            throws DriverException, DriverTimeoutException, ConcurrentCallException {
        SDOData ret = new SDOData();
        ret.otherData = nodeId + ":" + index + "/" + subindex;
        ret.nodeId = nodeId;

        // TODO CHECK return is ignored, get returns an SDODate
        new SyncCommandWrapper<SDOData>((sdocb, rd) -> {
            log.info("> rsdo async " + nodeId + " " + index + " " + subindex);
            rsdoAsync(nodeId, index, subindex, sdocb, rd);
            log.info("< rsdo async " + nodeId + " " + index + " " + subindex + " rd" + rd.toString());
        }, ret, () -> {
            if (nodeId != ret.nodeId) {
                log.error("nodeid discrepency call " + nodeId + " return " + ret.nodeId);
            }
            log.info("> rsdo clear " + nodeId + " " + index + " " + subindex + " ret " + ret.nodeId + " "
                    + ((SDOData) ret).data + " ret.errCode= " + ret.errCode + " ret.abortCode");
            clearCallBack(nodeId, 0);
        }, sdoTimeout).get();

        if (ret.errCode != 0) {
            throw new SDOException(ret.errCode, ret.abortCode);
        } else {
            return ret.data;
        }
    }

    private native void rsdoAsync(int nodeId, int index, int subindex, CallbackListener cb, ReturnData rd);

    /**
     * Slave start
     *
     * @param nodeId
     */
    @Override
    public native void ssta(int nodeId);

    /**
     * Slave stop
     *
     * @param nodeId
     */
    @Override
    public native void ssto(int nodeId);

    /**
     * Slave reset
     *
     * @param nodeId
     */
    @Override
    public native void reset(int nodeId);

    @Override
    public native void quit();

    // Lifecycle methods.
    @Override
    public void init() {

    }

    @Override
    public void start() {

    }

    @Override
    public void stop() {

    }

    @Override
    public boolean isReady() {
        return true;
    }

    @Override
    public void setSdoTimeout(long sdoTimeout) {
        this.sdoTimeout = sdoTimeout;
    }

    /**
     * set a timeout in milliseconds for PDO : after a sync, canfestival returns
     * DATA after this timeout.
     *
     * @param aPDOTimeout in milliseconds
     */
    public void setPdoTimeout(long aPDOTimeout) {
        this.pdoTimeout = aPDOTimeout;
    }

    private interface CommandWithCallback<T extends ReturnData> {

        public void call(CallbackListener cb, T rd);

    }

    private interface AbortCallbackCommand {

        public void abortCallback();
    }

    /**
     * Make an asynchronous call synchronous by calling a callback when the return
     * data is set.
     *
     * @param <T>
     */
    private class SyncCommandWrapper<T extends ReturnData> {

        private final CommandWithCallback<T> callable;
        private final AbortCallbackCommand abort;
        private final T rd;
        private final long to;

        SyncCommandWrapper(CommandWithCallback<T> asyncCommand, T rd, AbortCallbackCommand abortCommand, long timeout) {
            this.callable = asyncCommand;
            this.rd = rd;
            this.abort = abortCommand;
            this.to = timeout;
        }

        T get() throws DriverException {
            final Lock cblock = new ReentrantLock();
            Condition cb = cblock.newCondition();

            CallbackListener sdoCB = new CallbackListener() {

                @Override
                public void callback() {
                    cblock.lock();
                    try {
                        log.info(" canbus callback");
                        cb.signalAll();
                    } finally {
                        cblock.unlock();
                    }
                }
            };
            cblock.lock();
            try {
                callable.call(sdoCB, rd);
                while (!rd.set) {
                    if (!cb.await(to, TimeUnit.MILLISECONDS)) {
                        abort.abortCallback();
                        throw new DriverTimeoutException(
                                "command timeout : no response received after " + to + " milliseconds");
                    }
                }
                log.info("< sdo end wait " + rd.nodeId + " " + rd.otherData + "[" + rd.toString() + "]");

                return rd;
            } catch (InterruptedException ex) {
                abort.abortCallback();
                throw new DriverException(ex);
            } finally {
                cblock.unlock();
            }
        }

    }

    /**
     * to be used in case of sdoTimeout. Native call is not reentrant. * Clears the
     * callback for the specified nodeId.
     *
     * @param nodeId the node id to reset the callbacks for. 0 to reset the pdo
     *               callback.
     * @param rw
     */
    @Command
    public synchronized native void clearCallBack(int nodeId, int rw);

    @Override
    @Command
    public synchronized native void setNMTStateOperational(int nodeId);

}
