package org.lsst.ccs.drivers.canopenjni;

import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;

/**
 * An interface defining interaction with the CANOpen layer
 * @author emarin
 */
public interface CanOpenInterface {
    
    /**
     * Initialization of the master node. This command has to be issued first
     * for the subsequent one to be successful.
     *
     * @param master 1 for master, 0 for slave.
     * @param baud "125K","1M",...
     * @param busName the bus name (0 for can0, 1 for can1)
     * @param nodeID the nodeID
     * @throws org.lsst.ccs.drivers.commons.DriverException
     */
    public void init(int master, String baud, String busName, int nodeID) throws DriverException ;
    
    /**
     * Configures the pdo with the given cob_id for reception on the master node.
     * @param cobId 
     */
    public void addReceivedPDO(int cobId) throws DriverException;
    
    public void clearReceivedPDOs() throws DriverException;
    
    /**
     * Sends a sync message on the canbus and synchronously waits for the received 
     * pdo configured in the master to be received.
     * For it to work, all received pdos configured on the master node must have 
     * a transmission type equal to Ox1, otherwise the behaviour is undefined.
     * @return The PDOs
     * @throws DriverException in case of a failure accessing the hardware, or if the hardware returns an error
     * @throws DriverTimeoutException if the hardware doesn't respond within the specified timeout
     * @throws ConcurrentCallException if a sync command is running and hasn't completed yet
     */
    public PDOData sync() throws DriverException, DriverTimeoutException, ConcurrentCallException;
    
    /**
     * Sends a scan on the can bus.
     * @return 
     */
    public int scan() throws DriverException ;
    
    /**
     * Returns the information for the given nodeID.
     * @param nodeID the node id
     * @return 
     * @throws org.lsst.ccs.drivers.commons.DriverException 
     */
    public String info(int nodeID) throws DriverException;
    
    /**
     * Send a write sdo command.
     * @param nodeId
     * @param index
     * @param subindex
     * @param size
     * @param data
     * @throws DriverException in case of a failure accessing the hardware, or if the hardware returns an error
     * @throws DriverTimeoutException if the hardware doesn't respond within the specified timeout
     * @throws ConcurrentCallException if a write sdo / read sdo command was sent on the same node and hasn't completed yet
     */
    public void wsdo(int nodeId, int index, int subindex, int size, long data) throws DriverException, DriverTimeoutException, ConcurrentCallException;
    
    /**
     * Send a read sdo command.
     * @param nodeId
     * @param index
     * @param subindex
     * @return
     * @throws DriverException in case of a failure accessing the hardware, or if the hardware returns an error
     * @throws DriverTimeoutException if the hardware doesn't respond within the specified timeout
     * @throws ConcurrentCallException if a write sdo / read sdo command was sent on the same node and hasn't completed yet
     */
    public long rsdo(int nodeId, int index, int subindex) throws DriverException, DriverTimeoutException, ConcurrentCallException;
    
    /**
     * Slave start
     * @param nodeId
     */
    public void ssta(int nodeId) throws DriverException;
    
    /**
     * Slave stop
     * @param nodeId
     */
    public void ssto(int nodeId) throws DriverException;
    
    public void reset(int nodeId) throws DriverException;

    /**
     * Shuts down the master node.
     * @throws DriverException 
     */
    public void quit() throws DriverException;
    
    // Lifecycle
    public void init() throws DriverException;
    
    public void start() throws DriverException;
    
    public void stop() throws DriverException;
    
    public boolean isReady() throws DriverException;
    
    // Call backs for asynchronous information
    public void setEmergencyMessageListener(EmergencyMessageListener eml) throws DriverException;
    
    public void setBootMessageListener(BootMessageListener bml) throws DriverException;
    
    public void setTimeoutMillis(long to) throws DriverException;
    
}
