package org.lsst.ccs.drivers.canopenjni;

import java.io.IOException;
import java.time.Instant;
import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.utilities.logging.Logger;

public class SocketCanOpen implements CanOpenInterface {

    public static final Logger log = Logger.getLogger("org.lsst.ccs.driver.canopenjni");

    private long sdoTimeout = 500;
    private long pdoTimeout = 500;

    int nodeId = -1;
    String devName;
    CanSocket socket;
    CanSocket.CanInterface canIf;

    Set<Integer> registeredPDOs = new HashSet<>();

    Map<Integer, Instant> lastBeat = new ConcurrentHashMap<>();
    Map<Integer, Integer> lastState = new ConcurrentHashMap<>();
    Set<Integer> expectedSlaves = new HashSet<>();

    @Override
    public void addSlave(int s) {
        expectedSlaves.add(s);
    }

    @Override
    public void addSlaves(int[] s) {
        IntStream.of(s).forEach(i -> expectedSlaves.add(i));
    }

    @Override
    public Instant getLastBeat(int id) {
        return lastBeat.get(id);
    }

    @Override
    public int getLastState(int id) {
        Integer l = lastState.get(id);
        return l == null ? -1 : l;
    }

    @Override
    public Set<Integer> getRegisteredPDOs() {
        return Collections.unmodifiableSet(registeredPDOs);
    }

    @Override
    public void addReceivedPDO(int cobId) throws DriverException {
        registeredPDOs.add(cobId);
    }

    @Override
    public void clearReceivedPDOs() throws DriverException {
        registeredPDOs.clear();
    }

    @Override
    public int getNodeId() {
        return nodeId;
    }

    @Override
    public String info(int nodeID) throws DriverException {
        // TODO Auto-generated method stub
        return null;
    }

    @Override
    public void init(int master, String baud, String busName, int nodeID) throws DriverException {
        if (master == 0) {
            throw new DriverException("Slave mode not implemented");
        }
        log.info("canopenjni.SocketCanOpen 03-FEV-2023-10:40");

        if ("0".equals(busName)) {
            busName = "can0";
        } else if ("1".equals(busName)) {
            busName = "can1";
        }

        this.nodeId = nodeID;
        this.devName = busName;
        try {
            socket = new CanSocket(CanSocket.Mode.RAW);
            canIf = new CanSocket.CanInterface(socket, devName);
            socket.bind(canIf);

            startReceive();

            sendNMTResetCommAll();

            try {
                Thread.sleep(200);
            } catch (InterruptedException e) {
            }

            sendBootup();

            initSlaves();

        } catch (IOException e) {
            throw new DriverException("error during SocketCan init", e);
        }
    }

    private boolean allOperational() {
        for (int i : expectedSlaves) {
            if (lastState.get(i) == null || 5 != lastState.get(i)) {
                return false;
            }
        }
        return true;
    }

    private void activateHeartBeat(int device) {
        try {
            wsdo(device, 0x1017, 0, 2, 1000);
        } catch (DriverException e) {
            log.info("Could not activate heartbeat for device " + device + " wsdo failure");
        }
    }

    private void initSlaves() throws DriverException {
        // if we don't know our slaves, do a reset and a set operational
        if (expectedSlaves.isEmpty()) {
            log.info("booting blindly the devices");
            sendNMTResetCommAll();
            try {
                Thread.sleep(500);
            } catch (InterruptedException e) {
            }
            log.info("setting blindly the devices to operational");
            setNMTStateOperational(0x0);
            return;
        }

        log.info("devices init");

        setNMTStateOperational(0x0);
        // NMT frames are handled by the receiver thread

        long ts = System.currentTimeMillis();
        long initTimeout = 1300;
        while (System.currentTimeMillis() < ts + initTimeout && !allOperational()) {
            try {
                Thread.sleep(50);
            } catch (InterruptedException e) {
            }
        }

        if (allOperational()) {
            log.info("all devices operational");
            return;
        }

        Set<Integer> reboot = new HashSet<>();
        Set<Integer> booted = new HashSet<>();
        Set<Integer> hbActive = new HashSet<>();
        Set<Integer> ops = new HashSet<>();

        for (int i : expectedSlaves) {
            if (lastState.get(i) == null) {
                // lastState.remove(i);
                log.info(String.format("reboot device %02x", i));
                reboot.add(i);
                sendNMTResetNode(i);
            }
        }

        ts = System.currentTimeMillis();
        while (System.currentTimeMillis() < ts + initTimeout && (!reboot.isEmpty() || !booted.isEmpty())) {
            try {
                for (int i : reboot) {
                    if (lastState.get(i) != null) {
                        if (lastState.get(i) != 5) {
                            log.info(String.format("start device %02x", i));
                            setNMTStateOperational(i);
                        }
                        booted.add(i);
                    }
                }
                reboot.removeAll(booted);
                for (int i : booted) {
                    if (5 == lastState.get(i)) {
                        log.info(String.format("device %02x operational", i));
                        ops.add(i);
                    }
                }
                booted.removeAll(ops);
                if (!reboot.isEmpty() || !booted.isEmpty()) {
                    Thread.sleep(100);
                }
                if (System.currentTimeMillis() > ts + initTimeout / 2) {
                    for (int i : booted) {
                        if (!hbActive.contains(i)
                                && lastState.get(i) != 5) {
                            log.info(String.format("activate heartbeat device %02x", i));
                            activateHeartBeat(i);
                            hbActive.add(i);
                        }
                    }
                }

            } catch (InterruptedException e) {
            }
        }

    }

    @Override
    public boolean isReady() throws DriverException {
        return nodeId > 0;
    }

    private void checkState() throws DriverException {
        if (!isReady()) {
            throw new DriverException("driver in invalid state, before init or after quit");
        }
    }

    // we could be an autoclosable...
    @Override
    public void quit() throws DriverException {
        stopReceive();
        try {
            if (socket != null) {
                socket.close();
                socket = null;
                nodeId = -1;
            }
        } catch (IOException e) {
            throw new DriverException(e);
        }
    }

    @Override
    public synchronized int scan() throws DriverException {
        initSlaves();
        return 0;
    }

    @Override
    public void setBootMessageListener(BootMessageListener bml) throws DriverException {
        throw new DriverException("Unimplemented method");

    }

    @Override
    public void setEmergencyMessageListener(EmergencyMessageListener eml) throws DriverException {
        throw new DriverException("Unimplemented method");

    }

    @Override
    public void setPdoTimeout(long to) throws DriverException {
        pdoTimeout = to;
    }

    @Override
    public void setSdoTimeout(long to) throws DriverException {
        sdoTimeout = to;
    }

    @Override
    public synchronized PDOData sync() throws DriverException {
        checkState();
        sendSync();
        return rcvPDO();
    }

    @Override
    public synchronized long rsdo(int nodeId, int index, int subindex)
            throws DriverException, DriverTimeoutException, ConcurrentCallException {
        checkState();
        emptyQueue();

        byte[] respStart = sendRsdo(nodeId, index, subindex);
        CanSocket.CanFrame f = getNextFrame(0x580 + nodeId, respStart, sdoTimeout);
        if (f == null) {
            throw new DriverException("No RSDO ACK");
        }

        byte[] data = f.getData();
        // vcan0 582 [8] 4B 17 10 00 E8 03 00 00

        long value = ((long) data[4] & 0xff)
                | (((long) data[5] & 0xff) << 8)
                | (((long) data[6] & 0xff) << 16)
                | (((long) data[7] & 0xff) << 24);

        if (data[0] == (byte) 0x80) {
            throw new SDOException("RSDO error code " + String.format("%08x", value), nodeId, index, subindex, (int) value);
        }

        return value;
    }

    @Override
    public synchronized void wsdo(int nodeId, int index, int subindex, int size, long data)
            throws DriverException, DriverTimeoutException, ConcurrentCallException {
        checkState();
        emptyQueue();
        byte[] respStart = sendWsdo(nodeId, index, subindex, size, data);
        CanSocket.CanFrame f = getNextFrame(0x580 + nodeId, respStart, sdoTimeout);
        if (f == null) {
            throw new DriverException("No WSDO ACK");
        }
        byte[] respData = f.getData();
        long value = ((long) respData[4] & 0xff)
                | (((long) respData[5] & 0xff) << 8)
                | (((long) respData[6] & 0xff) << 16)
                | (((long) respData[7] & 0xff) << 24);
        if (respData[0] == (byte) 0x80) // value is the SDO abort code
        {
            throw new SDOException("WSDO error code " + String.format("%08x", value), nodeId, index, subindex, (int) value);
        }
    }

    // sending frames
    private void sendSync() throws DriverException {
        try {
            CanSocket.CanFrame syncFrame = new CanSocket.CanFrame(canIf, new CanSocket.CanId(0x80), new byte[]{});
            emptyQueue();
            socket.send(syncFrame);
        } catch (IOException e) {
            throw new DriverException(e);
        }
    }

    private void sendBootup() throws DriverException {
        try {
            CanSocket.CanFrame bootFrame = new CanSocket.CanFrame(canIf, new CanSocket.CanId(0x700 + nodeId),
                    new byte[]{0x05});
            socket.send(bootFrame);
        } catch (IOException e) {
            throw new DriverException(e);
        }
    }

    private void sendNMT(byte command, byte target) throws DriverException {
        try {
            CanSocket.CanFrame nmtFrame = new CanSocket.CanFrame(canIf, new CanSocket.CanId(0x0),
                    new byte[]{command, target});
            socket.send(nmtFrame);
        } catch (IOException e) {
            throw new DriverException(e);
        }
    }

    private void sendNMTResetCommAll() throws DriverException {
        sendNMT((byte) 0x82, (byte) 0x00);
    }

    @Override
    public void setNMTStateOperational(int nodeId) throws DriverException {
        checkState();
        sendNMT((byte) 0x01, (byte) nodeId);
    }

    @Override
    public void setNMTStateStop(int nodeId) throws DriverException {
        checkState();
        sendNMT((byte) 0x02, (byte) nodeId);
    }

    @Override
    public void setNMTStatePreOperational(int nodeId) throws DriverException {
        checkState();
        sendNMT((byte) 0x80, (byte) nodeId);
    }

    @Override
    public void resetNode(int nodeId) throws DriverException {
        sendNMTResetNode(nodeId);
    }

    @Override
    public void sendNMTResetNode(int nodeId) throws DriverException {
        checkState();
        sendNMT((byte) 0x81, (byte) nodeId);
    }

    @Override
    public void sendNMTResetComm(int nodeId) throws DriverException {
        checkState();
        sendNMT((byte) 0x82, (byte) nodeId);
    }

    private void emptyQueue() {
        while (true) {
            try {
                if (queue.poll(10, TimeUnit.MICROSECONDS) == null) {
                    return;
                }
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }
    }

    private byte[] sendWsdo(int nodeId, int index, int subindex, int size, long data) throws DriverException {
        if (size > 4) {
            throw new DriverException("Only expedited transfers are implemented, size " + size + " bytes unsopported");
        }
        if (size < 1) {
            throw new DriverException("bad size " + size + " bytes unsopported");
        }
        byte cmd = (byte) (0x2f - 4 * (size - 1));

        CanSocket.CanFrame wsdoFrame = new CanSocket.CanFrame(canIf, new CanSocket.CanId(0x600 + nodeId),
                new byte[] { cmd,
                        (byte) (index & 0xff), (byte) ((index >> 8) & 0xff), (byte) subindex,
                        (byte) (data & 0xff), (byte) ((data >> 8) & 0xff), (byte) ((data >> 16) & 0xff),
                        (byte) ((data >> 24) & 0xff) });
        try {
            socket.send(wsdoFrame);
            return new byte[] { (byte) 0x60, (byte) (index & 0xff), (byte) ((index >> 8) & 0xff), (byte) subindex };
        } catch (IOException e) {
            throw new DriverException(e);
        }
    }

    private byte[] sendRsdo(int nodeId, int index, int subindex) throws DriverException {
        CanSocket.CanFrame rsdoFrame = new CanSocket.CanFrame(canIf, new CanSocket.CanId(0x600 + nodeId),
                new byte[] { (byte) 0x40,
                        (byte) (index & 0xff), (byte) ((index >> 8) & 0xff), (byte) subindex,
                        (byte) 0x00, (byte) 0x00, (byte) 0x00, (byte) 0x00 });
        try {
            socket.send(rsdoFrame);
            return new byte[] { (byte) 0xff, (byte) (index & 0xff), (byte) ((index >> 8) & 0xff), (byte) subindex };
        } catch (IOException e) {
            throw new DriverException(e);
        }

    }

    // expedited wsdo vcan0 602 [8] 2B 17 10 00 E8 03 00 00 // 2 write 0x1017 0 u16
    // 1000
    // => vcan0 582 [8] 60 17 10 00 00 00 00 00
    // rsdo 2 read 0x1017 0 u16
    // vcan0 602 [8] 40 17 10 00 00 00 00 00
    // vcan0 582 [8] 4B 17 10 00 E8 03 00 00
    // receving frames
    LinkedBlockingQueue<CanSocket.CanFrame> queue = new LinkedBlockingQueue<>();

    volatile boolean loopReceive = true;

    private void runReceive() {
        while (loopReceive) {
            try {
                CanSocket.CanFrame f = socket.recv(); // blocking
                if (f == null) {
                    continue;
                }
                // beat are handled separately
                int cobid = f.getCanId().getCanId_SFF();
                // cobid 11 bits, 4 bit COB + 7 bit ID
                int cob = cobid >> 7;
                int id = cobid & 0x07F;
                if (cob == 0xe) { // beat
                    lastBeat.put(id, Instant.now());
                    lastState.put(id, (int) f.getData()[0]);
                    continue;
                }
                // handle also boot message, keep track of last state.
                queue.put(f);
            } catch (IOException e) {
                try {
                    Thread.sleep(100);
                } catch (InterruptedException e1) {
                }
            } catch (InterruptedException e1) {
            }
        }
    }

    private void startReceive() {
        loopReceive = true;
        new Thread(() -> runReceive()).start();
    }

    private void stopReceive() {
        loopReceive = false;
    }

    private CanSocket.CanFrame getNextFrame(long timeout) {
        try {
            return queue.poll(timeout, TimeUnit.MILLISECONDS);
        } catch (InterruptedException e) {
            return null;
        }
    }

    private CanSocket.CanFrame getNextFrame(int canid, long timeout) {
        long ts = System.currentTimeMillis();
        while (System.currentTimeMillis() < ts + timeout) {
            CanSocket.CanFrame f = getNextFrame(timeout);
            if (f == null) {
                continue;
            }
            if (f.getCanId().getCanId_SFF() == canid) {
                return f;
            }
        }
        return null;
    }

    private CanSocket.CanFrame getNextFrame(int canid, byte[] dataStart, long timeout) {
        long ts = System.currentTimeMillis();
        loop: while (System.currentTimeMillis() < ts + timeout) {
            CanSocket.CanFrame f = getNextFrame(timeout);
            if (f == null) {
                continue;
            }
            if (f.getCanId().getCanId_SFF() != canid) {
                continue loop;

            }
            byte[] data = f.getData();
            if (data[0] != dataStart[0] && data[0] != (byte) 0x80 && dataStart[0] != (byte) 0xff) {
                continue loop;
            }
            for (int i = 1; i < dataStart.length; i++) {
                if (data[i] != dataStart[i]) {
                    continue loop;
                }
            }

            return f;
        }
        return null;
    }

    private PDOData rcvPDO() {
        PDOData pdata = new PDOData();
        Set<Integer> expectedPDOs = new HashSet<>();
        expectedPDOs.addAll(registeredPDOs);
        long ts = System.currentTimeMillis();
        while (System.currentTimeMillis() < ts + pdoTimeout && !expectedPDOs.isEmpty()) {
            CanSocket.CanFrame f = getNextFrame(pdoTimeout);
            if (f == null) {
                continue;
            }
            int canid = f.getCanId().getCanId_SFF();
            if (expectedPDOs.contains(canid)) {
                expectedPDOs.remove(canid);
                byte[] fdata = f.getData();
                long value = 0;
                for (int i = fdata.length - 1; i >= 0; i--) {
                    value = (value << 8) | (((long) fdata[i]) & 0xff);
                }
                // System.out.println(value);
                pdata.updatePDO(canid, value);
            }
        }
        if (!expectedPDOs.isEmpty()) {
            pdata.errCode = 0x99;
            String missingHex = expectedPDOs.stream()
                    .map(Integer::toHexString)
                    .collect(Collectors.joining(", ", "[", "]"));
            log.warning("CanOpenJNI : Missing PDOs " + missingHex);
        }

        return pdata;
    }

}
