package org.lsst.ccs.drivers.canopenjni;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Implementation of a master node.
 * Initializes a master node.
 * @author emarin
 */
public class CanopenMaster {
    
    private static final Logger log = Logger.getLogger("org.lsst.ccs.drivers.canopenjni");
    
    private final Map<Integer, CanopenSlave> slaves = new HashMap<>();
    
    private final CanOpenInterface coi;
  
    public CanopenMaster(CanOpenInterface coi) {
        this.coi = coi;
    }

    public synchronized void addSlave(CanopenSlave slave) throws DriverException {
        slaves.put(slave.getNodeId(), slave);
        for(int cobId : slave.getCobIds()) {
            coi.addReceivedPDO(cobId);
        }
    }
    
    /**
     * Sends a sync message on the canbus and synchronously waits for the received 
     * pdo configured in the master to be received.
     * For it to work, all received pdos configured on the master node must have 
     * a transmission type equal to Ox1, otherwise the behaviour is undefined.
     * @throws DriverException 
     */
    @Command
    public void sync() throws DriverException {
        PDOData rd = coi.sync();
        for (Map.Entry<Integer, Long> e : rd.pdos.entrySet()) {
            slaves.get(e.getKey() & 0xFF - 0x80).updateWithPDO(e.getKey(), e.getValue());
        }
    }
    
    /**
     * Returns the information for the given nodeID.
     * @param nodeID the node id
     * @return 
     * @throws org.lsst.ccs.drivers.commons.DriverException 
     */
    @Command
    public synchronized String info(int nodeID) throws DriverException {
        return coi.info(nodeID);
    }
    
    /**
     * 
     * @param nodeId the node id
     * @param index the index
     * @param subindex the sub index
     * @param size the size
     * @param data the data
     * @throws DriverException 
     */
    @Command
    public void wsdo(int nodeId, int index, int subindex, int size, int data) throws DriverException {
        coi.wsdo(nodeId, index, subindex, size, data);
    }
    
    /**
     * Read device entry
     * @param nodeId the node id
     * @param index the index
     * @param subindex the subindex
     * @return the data
     * @throws org.lsst.ccs.drivers.commons.DriverException
     */
    @Command
    public long rsdo(int nodeId, int index, int subindex) throws SDOException, DriverException {
      return coi.rsdo(nodeId, index, subindex);
    }
            
}
